use std::fmt;

#[derive(Copy, Clone)]
pub struct Vect3d {
  pub x: f64,
  pub y: f64,
  pub z: f64,
}

impl fmt::Display for Vect3d {
  fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
    write!(f, "({:.6e},{:.6e},{:.6e})", self.x, self.y, self.z)
  }
}

// member functions here

impl Vect3d {
  fn sumsq(self) -> f64 {
    return self.x * self.x + self.y * self.y + self.z * self.z;
  }

  // optimized orbital acceleration function
  pub fn acceleration(self) -> f64 {
    return self.sumsq().powf(-1.5);
  }

  pub fn mag(self) -> f64 {
    return self.sumsq().sqrt();
  }

  // return values as a tuple
  pub fn array(self) -> (f64, f64,f64) {
    return (self.x, self.y, self.z);
  }

  // 3D rotation matrices

  // see https://mathworld.wolfram.com/RotationMatrix.html

  // generic rotate function using provided rotation matrix

  // assigns result to instance variables

  fn rotate(&mut self,rotmatrix:[[f64;3];3]) {
    let values = [self.x, self.y, self.z];
    let mut result = [0.0, 0.0, 0.0];
    for col in 0..3 {
      for row in 0..3 {
        result[col] += values[row] * rotmatrix[col][row];
      }
    }
    // see https://github.com/rust-lang/rfcs/issues/372
    // for why this homely hack is still needed
    self.x = result[0];
    self.y = result[1];
    self.z = result[2];
  }

  pub fn rotx(&mut self, a: f64) {
    let rotmatrix = [
      // X
      [1.0, 0.0, 0.0],
      [0.0, a.cos(), a.sin()],
      [0.0, -a.sin(), a.cos()],
    ];
    self.rotate(rotmatrix);
  }

  pub fn roty(&mut self, a: f64) {
    let rotmatrix = [
      // Y
      [a.cos(), 0.0, -a.sin()],
      [0.0, 1.0, 0.0],
      [a.sin(), 0.0, a.cos()],
    ];
    self.rotate(rotmatrix);
  }

  pub fn rotz(&mut self, a: f64) {
    let rotmatrix = [
      // Z
      [a.cos(), a.sin(), 0.0],
      [-a.sin(), a.cos(), 0.0],
      [0.0, 0.0, 1.0],
    ];
    self.rotate(rotmatrix);
  }
}

// initialize with an array [f64;3]

impl From<[f64; 3]> for Vect3d {
  fn from(t: [f64; 3]) -> Vect3d {
    Vect3d {
      x: t[0],
      y: t[1],
      z: t[2],
    }
  }
}

// operator overloads

// ref: https://doc.rust-lang.org/std/ops/index.html

// a + b

impl std::ops::Add for Vect3d {
  type Output = Vect3d;
  fn add(self, rhs: Vect3d) -> Vect3d {
    Vect3d {
      x: self.x + rhs.x,
      y: self.y + rhs.y,
      z: self.z + rhs.z,
    }
  }
}

// a += b

impl std::ops::AddAssign for Vect3d {
  fn add_assign(&mut self, rhs: Self) {
    *self = Self {
      x: self.x + rhs.x,
      y: self.y + rhs.y,
      z: self.z + rhs.z,
    }
  }
}

// a - b

impl std::ops::Sub for Vect3d {
  type Output = Vect3d;
  fn sub(self, rhs: Vect3d) -> Vect3d {
    Vect3d {
      x: self.x - rhs.x,
      y: self.y - rhs.y,
      z: self.z - rhs.z,
    }
  }
}

// a -= b

impl std::ops::SubAssign for Vect3d {
  fn sub_assign(&mut self, rhs: Self) {
    *self = Self {
      x: self.x - rhs.x,
      y: self.y - rhs.y,
      z: self.z - rhs.z,
    }
  }
}

// a:Vect3d * b: f64

impl std::ops::Mul<f64> for Vect3d {
  type Output = Vect3d;
  fn mul(self, rhs: f64) -> Vect3d {
    Vect3d {
      x: self.x * rhs,
      y: self.y * rhs,
      z: self.z * rhs,
    }
  }
}

// a:Vect3d * b:Vect3d

impl std::ops::Mul for Vect3d {
  type Output = Vect3d;
  fn mul(self, rhs: Vect3d) -> Vect3d {
    Vect3d {
      x: self.x * rhs.x,
      y: self.y * rhs.y,
      z: self.z * rhs.z,
    }
  }
}

// a:Vect3d *= b:Vect3d

impl std::ops::MulAssign for Vect3d {
  fn mul_assign(&mut self, rhs: Self) {
    *self = Self {
      x: self.x * rhs.x,
      y: self.y * rhs.y,
      z: self.z * rhs.z,
    }
  }
}

// a:Vect3d *= b:f64

impl std::ops::MulAssign<f64> for Vect3d {
  fn mul_assign(&mut self, rhs: f64) {
    *self = Self {
      x: self.x * rhs,
      y: self.y * rhs,
      z: self.z * rhs,
    }
  }
}
