#![allow(dead_code, non_snake_case, non_upper_case_globals)]

pub mod vect3d;

use vect3d::Vect3d;

fn maxmin(v: f64, maxv: &mut f64, minv: &mut f64) {
  *maxv = maxv.max(v);
  *minv = minv.min(v);
}

// a simple circular orbit velocity function
// not used here
fn cov(G: f64, m: f64, r: f64) -> f64 {
  return (G * m / r).sqrt();
}

// https://en.wikipedia.org/wiki/Vis-viva_equation

// original vis-viva equation, not used here
// v^2 = G M (2/r - 1/a)
// r = radius between bodies
// a = semimajor axis
fn vis_viva(G: f64, m: f64, r: f64, a: f64) -> f64 {
  return (G * m * (2.0 / r - 1.0 / a)).sqrt();
}

// function derived from the vis-viva equation
// https://en.wikipedia.org/wiki/Vis-viva_equation
// for initial orbital velocity in an elliptical orbit
//   G = universal gravitational constant
//   m = mass of central body
//   a = aphelion or perihelion radius
//   p = perihelion or aphelion radius
//   a and p are interchangeable based on
//   the desired outcome
// position orbiting body initially at the a radius
fn eov(G: f64, m: f64, a: f64, p: f64) -> f64 {
  return ((2.0 * G * m * p) / (a * (a + p))).sqrt();
}

// potential energy
fn pe(G: f64, m1: f64, m2: f64, r: f64) -> f64 {
  return -G * m1 * m2 / r;
}

// kinetic energy
fn ke(m: f64, v: f64) -> f64 {
  return m * v * v / 2.0;
}

fn main() {
  // "Big G", universal gravitational constant
  // https://physics.nist.gov/cgi-bin/cuu/Value?bg

  const G: f64 = 6.67430e-11;

  // model time slice seconds
  const Δt: f64 = 10.0;

  // Sun: https://nssdc.gsfc.nasa.gov/planetary/factsheet/sunfact.html
  // Earth: https://nssdc.gsfc.nasa.gov/planetary/factsheet/earthfact.html

  let sunmass = 1.9885e30;

  let pph: f64;
  let pah: f64;
  let pmass: f64;

  // choose earth/mars here

  let name = "Earth";

  match name {
    "Earth" => {
      pph = 147.095e9; // earth perihelion meters
      pah = 152.100e9; // earth aphelion meters
      pmass = 5.9722e24; // earth mass kilograms
    }
    "Mars" => {
      pph = 206.650e9; // mars perihelion meters
      pah = 249.261e9; // mars aphelion meters
      pmass = 0.64169e24; // mars mass kilograms
    }
    _ => {
      panic!("Error: unknown target planet {name}");
    }
  }
  // my own orbital function for initial orbital velocity
  // in an elliptical orbit
  // aphelion velocity
  let plv = eov(G, sunmass, pah, pph);
  // perihelion velocity
  let phv = eov(G, sunmass, pph, pah);

  // position vector
  let mut pos = Vect3d {
    x: pah,
    y: 0.0,
    z: 0.0,
  };

  // velocity vector
  let mut vel = Vect3d {
    x: 0.0,
    y: plv,
    z: 0.0,
  };

  let init_ke = ke(pmass, plv);
  let init_pe = pe(G, pmass, sunmass, pah);
  let init_e = init_ke + init_pe;

  println!("Model: {name} orbit");
  println!("Simulator step size Δt: {Δt:.2} seconds");
  println!("Calculated/measured orbital parameters:");
  println!("  Orbital radius meters:");
  println!("    max (aphelion):   {pah:.6e}");
  println!("    min (perihelion): {pph:.6e}");
  println!("  Orbital velocity m/s:");
  println!("    max: {phv:.6e}");
  println!("    min: {plv:.6e}");
  println!("  Orbital energy joules (should be constant):");
  println!("    mean: {init_e:.6e}");
  // 1 year orbit in seconds.
  // for students:
  // one year
  //   * days in year
  //   * hours in day
  //   * minutes in hour
  //   * seconds in minute
  let duration = 1.0 * 365.25 * 24.0 * 60.0 * 60.0;

  // model elapsed time seconds
  let mut elapsed_time = 0.0;

  // accumulators for tracking maxima and minima
  let mut maxr: f64 = -1e300;
  let mut minr: f64 = 1e300;
  let mut maxv: f64 = -1e300;
  let mut minv: f64 = 1e300;
  let mut maxe: f64 = -1e300;
  let mut mine: f64 = 1e300;

  let mut iter = 0;

  use std::time::Instant;

  let now = Instant::now();

  let accel_term = -G * sunmass * Δt;
  
  loop {
    // compute acceleration
    let acc = pos * pos.acceleration() * accel_term;

    // update velocity
    vel += acc;

    // collect velocity min/max data
    let mv = vel.mag();
    maxmin(mv, &mut maxv, &mut minv);

    // update position
    pos += vel * Δt;

    // update orbital radius min/max data
    let r = pos.mag();
    maxmin(r, &mut maxr, &mut minr);

    // kinetic energy
    let ke = ke(pmass, vel.mag());

    // potential energy
    let pe = pe(G, pmass, sunmass, r);

    // sum of ke and pe should be constant
    maxmin(ke + pe, &mut maxe, &mut mine);

    // update elapsed time
    elapsed_time += Δt;
    if elapsed_time > duration {
      break;
    }
    iter += 1;
  }

  let elapsed = now.elapsed().as_micros();

  let error = 100.0 * (maxe - mine).abs() / maxe.abs();

  println!("Simulation results:");
  println!("  Iterations: {iter}");
  println!("  Orbital radius meters:");
  println!("    max: {maxr:.6e}");
  println!("    min: {minr:.6e}");
  println!("  Orbital velocity m/s:");
  println!("    max: {maxv:.6e}");
  println!("    min: {minv:.6e}");
  println!("  Orbital energy joules:");
  println!("    max:   {maxe:.6e}");
  println!("    min:   {mine:.6e}");
  println!("    error: {error:.7}%");
  
  println!("\nElapsed algorithm time: {:.3} ms", (elapsed as f64/1000.0));
}
