#!/usr/bin/env python3
# -*- coding: utf-8 -*-

import math as m


class Vect3d:
  def __init__(self, x, y=0, z=0):
    if isinstance(x, Vect3d):
      self.x = x.x
      self.y = x.y
      self.z = x.z
    else:
      self.x = x
      self.y = y
      self.z = z

  # 2D position for plotting
  def pos2d(self):
    return(self.x, self.y)

  # Cartesian vector
  def vector(self):
    return (self.x, self.y, self.z)

  # operator overloading functions

  # multiply by vector or scalar
  def __mul__(self, v):
    if isinstance(v, Vect3d):
      return Vect3d(self.x * v.x, self.y * v.y, self.z * v.z)
    return Vect3d(self.x * v, self.y * v, self.z * v)

  # vector argument
  def __add__(self, v):
    return Vect3d(self.x + v.x, self.y + v.y, self.z + v.z)

  # vector or scalar argument (not working)
  def __iadd__(self, v):
    #print("entering += ... " + str(type(self)) + str(type(v)))
    if isinstance(v, Vect3d):
      self.x += v.x
      self.y += v.y
      self.z += v.z
    else:
      self.x += v
      self.y += v
      self.z += v
    return self

  # vector argument
  def __sub__(self, v):
    return Vect3d(self.x - v.x, self.y - v.y, self.z - v.z)

  # scalar argument
  def __pow__(self, p):
    return Vect3d(self.x**p, self.y**p, self.z**p)

  # vector argument
  def __neg__(self):
    return Vect3d(-self.x, -self.y, -self.z)

  def sumsq(self):
    return self.x*self.x + self.y*self.y + self.z*self.z

  # vector magnitude
  def mag(self):
    return m.sqrt(self.sumsq())

  # orbital physics operator

  def acceleration(self):
    return m.pow(self.sumsq(), -1.5)

  def __repr__(self):
    return f'({self.x:12.6e},{self.y:12.6e},{self.z:12.6e})'

  # 3D rotation matrices

  # see https://mathworld.wolfram.com/RotationMatrix.html

  # generic rotate function common to all axes
  # accepts a prepared 3x3 rotation matrix
  # assigns result to instance x,y,z elements

  def rotate(self,matrix):
    arg = self.x,self.y,self.z
    result = [0.0, 0.0, 0.0]
    for col in range(3):
      for row in range(3):
        result[col] += arg[row] * matrix[col][row]
    self.x,self.y,self.z = result

  # rotx rotates y and z

  def rotx(self,a):
    rm = [
        [1,        0,         0],
        [0,  m.cos(a), m.sin(a)],
        [0, -m.sin(a), m.cos(a)]
    ]
    self.rotate(rm)

  # roty rotates x and z

  def roty(self,a):
    rm = [
        [m.cos(a), 0, -m.sin(a)],
        [0,        1,        0],
        [m.sin(a), 0,  m.cos(a)]
    ]
    self.rotate(rm)

  # rotz rotates x and y

  def rotz(self,a):
    rm = [
        [m.cos(a), m.sin(a),  0],
        [-m.sin(a), m.cos(a),  0],
        [0,         0,         1]
    ]
    self.rotate(rm)

# test each rotation matrix

def main():
  args = {'rotx':(0,0,1),'roty':(1,0,0),'rotz':(0,1,0)}
  for key in args.items():
    #funct = functs[key]
    print(f"\nself.{key}(arg):\n")
    print(f"{'':5}{'arg':9}{'x':7}{'y':7}{'z'}")
    print(f"{'':3}{('-' * 28)}")
    for deg in range(0,100,10):
      rad = m.radians(deg)
      x,y,z = args[key]
      v3d = Vect3d(x,y,z)
      {
        'rotx':v3d.rotx,
        'roty':v3d.roty,
        'rotz':v3d.rotz
      }[key](rad)
      sr = [f"{v:7.3f}" for v in v3d.vector()]
      sr = ''.join(sr)
      print(f"{'':2}{deg:6.2f}° {sr}")

if __name__ == "__main__":
  main()
