#!/usr/bin/env python3
# -*- coding: utf-8 -*-

import math

from timeit import default_timer as timer

from vect3d import Vect3d

def maxmin(v, maxv, minv):
  return max(v,maxv),min(v,minv)

# function derived from the vis-viva equation
# https://en.wikipedia.org/wiki/Vis-viva_equation
# for initial orbital velocity in an elliptical orbit
#   G = universal gravitational constant
#   m = mass of central body
#   a = aphelion or perihelion radius
#   p = perihelion or aphelion radius
#   a and p are interchangeable based on
#   the desired outcome
# position orbiting body initially at the a radius

def eov(G, m, a, p):
  return math.sqrt((2.0 * G * m * p) / (a * (a + p)))

# potential energy
def pe(G, m1, m2, r):
  return -G * m1 * m2 / r

# kinetic energy
def ke(m,v):
  return m * v*v / 2.0

def main():

  # "Big G", universal gravitational constant
  # https://physics.nist.gov/cgi-bin/cuu/Value?bg

  G = 6.67430e-11

  # model time slice seconds
  Δt = 10.0

  # Sun: https://nssdc.gsfc.nasa.gov/planetary/factsheet/sunfact.html
  # Earth: https://nssdc.gsfc.nasa.gov/planetary/factsheet/earthfact.html

  sunmass = 1.9885e30
  eph = 147.095e9 # earth perihelion meters
  eah = 152.100e9 # earth aphelion meters
  emass = 5.9722e24 # earth mass kilograms

  # my own orbital function for initial orbital velocity
  # in an elliptical orbit
  # aphelion velocity
  elv = eov(G, sunmass, eah, eph)
  # perihelion velocity
  ehv = eov(G, sunmass, eph, eah)

  # position vector
  pos = Vect3d (
    x = eah,
    y = 0.0,
    z = 0.0,
  )

  # velocity vector
  vel = Vect3d (
    x = 0.0,
    y = elv,
    z = 0.0,
  )

  init_ke = ke(emass,elv)
  init_pe = pe(G,emass,sunmass,eah)
  init_e = init_ke+init_pe

  print("Model: Earth orbit")
  print(f"Step Δt, seconds: {Δt:.2}")
  print("Calculated/measured orbital parameters:")
  print("  Orbital radius meters:")
  print(f"    max (aphelion):   {eah:.6e}")
  print(f"    min (perihelion): {eph:.6e}")
  print("  Orbital velocity m/s:")
  print(f"    max: {ehv:.6e}")
  print(f"    min: {elv:.6e}")
  print("  Orbital energy joules (should be constant):")
  print(f"    mean: {init_e:.6e}")

  # 1 year orbit in seconds.
  # for students:
  # one year
  #   * days in year
  #   * hours in day
  #   * minutes in hour
  #   * seconds in minute
  duration = 1.0 * 365.25 * 24.0 * 60.0 * 60.0

  # model elapsed time seconds
  elapsed_time = 0.0

  # accumulators for measuring maxima and minima
  maxr = -1e300
  minr = 1e300
  maxv = -1e300
  minv = 1e300
  maxe = -1e300
  mine = 1e300

  iterv = 0

  start = timer()

  accel_term = -G * sunmass * Δt

  while True:

    # compute acceleration
    acc = pos * pos.acceleration() * accel_term

    # update velocity
    vel += acc

    # collect velocity min/max data
    mv = vel.mag()
    maxv,minv = maxmin(mv,maxv,minv)

    # update position
    pos += vel * Δt

    # collect radius min/max data
    r = pos.mag()
    maxr, minr = maxmin(r,maxr,minr)

    # kinetic energy
    vke = ke(emass,vel.mag())

    # potential energy
    vpe = pe(G,emass,sunmass,r)

    # sum of ke and pe should be constant
    maxe, mine = maxmin(vke+vpe, maxe, mine)

    # update elapsed time
    elapsed_time += Δt
    if elapsed_time > duration: break
    iterv += 1

  end = timer()

  error = 100.0 * abs(maxe-mine) / abs(maxe)

  print("Simulation results:")
  print(f"  Iterations: {iter}")
  print("  Orbital Radius meters:")
  print(f"    max: {maxr:.6e}")
  print(f"    min: {minr:.6e}")
  print("  Orbital velocity m/s:")
  print(f"    max: {maxv:.6e}")
  print(f"    min: {minv:.6e}")
  print("  Orbital energy joules:")
  print(f"    max:   {maxe:.6e}")
  print(f"    min:   {mine:.6e}")
  print(f"    error: {error:.7f}%")

  elapsed = (end-start)* 1000

  print(f"\nElapsed algorithm time: {elapsed:.3f} ms")

if __name__ == "__main__":
  main()
