/***************************************************************************
 *   Copyright (C) 2021, Paul Lutus                                        *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

'use strict'

class Vect {

  constructor(x = 0, y = 0, z = 0) {
    if (x instanceof Vect) {
      this.x = x.x;
      this.y = x.y;
      this.z = x.z;
    }
    else {
      this.x = x;
      this.y = y;
      this.z = z;
    }
  }

  // used in orbital calculations

  invSumCube() {
    return Math.pow(this.x ** 2 + this.y ** 2 + this.z ** 2, -1.5);
  }

  // hypotenuse of three-element vector

  mag() {
    return Math.sqrt(this.x ** 2 + this.y ** 2 + this.z ** 2);
  }

  arg() {
    return Math.atan2(this.y, this.x);
  }

  // accumulate values

  add(v) {
    return new Vect(this.x + v.x, this.y + v.y, this.z + v.z);
  }

  // add to existing class instance

  addTo(v) {
    this.x += v.x;
    this.y += v.y;
    this.z += v.z;
  }

  // subtract a vector

  sub(v) {
    return new Vect(this.x - v.x, this.y - v.y, this.z - v.z);
  }

  // multiply by a scalar or vector

  mult(v) {
    if (v instanceof Vect) {
      return new Vect(this.x * v.x, this.y * v.y, this.z * v.z);
    }
    else {
      return new Vect(this.x * v, this.y * v, this.z * v);
    }
  }

  real() {
    return x;
  }

  imag() {
    return y;
  }

  show() {
    return ("Vect:" + this.x + "," + this.y + "," + this.z);
  }
}

class CartesianPair {
  constructor(x = 0, y = 0) {
    this.x = x;
    this.y = y;
  }
}

class Body {
  constructor(name = "Unnamed", p = new Vect(0, 0, 0), v = new Vect(0, 0, 0), rb = 1, m = 1, col = 'rgb(255,255,255,)', active = false) {
    this.name = name;
    this.p = p;
    this.v = v;
    this.rb = rb;
    this.m = m;
    this.col = col;
    this.active = active;
  }
  // kinetic energy
  ke() {
    return ((this.m * this.v.mag() ** 2) / 2);
  }
  // gravitational potential energy
  pe(othermass) {
    // this assumes the sun is at the center position: {0,0,0}
    return (-G * this.m * othermass / this.p.mag());
  }
  show() {
    return ("Body " + this.name + " : " + this.p.x + "," + this.p.y + "|" + this.v.x + "," + this.v.y + "|" + this.rb + "|" + this.m);
  }
}

// the all-important gravitational constant
const G = 6.67408e-11; // N m^2 / kg^-2

class Planets {
  constructor() {
    // sun parameters

    var sor = 0; // position 0
    var sov = 0; // velocity 0

    // solar radius: https://en.wikipedia.org/wiki/Solar_radius

    var sr = 6.957e8; // meters

    // solar mass: https://en.wikipedia.org/wiki/Solar_mass

    var sm = 1.989e30; // mass kilos

    this.sun = new Body('Sun', new Vect(0, 0, 0), new Vect(0, 0, 0), sr, sm, 'rgb(153,102,76)', false);

    // mercury

    var meor = 57.9e9;
    var meov = this.circular_orbit_velocity(G, sm, meor);
    var mer = 2.44e6;
    var mem = 0.33e24;

    this.mercury = new Body('Mercury', new Vect(meor, 0, 0), new Vect(0, meov, 0), mer, mem, (255, 255, 128), true);

    // venus

    var vor = 108.2e9; // planet orbital radius meters
    var vov = this.circular_orbit_velocity(G, sm, vor);
    var vr = 6.052e6; // planet radius meters
    var vm = 4.87e24; // planet mass kilograms

    this.venus = new Body('Venus', new Vect(vor, 0, 0), new Vect(0, vov, 0), vr, vm, 'rgb(255,255,0)', true);

    // earth

    var eor = 149.6e9; // orbital radius meters
    var eov = this.circular_orbit_velocity(G, sm, eor);
    var er = 6.3781e6; // planet radius meters
    var em = 5.972e24; // mass kilograms

    this.earth = new Body('Earth', new Vect(eor, 0, 0), new Vect(0, eov, 0), er, em, 'rgb(128,255,255)', true);

    // earth's moon

    var emor = 384.4e6; // orbital radius meters (from earth)
    var emov = this.circular_orbit_velocity(G, em, emor);
    var emr = 1738.1e3; // radius meters
    var emm = 7.34767e22; // mass kilos

    this.moon = new Body('Moon', new Vect(eor + emor, 0, 0), new Vect(0, eov + emov, 0), emr, emm, 'rgb(255,255,255)', false);

    // mars

    var mor = 227.92e9; // orbital radius meters
    var mov = this.circular_orbit_velocity(G, sm, mor);
    var mr = 3.3972e6; // planetary radius meters
    var mm = 6.39e23; // mass kilos

    this.mars = new Body('Mars', new Vect(mor, 0, 0), new Vect(0, mov, 0), mr, mm, 'rgb(255,128,64)', true);

    // jupiter

    var jor = 778.6e9; // orbital radius meters
    var jov = this.circular_orbit_velocity(G, sm, jor);
    var jr = 7.1492e7; // planet radius
    var jm = 1.898e27; // mass kilos

    // initially at screen bottom

    this.jupiter = new Body('Jupiter', new Vect(0, -jor, 0), new Vect(jov, 0, 0), jr, jm, 'rgb(255,255,128)', true)

    var saor = 1433.5e9;
    var saov = this.circular_orbit_velocity(G, sm, saor);
    var sar = 6.02e7;
    var sam = 5.68e26;

    this.saturn = new Body('Saturn', new Vect(saor, 0, 0), new Vect(0, saov, 0), sar, sam, 'rgb(255,204,153)', false);

    // spacecraft placeholder

    this.spacecraft = new Body('Spacecraft', new Vect(0, 0, 0), new Vect(0, 0, 0), 10, 1000, 'rgb(255,0,0)', false);;

    // show spacecraft velocity in display
    this.spacecraft.special = true;

    this.bodies = [
      this.sun,
      this.mercury,
      this.venus,
      this.earth,
      this.moon,
      this.mars,
      this.jupiter,
      this.saturn,
      this.spacecraft
    ];
  }

  circular_orbit_velocity(g, m, r) {
    return Math.sqrt(g * m / r);
  }
}

class Orbit {

  constructor(hs, vs) {
    //console.log("entering constructor");

    this.busy = false;
    this.framerate = 30; // frames per second
    this.running = true;
    this.etime = 0;
    this.n = 0;
    this.bg_color = "black";
    this.sun_color = 'rgb(255,255,0)';
    this.earth_color = 'rgb(255,0,0)';
    this.degree = 180.0 / Math.PI;
    this.radian = Math.PI / 180.0;
    this.planet_radius = 12;
    this.sun_radius = 20;

    // critical program constants

    this.dt = 900;
    this.div = 300;

    this.pi2 = Math.PI * 2.0;
    this.planets = new Planets();
    this.hs = hs;
    this.vs = vs;
    this.canvas = false;

    this.webpage = (vs === undefined);
    if (this.webpage) {
      //console.log(`webpage: ${hs}`);
      this.canvas = document.getElementById(hs);
      this.sprintf = function () { }
    }
    else {
      //console.log("node.js");
      if (hs !== undefined) {
        //console.log('creating canvas');
        this.cvs = _8d3‍.t("require")("canvas");
        this.canvas = this.cvs.createCanvas(hs, vs);
        this.sprintf = _8d3‍.t("require")('sprintf-js').sprintf;
      }
    }
    if (this.canvas) {
      this.hs = this.canvas.width;
      this.vs = this.canvas.height;
    }
    // planet image scaling
    this.body_radius = 24;

    //console.log("image dims: " + this.hs + "," + this.vs);

    this.image_size = [this.hs, this.vs];

    //console.log("image size: " + this.image_size);

    this.screen_center = new Vect(this.image_size[0] / 2, this.image_size[1] / 2, 0);

    this.image_scale = Math.min(...this.image_size) / 2; // must be a radius

    //console.log("image scale: " + this.image_scale);
    this.ctx = false;

    if (this.canvas) {
      this.ctx = this.canvas.getContext("2d");
      this.ctx.font = '16px monospace';
    }

    // set the display framerate time delay
    this.update_delay_ms = 0;

    if (this.webpage) {
      this.mydraw();
    }
  }

  // only for a webpage
  mydraw(timestamp = 0) {
    //console.timeEnd("mydraw");
    //console.time("mydraw");
    this.process();
    if (this.running) {
      setTimeout(this.mydraw.bind(this), this.update_delay_ms, false);
    }
  }

  ntrp(x, xa, xb, ya, yb) {
    return (x - xa) * (yb - ya) / (xb - xa) + ya;
  }

  compute_orbital_energy(v, r) {
    this.energy_k = 0.5 * v * v; // i.e. 1/2 v^2, unit mass
    this.energy_p = -this.G * this.e_mass / r; // i.e. GM/r
    this.energy_t = this.energy_k + this.energy_p;
  }

  convertCoordinates(x, y) {
    var rx = this.ntrp(x, 0, 1, 0, this.canvas.width);
    var ry = this.ntrp(y, 0, 1, this.canvas.height, 0);
    return new CartesianPair(rx, ry);
  }

  displayresult() {
    var s = this.sprintf(
      "Orbital Radius   m =  %+16.4f\n"
      + "Kinetic energy   Ek = %+16.4f\n"
      + "Potential energy EP = %+16.4f\n"
      + "Total Energy  Ek+Ep = %+16.4f"
      , this.pos.abs(), this.energy_k, this.energy_p, this.energy_t);
    this.ctx.font = '24pt Courier';
    this.ctx.textAlign = 'left';
    this.ctx.fillStyle = 'rgb(255,255,255)';
    this.ctx.fillText(s, 40, 950);
  }

  getbuffer(s = 'image/png') {
    return this.canvas.toBuffer(s);
  }

  perform_physics(dt) {
    this.planets.bodies.forEach((a) => {
      // avoid numerical difficulties
      // caused by asymmetrical accelerations
      if (a.active) {
        this.planets.bodies.forEach((b) => {
          if (a != b) {
            // acquire inter-body radius
            var dr = a.p.sub(b.p); // radius as vector
            // compute acceleration
            var acc = dr.mult(dr.invSumCube() * -G * b.m);
            // update velocity
            a.v.addTo(acc.mult(dt));
          }
        });
        // update position
        a.p.addTo(a.v.mult(dt));
      }
    });
  }

  ntrp(x, xa, xb, ya, yb) {
    return (x - xa) * (yb - ya) / (xb - xa) + ya;
  }

  rescale_pt(v, m) {
    return this.ntrp(v, 0, m, 0, this.image_scale);
  }

  rescale(p, m) {
    var x = this.rescale_pt(p.x, m);
    var y = this.rescale_pt(p.y, m);
    var v = new Vect(x, y, 0);
    v = this.center(v);
    return v;
  }

  center(p) {
    return new Vect(p.x + this.screen_center.x, this.screen_center.y - p.y, 0);
  }

  show_text(ctx, x, y, text) {
    ctx.fillStyle = 'white';
    ctx.fillText(text, x + 4, y + 6);
  }

  format_time(t) {
    var secs = t % 60;
    t = parseInt(t / 60);
    var mins = t % 60;
    t = parseInt(t / 60);
    var hours = t % 24;
    t = parseInt(t / 24);
    var days = t % 365;
    t = parseInt(t / 365);
    return (t + ' year(s) ' + days + ' days')
  }

  draw_space() {
    if (!this.ctx) {
      return;
    }
    // erase display area
    this.ctx.save();
    this.ctx.globalCompositeOperation = "source-over";
    this.ctx.fillStyle = 'rgb(0,0,0)';
    this.ctx.fillRect(0, 0, this.hs, this.vs);
    this.ctx.restore();
    //this.ctx.globalCompositeOperation = "lighter";

    this.planets.bodies.forEach((p) => {
      if (p.active || p.name == "Sun") {
        var oklabel = true;
        // hide Jupiter label when overlaid
        if (p == this.planets.jupiter && this.planets.spacecraft.active) {
          var delta = (this.planets.jupiter.p.sub(this.planets.spacecraft.p)).mag()
          oklabel = delta > 1e11;
        }
        // set origin location in display
        var pos = p.p.sub(new Vect(80e10, -15e10, 0));
        var disp_radius = this.image_scale * 3e9;
        var ppos = this.rescale(pos, disp_radius);
        p.sp = ppos;
        this.ctx.fillStyle = p.col;
        var ps = this.ntrp(p.rb, 0, this.planets.sun.rb, 2, this.body_radius);
        this.ctx.beginPath();
        this.ctx.arc(ppos.x, ppos.y, ps, 0, this.pi2);
        this.ctx.stroke();
        this.ctx.fill();
        if (oklabel) {
          ps += (p == this.planets.sun) ? -45 : 0;
          this.show_text(this.ctx, ppos.x + ps, ppos.y, p.name);
          if (p.special) {
            var s = "v:" + p.v.mag().toExponential(4) + " m/s";
            this.show_text(this.ctx, ppos.x + ps, ppos.y + 16, s);
          }
        }
      }
    });
    var lbl = "Duration: " + this.format_time(this.etime);
    this.show_text(this.ctx, 4, 18, lbl);
  }


  process() {
    var sc = this.planets.spacecraft;
    var ea = this.planets.earth;
    if (this.running) {
      if (this.n % this.div == 0) {
        this.draw_space();
      }
      this.perform_physics(this.dt);
      // etime units are seconds
      this.etime += this.dt;
      var day = this.etime / (24 * 3600);
      // these specific numbers are critical to the outcome
      // set spacecraft launch day to 189.55, more or less
      // for reverse path, use 189.73
      if (!sc.active && day >= 189.55) {
        sc.p = ea.p.mult(1.01);
        sc.v = ea.v.mult(1.35);
        sc.active = true;
      }
      // set a sequence duration
      // was var duration = 3.3 * 365.25 * 24 * 3600;
      var duration = 3.3 * 365.25 * 24 * 3600;
      if (this.etime >= duration) {
        this.running = false;
      }
      this.n += 1;
    }
  }


  static addEvent(o, e, f) {
    if (o.addEventListener) {
      o.addEventListener(e, f, false);
      return true;
    }
    else if (o.attachEvent) {
      return o.attachEvent("on" + e, f);
    }
    else {
      return false;
    }
  }
}

// distinguish between node.js and browser

if (typeof window !== 'undefined') {
  Orbit.addEvent(window, 'load', function () {
    //console.log("entering addEvent");
    window.requestAnimationFrame = window.requestAnimationFrame
      || window.mozRequestAnimationFrame
      || window.webkitRequestAnimationFrame
      || window.msRequestAnimationFrame;
    var ob = new Orbit("displaycanvas");
    //mydraw();
  });
}

// meet node.js requirement
// this is ES6 syntax

if (typeof _8d3‍.g.module !== 'undefined') {
  _8d3‍.t("module").exports = { Orbit };
}



