/***************************************************************************
 *   Copyright (C) 2021, Paul Lutus                                        *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

"use strict"

let { Vect3d } = require(`./vect3d.js`);

class Main {

  // format scientific notation with p decimal places
  sci(v, p) {
    return (v.toExponential(p));
  }

  // format fixed-place notation with p decimal places
  fix(v, p) {
    return (v.toFixed(p));
  }

  // compute maximum and minimum in one go
  maxmin(v, maxv, minv) {
    return [Math.max(maxv, v), Math.min(minv, v)];
  }

  // function derived from the vis-viva equation
  // https://en.wikipedia.org/wiki/Vis-viva_equation
  // for initial orbital velocity in an elliptical orbit
  //   G = universal gravitational constant
  //   m = mass of central body
  //   a = aphelion or perihelion radius
  //   p = perihelion or aphelion radius
  //   a and p are interchangeable based on
  //   the desired outcome
  // position orbiting body initially at the a radius

  eov(G, m, a, p) {
    return Math.sqrt((2.0 * G * m * p) / (a * (a + p)));
  }

  // potential energy
  pe(G, m1, m2, r) {
    return -G * m1 * m2 / r;
  }

  // kinetic energy
  ke(m, v) {
    return m * v * v / 2.0;
  }

  main() {
    // `Big G`, universal gravitational constant
    // https://physics.nist.gov/cgi-bin/cuu/Value?bg

    const G = 6.67430e-11;

    // model time slice seconds
    const Δt = 10.0;

    // Sun: https://nssdc.gsfc.nasa.gov/planetary/factsheet/sunfact.html
    // Earth: https://nssdc.gsfc.nasa.gov/planetary/factsheet/earthfact.html

    let sunmass = 1.9885e30;
    let eph = 147.095e9; // earth perihelion meters
    let eah = 152.100e9; // earth aphelion meters
    let emass = 5.9722e24; // earth mass kilograms

    // my own orbital function for initial orbital velocity
    // in an elliptical orbit

    // aphelion velocity
    let elv = this.eov(G, sunmass, eah, eph);

    // perihelion velocity
    let ehv = this.eov(G, sunmass, eph, eah);

    // position vector
    let pos = new Vect3d(
      eah,
      0.0,
      0.0,
    );

    // velocity vector
    let vel = new Vect3d(
      0.0,
      elv,
      0.0,
    );

    let init_ke = this.ke(emass, elv);
    let init_pe = this.pe(G, emass, sunmass, eah);
    let init_e = init_ke + init_pe;

    console.log(`Model: Earth orbit`);
    console.log(`Step Δt, seconds: ${this.fix(Δt, 2)}`);
    console.log(`Calculated/measured orbital parameters:`);
    console.log(`  Orbital radius meters:`);
    console.log(`    max (aphelion):   ${this.sci(eah, 6)}`);
    console.log(`    min (perihelion): ${this.sci(eph, 6)}`);
    console.log(`  Orbital velocity m/s:`);
    console.log(`    max: ${this.sci(ehv, 6)}`);
    console.log(`    min: ${this.sci(elv, 6)}`);
    console.log(`  Orbital energy joules (should be constant):`);
    console.log(`    mean: ${this.sci(init_e, 6)}`);

    // 1 year orbit in seconds.
    // for students:
    // one year
    //   * days in year
    //   * hours in day
    //   * minutes in hour
    //   * seconds in minute
    let duration = 1.0 * 365.25 * 24.0 * 60.0 * 60.0;

    // model elapsed time seconds
    let elapsed_time = 0.0;

    // accumulators for tracking maxima and minima
    let maxr = -1e300;
    let minr = 1e300;
    let maxv = -1e300;
    let minv = 1e300;
    let maxe = -1e300;
    let mine = 1e300;

    let iter = 0;

    let start = Date.now();

    let accel_term = -G * sunmass * Δt;

    while (true) {

      // compute acceleration
      let acc = pos.mult(pos.acceleration() * accel_term);

      // update velocity
      vel.addTo(acc);

      // collect velocity min/max data
      let mv = vel.mag();
      [maxv, minv] = this.maxmin(mv, maxv, minv);
      //minv = Math.min(minv, mv);

      // update position
      pos.addTo(vel.mult(Δt));

      // collect radius min/max data
      let r = pos.mag();
      [maxr, minr] = this.maxmin(r, maxr, minr);
      //minr = Math.min(minr, mr);

      // kinetic energy
      let ke = this.ke(emass, vel.mag());

      // potential energy
      let pe = this.pe(G, emass, sunmass, r);

      [maxe, mine] = this.maxmin(ke + pe, maxe, mine);

      // update elapsed time
      elapsed_time += Δt;
      if (elapsed_time > duration) {
        break;
      }
      iter += 1;
    }

    let end = Date.now();

    let error = 100.0 * Math.abs(maxe - mine) / Math.abs(maxe);

    console.log(`Simulation results:`);
    console.log(`  Iterations: ${iter}`);
    console.log(`  Orbital radius meters:`);
    console.log(`    max: ${this.sci(maxr, 6)}`);
    console.log(`    min: ${this.sci(minr, 6)}`);
    console.log(`  Orbital velocity m/s:`);
    console.log(`    max: ${this.sci(maxv, 6)}`);
    console.log(`    min: ${this.sci(minv, 6)}`);
    console.log(`  Orbital energy joules:`);
    console.log(`    max:   ${this.sci(maxe, 6)}`);
    console.log(`    min:   ${this.sci(mine, 6)}`);
    console.log(`    error: ${this.fix(error, 7)}%`);

    console.log(`\nElapsed algorithm time: ${this.fix(end - start, 3)} ms`);
  }

}

let m = new Main();
m.main();