final public class OrbitalAnalysis {

  public OrbitalAnalysis() {

  }

  // "Big G", universal gravitational constant
  // https://physics.nist.gov/cgi-bin/cuu/Value?bg

  private double G = 6.67430e-11;

  // model time slice seconds
  private double Δt = 10.0;

  // Sun: https://nssdc.gsfc.nasa.gov/planetary/factsheet/sunfact.html

  private double sunmass = 1.9885e30;

  // Earth: https://nssdc.gsfc.nasa.gov/planetary/factsheet/earthfact.html

  private double pph = 147.095e9; // earth perihelion meters
  private double pah = 152.100e9; // earth aphelion meters
  private double pmass = 5.9722e24; // earth mass kilograms

  private double[] maxmin(double v, double maxv, double minv) {
    maxv = Math.max(maxv, v);
    minv = Math.min(minv, v);
    return new double[] { maxv, minv };
  }

  // function derived from the vis-viva equation
  // https://en.wikipedia.org/wiki/Vis-viva_equation
  // for initial orbital velocity in an elliptical orbit
  // G = universal gravitational constant
  // m = mass of central body
  // a = aphelion or perihelion radius
  // p = perihelion or aphelion radius
  // a and p are interchangeable based on
  // the desired outcome
  // position orbiting body initially at the a radius
  private double eov(double G, double m, double a, double p) {
    return Math.sqrt((2.0 * G * m * p) / (a * (a + p)));
  }

  // potential energy
  private double pe(double G, double m1, double m2, double r) {
    return -G * m1 * m2 / r;
  }

  // kinetic energy
  private double ke(double m, double v) {
    return m * v * v / 2.0;
  }

  public void performOrbit() {
    // my own orbital function for initial orbital velocity
    // in an elliptical orbit
    // aphelion velocity
    double plv = eov(G, sunmass, pah, pph);
    // perihelion velocity
    double phv = eov(G, sunmass, pph, pah);

    Vect3d pos = new Vect3d(pah, 0, 0);

    Vect3d vel = new Vect3d(0, plv, 0);

    double init_ke = ke(pmass, plv);
    double init_pe = pe(G, pmass, sunmass, pah);
    double init_e = init_ke + init_pe;

    System.out.printf("Model: Earth orbit\n");
    System.out.printf("Simulator step size Δt: %.2f seconds\n", Δt);
    System.out.printf("Calculated/measured orbital parameters:\n");
    System.out.printf("  Orbital radius meters:\n");
    System.out.printf("    max (aphelion):   %.6e\n", pah);
    System.out.printf("    min (perihelion): %.6e\n", pph);
    System.out.printf("  Orbital velocity m/s:\n");
    System.out.printf("    max: %.6e\n", phv);
    System.out.printf("    min: %.6e\n", plv);
    System.out.printf("  Orbital energy joules (should be constant):\n");
    System.out.printf("    mean: %.6e\n", init_e);

    // 1 year orbit in seconds.
    // for students:
    // one year
    // * days in year
    // * hours in day
    // * minutes in hour
    // * seconds in minute
    double duration = 1.0 * 365.25 * 24.0 * 60.0 * 60.0;

    // model elapsed time seconds
    double elapsed_time = 0.0;

    // accumulators for tracking maxima and minima
    double maxr = -1e300;
    double minr = 1e300;
    double maxv = -1e300;
    double minv = 1e300;
    double maxe = -1e300;
    double mine = 1e300;

    int iter = 0;

    long start = System.nanoTime();

    double accel_term = -G * sunmass * Δt;

    while (true) {
      // compute acceleration
      Vect3d acc = pos.mult(pos.acceleration() * accel_term);

      // update velocity
      vel.addTo(acc);

      // collect velocity min/max data
      double mv = vel.mag();
      double[] a = maxmin(mv, maxv, minv);
      maxv = a[0];
      minv = a[1];

      // update position
      pos.addTo(vel.mult(Δt));

      // update orbital radius min/max data
      double r = pos.mag();
      a = maxmin(r, maxr, minr);
      maxr = a[0];
      minr = a[1];

      // kinetic energy
      double ke = ke(pmass, vel.mag());

      // potential energy
      double pe = pe(G, pmass, sunmass, r);

      // sum of ke and pe should be constant
      a = maxmin(ke + pe, maxe, mine);
      maxe = a[0];
      mine = a[1];

      // update elapsed time
      elapsed_time += Δt;
      if (elapsed_time > duration) {
        break;
      }
      iter += 1;

    }

    long end = System.nanoTime();

    double error = 100.0 * Math.abs(maxe - mine) / Math.abs(maxe);

    System.out.printf("Simulation results:\n");
    System.out.printf("  Iterations: %d\n", iter);
    System.out.printf("  Orbital radius meters:\n");
    System.out.printf("    max: %.6e\n", maxr);
    System.out.printf("    min: %.6e\n", minr);
    System.out.printf("  Orbital velocity m/s:\n");
    System.out.printf("    max: %.6e\n", maxv);
    System.out.printf("    min: %.6e\n", minv);
    System.out.printf("  Orbital energy joules:\n");
    System.out.printf("    max:   %.6e\n", maxe);
    System.out.printf("    min:   %.6e\n", mine);
    System.out.printf("    error: %.7f%%\n", error);

    double elapsed_ms = (end - start)/1e6;

    System.out.printf("\nElapsed algorithm time: %.3f ms\n", elapsed_ms);

  } // end of performOrbit()

  public static void main(String[] args) {
    OrbitalAnalysis oa = new OrbitalAnalysis();
    oa.performOrbit();

  }

}
