#include <iostream>
#include <cmath>
#define FMT_HEADER_ONLY
#include <fmt/format.h>
#include <fmt/core.h>
#include "vect3d.h"
#include <ctime>

using namespace std;

// function derived from the vis-viva equation
// https://en.wikipedia.org/wiki/Vis-viva_equation
// for initial orbital velocity in an elliptical orbit
//   G = universal gravitational constant
//   m = mass of central body
//   a = aphelion or perihelion radius
//   p = perihelion or aphelion radius
//   a and p are interchangeable based on
//   the desired outcome
// position orbiting body initially at the a radius

double eov(double G, double m, double a, double p)
{
  return sqrt((2.0 * G * m * p) / (a * (a + p)));
}

// potential energy
double pe(double G, double m1, double m2, double r)
{
  return -G * m1 * m2 / r;
}

// kinetic energy
double ke(double m, double v)
{
  return m * v * v / 2.0;
}

void maxmin(double v, double &maxv, double &minv)
{
  maxv = max(maxv, v);
  minv = min(minv, v);
}

int main()
{

  // "Big G", universal gravitational constant
  // https://physics.nist.gov/cgi-bin/cuu/Value?bg
  double G = 6.67430e-11;

  // model time slice seconds
  double Δt = 10.0;

  // Sun: https://nssdc.gsfc.nasa.gov/planetary/factsheet/sunfact.html
  // Earth: https://nssdc.gsfc.nasa.gov/planetary/factsheet/earthfact.html

  double sunmass = 1.9885e30;
  double eph = 147.095e9; // earth perihelion meters
  double eah = 152.100e9; // earth aphelion meters
  double emass = 5.9722e24; // earth mass kilograms

  // my own orbital function for initial orbital velocity
  // in an elliptical orbit

  // aphelion velocity
  double elv = eov(G, sunmass, eah, eph);
  // perihelion velocity
  double ehv = eov(G, sunmass, eph, eah);

  // position vector
  Vect3d pos = Vect3d(
      eah,
      0.0,
      0.0);

  // velocity vector
  Vect3d vel = Vect3d{
      0.0,
      elv,
      0.0};

  double init_ke = ke(emass,elv);
  double init_pe = pe(G,emass,sunmass,eah);
  double init_e = init_ke+init_pe;

  fmt::print("Model: Earth orbit\n");
  fmt::print("Simulator step size Δt: {:.2} seconds\n",Δt);
  fmt::print("Calculated/measured orbital parameters:\n");
  fmt::print("  Orbital radius meters:\n");
  fmt::print("    max (aphelion):   {:.6e}\n",eah);
  fmt::print("    min (perihelion): {:.6e}\n",eph);
  fmt::print("  Orbital velocity m/s:\n");
  fmt::print("    max: {:.6e}\n",ehv);
  fmt::print("    min: {:.6e}\n",elv);
  fmt::print("  Orbital energy joules (should be constant):\n");
  fmt::print("    mean: {:.6e}\n",init_e);

  
  // 1 year orbit in seconds.
  // for students:
  // one year
  //   * days in year
  //   * hours in day
  //   * minutes in hour
  //   * seconds in minute
  double duration = 1.0 * 365.25 * 24.0 * 60.0 * 60.0;

  // model elapsed time seconds
  double elapsed_time = 0.0;

  // accumulators for measuring maxima and minima
  double maxr = 0.0;
  double minr = 1e300;
  double maxv = 0.0;
  double minv = 1e300;
  double maxe = -1e300;
  double mine = 1e300;

  int iter = 0;

  clock_t begin = clock();

  double accel_term = -G * sunmass * Δt;

  while (true)
  {
    // compute acceleration
    Vect3d acc = pos * pos.acceleration() * accel_term;

    // update velocity
    vel += acc;

    // collect velocity min/max data
    double mv = vel.mag();
    maxmin(mv,maxv, minv);

    // update position
    pos += vel * Δt;

    // collect radius min/max data
    double r = pos.mag();
    maxmin(r,maxr, minr);

    // kinetic energy
    double dke = ke(emass, vel.mag());

    // potential energy
    double dpe = pe(G, emass, sunmass, r);

    // sum of ke and pe should be constant
    maxmin(dke + dpe, maxe, mine);

    // update elapsed time
    elapsed_time += Δt;
    if (elapsed_time > duration)
    {
      break;
    }
    iter += 1;
  }

  clock_t end = clock();

  double error = 100.0 * abs(maxe - mine) / abs(maxe);

  fmt::print("Simulation results:\n");
  fmt::print("  Iterations: {}\n",iter);
  fmt::print("  Orbital Radius meters:\n");
  fmt::print("    max: {:.6e}\n",maxr);
  fmt::print("    min: {:.6e}\n",minr);
  fmt::print("  Orbital velocity m/s:\n");
  fmt::print("    max: {:.6e}\n",maxv);
  fmt::print("    min: {:.6e}\n",minv);
  fmt::print("  Orbital energy joules:\n");
  fmt::print("    max:   {:.6e}\n",maxe);
  fmt::print("    min:   {:.6e}\n",mine);
  fmt::print("    error: {:.7f}%\n",error);

  double elapsed = (end-begin)/1000.0;

   fmt::print("\nElapsed algorithm time: {:.3f} ms\n", elapsed);

  return 0;
}
