/***************************************************************************
 *   Copyright (C) 2020 by Paul Lutus                                      *
 *   lutusp@arachnoid.com                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

"use strict"

/*

Reminder: these are the 8bitdo codes

In keyboard mode:

Select button = o (lowercase O)
Start button = n

Letter buttons:

Y = i
B = j
A = g
X = h

Main Buttons on top:

Left = k
Right = m

Joystick or "D pad":

Up = c
Right = f
Down = d
Left = e

 */

class TP {
  constructor() {
    this.setDefaults();
    
    this.readStorage();
    
    this.readLocalFile('content.txt');
    
    document.onkeydown = this.keyDown.bind(this);
    
    document.onkeypress = this.keyPress.bind(this);
    
    this.reset();
  }
  
  // some shorthand
  
  getID(s) {
    return document.getElementById(s);
  }
  
  setDefaults() {
    // enable debug printing
    this.debug = false;
    this.clearPageScroll();
    this.accept = false;
    this.intervalTimer = false;
    this.displaying = false;
    this.pauseFlag = true; // press space bar to start
    this.shiftFlag = false;
    this.padLeftTrigger = false;
    this.fullScreenFlag = false;
    this.delayTime = 20; // default, milliseconds
    this.fontSize = 22;
    this.hborder = 10; // pixels
    this.vborder = 10; // pixels
    this.borderModified = false;
    this.reverseXFlag = false;
    this.reverseYFlag = false;
    this.yPos = 0;
    this.maxScrollY = 0;
    this.content = false;
    this.prefix = "TelePrompterData.";
    this.body = this.getID('body');
    this.outer = this.getID('outer');
    this.menu = this.getID('menu');
    this.displayContent = this.getID('displayContent');
    this.userContent = this.getID('userContent');
    this.userContent.addEventListener('paste',(event) => {
        // this action is being debated:
        // automatically erase all prior content
        // on a paste action
        //this.userContent.value = '';
        
        // scroll to top of pasted content
        // must be delayed as shown
        window.setTimeout(function() { this.userContent.scrollTop = 0; },100);
    });
    var _parent_this = this; // needed for nested "this" reference below
    // a user button press creates a file dialog
    this.getID('fileButton').onclick = function() {
      // create a file input dialog
      var fileInput = document.createElement("input");
      fileInput.type = "file";
      fileInput.accept = ".txt";
      fileInput.addEventListener('change', function() {
          // on user selection, read the file
          var fr=new FileReader();
          fr.onload=function(){
            _parent_this.userContent.value=fr.result;
          }
          fr.readAsText(this.files[0]);
      })
      // launch the file dialog
      fileInput.click();
    }
    
    this.setBorder();
    this.displayUserContent(false);
    this.setfontSize();
  }
  
  debugText(s) {
    if(this.debug) {
      console.log(s);
    }
  }
  
  setPageScroll() {
    this.clearPageScroll();
    if(this.displaying) {
      this.delayTime = (this.delayTime > 100)?100:this.delayTime;
      this.delayTime = (this.delayTime < 1.1)?1.1:this.delayTime;
      this.intervalTimer = setInterval(this.pageScroll.bind(this),this.delayTime);
    }
  }
  
  clearPageScroll() {
    if(this.intervalTimer) {
      clearInterval(this.intervalTimer);
      this.intervalTimer = false;
    }
  }
  
  pageScroll() {
    if(!this.pauseFlag) {
      var sz = (this.shiftFlag|this.padLeftTrigger)?-1:1;
      window.scroll(0,this.yPos);
      this.yPos += this.reverseYFlag?-sz:sz;
      this.yPos = Math.max(0,this.yPos);
      this.yPos = Math.min(this.maxScrollY,this.yPos);
    }
  }
  
  // specal handling for arrow keys
  keyDown(event) {
    var c = event.keyCode;
    this.debugText('keydown: ' + c)
    // include alternate rubout and linefeed characters for browser translations
    if(c == 37 || c == 38 || c == 39 || c == 40 || c == 10 || c == 13 || c == 127 || c == 8) {
      this.keyPress(event);
      return false;
    }
    return true;
  }
  
  // start display if not enabled
  // or run/pause if enabled
  
  showPause(c) {
    if(this.displaying) {
      // only reset the shift flag here
      this.shiftFlag = event.shiftKey;
      this.pauseFlag = !this.pauseFlag;
      this.pageScroll();
      this.accept = true;
    }
    else {
      this.displaying = true;
      this.pauseFlag = true;
      this.loadContent();
      this.displayUserContent(true);
      this.reset();
      this.accept = true;
    }
  }
  
  keyPress(event) {
    this.accept = false;
    this.getMaxScrollY();
    this.debugText("keypress: " + event.keyCode);
    this.debugText("key: " + event.key);
    var c = event.keyCode;
    var k = event.key;
    // shift user-entered letters to lowercase
    if(c >= 65 && c <= 90) {
      c += 32;
    }
    this.debugText('shifted result: ' + c);
    switch (c) {
      
      // reverse X left/right
      case 120: // 'x'
      case 101: // 'e': 8bitdo joystick left
      case 102: // 'f': 8bitdo joystick right
        this.pauseFlag = true;
      this.reverseXFlag = !this.reverseXFlag;
      this.accept = true;
      break;
      
      // reverse top/bottom
      case 121: // 'y'
      case 99: // 'c': 8bitdo joystick up
      case 100: // 'd': 8bitdo joystick down
        this.pauseFlag = true;
      this.reverseYFlag = !this.reverseYFlag;
      this.scrollToTop();
      this.accept = true;
      break;
      
      // increase H margins
      case 125: // '}'
      if(this.displaying) {
        this.pauseFlag = true;
        this.hborder += 5;
        this.borderModified = true;
        this.accept = true;
      }
      break;
      
      // decrease H margins
      case 123: // '{'
      if(this.displaying) {
        this.pauseFlag = true;
        this.hborder -= 5;
        this.borderModified = true;
        this.accept = true;
      }
      break;
      
      // increase V margins
      case 93: // ']'
      if(this.displaying) {
        this.pauseFlag = true;
        this.vborder += 5;
        this.borderModified = true;
        this.accept = true;
      }
      break;
      
      // decrease V margins
      case 91: // '[' decrease V margin
      if(this.displaying) {
        this.pauseFlag = true;
        this.vborder -= 5;
        this.borderModified = true;
        this.accept = true;
      }
      break;
      
      // start, then scroll / pause
      case 115: // 's' start/restart
      case 32: // ' ', space
      case 111: // 'o': 8bitdo 'start' button
      case 109: // 'm': 8bitdo right trigger
      case 107: // 'k': 8bitdo left trigger
        this.padLeftTrigger = (c == 107);
      this.showPause(c);
      break;
      
      // increase scroll speed
      case 38: // conventional keyboard up-arrow
      case 105: // 8bitdo 'Y' in keyboard mode
      if(this.displaying && !this.pauseFlag) {
        this.delayTime *= 0.8;
      }
      this.accept = true;
      break;
      
      // decrease scroll speed
      case 40: // conventional keyboard down-arrow
      case 103: // 8bitdo 'A' in keyboard mode
      if(this.displaying && !this.pauseFlag) {
        this.delayTime *= 1.2;
      }
      this.accept = true;
      break;
      
      // decrease font size
      case 37: // conventional keyboard left-arrow
      case 104: // 8bitdo 'X' in keyboard mode
      if(this.displaying) {
        this.setfontSize(-1);
        this.accept = true;
      }
      break;
      
      // increase font size
      case 39: // conventional keyboard right-arrow
      case 106: // 8bitdo 'B' in keyboard mode
      if(this.displaying) {
        this.setfontSize(1);
        this.accept = true;
      }
      break;
      
      case 113: // 'q' quit, reloads help page
        //case 27: // Esc, also exits full-screen
      case 110: // '~': 8bitdo select button
        this.displaying = false;
      this.pauseFlag = true;
      this.displayUserContent(false);
      this.reset();
      this.accept = (c == 113 || c == 110);
      break;
      
      // full screen
      case 119: // 'w'
        this.fullScreenToggle();
      this.accept = true;
      break;
      
      // reset to all default values
      case 114: // 'r'
      if(confirm('Reset to all defaults?')) {
        this.setDefaults();
        this.displayUserContent(false);
      }
      this.accept = true;
      break;
      
      // clear user-entered text
      case 122: // 'z': clear text
      if(confirm('Clear user text?')) {
        this.userContent.value = '';
        this.loadContent();
        this.displayUserContent(false);
      }
      this.accept = true;
      break;
    }
    
    if(this.borderModified) {
      this.scrollToTop();
      this.borderModified = false;
    }
    this.setPageScroll();
    this.setBorder();
    this.setRotate();
    return !this.accept; // prevent propagation of used keystrokes
  }
  
  
  fullScreenToggle() {
    this.fullScreenFlag = !this.fullScreenFlag;
    if (this.fullScreenFlag) {
      var docElm = document.documentElement;
      if (docElm.requestFullscreen) {
        docElm.requestFullscreen();
      }
      else if (docElm.mozRequestFullScreen) {
        docElm.mozRequestFullScreen();
      }
      else if (docElm.webkitRequestFullScreen) {
        docElm.webkitRequestFullScreen();
      }
      else if (docElm.msRequestFullscreen) {
        docElm.msRequestFullscreen();
      }
    }
    else {
      if (document.exitFullscreen) {
        document.exitFullscreen();
      }
      else if (document.mozCancelFullScreen) {
        document.mozCancelFullScreen();
      }
      else if (document.webkitCancelFullScreen) {
        document.webkitCancelFullScreen();
      }
      else if (document.msExitFullscreen) {
        document.msExitFullscreen();
      }
    }
    
  }
  
  setfontSize(n=0) {
    this.fontSize += n;
    this.fontSize = (this.fontSize < 8)?8:this.fontSize;
    this.fontSize = (this.fontSize > 100)?100:this.fontSize;
    try {
      if(this.displayContent) {
        this.displayContent.style.fontSize = '' + this.fontSize + 'pt';
      }
    }
    catch(err) {
      console.log(err);
    }
  }
  
  setRotate() {
    try {
      var rotX = (this.reverseXFlag)?180:0;
      var rotY = (this.reverseYFlag)?180:0;
      // the order of arguments is not an error -- think about it
      if (this.outer) {
        this.outer.style.transform = 'rotateX('+rotY+'deg) rotateY('+rotX+'deg)';
      }
    }
    catch(err) {
      console.log(err);
    }
  }
  
  minmax(x,min,max) {
    x = Math.min(max,x);
    x = Math.max(x,min);
    return x;
  }
  
  setBorder() {
    var minb = 4;
    this.hborder = this.minmax(this.hborder,minb,500);
    this.vborder = this.minmax(this.vborder,minb,2000);
    var hb = minb;
    var vb = minb;
    if(this.displaying) {
      hb = this.hborder;
      vb = this.vborder;
    }
    if(this.body) {
      this.body.style.borderBottom = vb + 'px solid black';
      this.body.style.borderTop = vb + 'px solid black';
      this.body.style.borderLeft = hb + 'px solid black';
      this.body.style.borderRight = hb + 'px solid black';
    }
  }
  
  getMaxScrollY() {
    var y = window.scrollY;
    window.scroll(0,1000000);
    this.maxScrollY = window.scrollY;
    //console.log(this.maxScrollY);
    window.scroll(0,y);
  }
  
  scrollToTop() {
    this.getMaxScrollY();
    //console.log("rev y = "+this.reverseYFlag);
    //console.log("max y = "+this.maxScrollY);
    this.yPos = (this.reverseYFlag)?this.maxScrollY:0;
    window.scroll(0,this.yPos);
  }
  
  reset() {
    this.setRotate();
    this.setfontSize();
    this.setBorder();
    this.scrollToTop();
  }
  
  loadContent() {
    var text = this.userContent.value.trim();
    this.userContent.value = text;
    var markup = text.replace(/\n/g,"\n<br/>");
    this.displayContent.innerHTML = markup;
  }
  
  displayUserContent(state) {
    if(state && this.userContent.value == '') {
      alert('Please paste your text into the user text area.');
      state = false;
    }
    this.displaying = state;
    if(state) {
      this.menu.style.display = "none";
      this.displayContent.style.display = "block";
      body.style.overflowY = 'hidden';
    }
    else {
      //console.log('show menu');
      this.clearPageScroll();
      this.displayContent.innerHTML = '';
      this.displayContent.style.display = "none";
      this.menu.style.display = "block";
      this.userContent.display = 'none';
      body.style.overflowY = '';
    }
    this.setBorder();
  }
  
  // if local file 'content.txt' exists, this routine
  // will read it and, if it's different than the saved text,
  // will offer it to the user
  
  readLocalFile(path,clear = false) {
    this.readLocalFileRequest(path,this.readLocalFileCallback.bind(this),clear);
  }
  
  // Javascript may read a local file
  // this way, carefully
  
  readLocalFileRequest(path, callback,clear) {
    var xhr = new XMLHttpRequest();
    xhr.onreadystatechange = function() {
      if (xhr.readyState == 4 && xhr.status == 200) {
        // success
        callback(xhr.responseText,clear);
        } else {
        // fail
        callback(null);
      }
    };
    xhr.open("GET", path);
    xhr.send();
  }
  
  // execute the following rule:
  // if a local file exists
  // and if the user area is empty
  // then use the local file's contents
  
  readLocalFileCallback(txtfile,clear = false) {
    if(!txtfile || txtfile.trim().length == 0) return;
    var current = this.userContent.value.trim();
    if (current.length == 0 || clear) {
      this.userContent.value = txtfile.trim();
    }
  }
  
  readItem(tag,defaultValue) {
    var result = window.localStorage.getItem(this.prefix + tag);
    if(null == result) {
      result = defaultValue;
    }
    return result;
  }
  
  readStorage() {
    if(window.localStorage) {
      this.userContent.value = this.readItem('text',this.userContent.value).trim();
      this.vborder = parseFloat(this.readItem('vb',this.vborder));
      this.hborder = parseFloat(this.readItem('hb',this.hborder));
      this.fontSize = parseFloat(this.readItem('fs',this.fontSize));
      this.delayTime = parseFloat(this.readItem('dt',this.delayTime));
      this.reverseXFlag = this.readItem('X',this.reverseXFlag) == 'true';
      this.reverseYFlag = this.readItem('Y',this.reverseYFlag) == 'true';
    }
  }
  
  writeItem(tag,data) {
    window.localStorage.setItem(this.prefix + tag,''+data);
  }
  
  writeStorage() {
    if(window.localStorage) {
      this.writeItem('text',this.userContent.value.trim());
      this.writeItem('vb',this.vborder);
      this.writeItem('hb',this.hborder);
      this.writeItem('fs',this.fontSize);
      this.writeItem('dt',this.delayTime);
      this.writeItem('X',this.reverseXFlag);
      this.writeItem('Y',this.reverseYFlag);
    }
  }
}

var tp = false;

window.addEventListener('load',function() { tp = new TP(); });
window.addEventListener('unload',function() { if(tp) {tp.writeStorage();} });
