/***************************************************************************
 *   Copyright (C) 2008 by Paul Lutus                                      *
 *   lutusp@pl-alpha                                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/


#include "gravity_class.h"

const string Gravity::version = "1.5";

const  string Gravity::anim_strings[] =
    { "7.5 minutes","15 minutes","30 minutes","1 hour","2 hours","4 hours","8 hours","16 hours",
      "1 day","2 days","4 days","8 days","16 days","32 days","64 days","128 days","256 days",""
    };

const string Gravity::comet_strings[] = { "1","2","4","8","16","32","64","128","256","512","1024","2048","4096","8192","" };

Gravity::Gravity ( BaseObjectType* base_object,
                   const Glib::RefPtr<Gnome::Glade::Xml>& glade_xml )
		: Gtk::Window ( base_object )
{

// create color table for planet display

	int temp[8][3] =
	{
		{255,255,255},
		{255,255,0},
		{0,255,255},
		{128,128,255},
		{255,0,0},
		{0,255,0},
		{255,0,255},
		{0,0,255}
	};

	for ( int i = 0;i < 8;i++ )
	{
		Gravity::planet_colors[i] = create_color ( temp[i][0],temp[i][1],temp[i][2] );
	}
	color_red = create_color ( 255,0,0 );
	color_cyan = create_color ( 0,255,255 );
	color_black = create_color ( 0,0,0 );
	color_white = create_color ( 255,255,255 );
	string title = ( string ) "Gravity " + version;
	set_title ( title.c_str() );
	mouse_down = false;
	sun_only = false;
	erase_option = true;
	symmetric = false;
	nice = true;
	total_time_hours = 0;
	gui_graphic_pane = NULL;
	thread_active = false;
	anim_time = 1; // ms
	oldXSize = -1;
	oldYSize = -1;
	rotx = -20;
	roty = 0;
	min_draw_radius = 5;
	anaglyph_mode = false;
	drawing_scale = 6e-12;
	rotator = new RotationMatrix();
	setup_controls ( glade_xml );
	set_time_step();
	gui_pixels_spinbutton->set_value ( 5 );
	load_objects();
	draw_image ( true );
}

Gravity::~Gravity()
{
	delete rotator;
	erase_planet_list();
}

void Gravity::setup_controls ( Glib::RefPtr<Gnome::Glade::Xml> glade_xml )
{
	// define local variables for controls
	glade_xml->get_widget ( "status_bar",gui_status_bar );
	glade_xml->get_widget ( "graphic_pane",gui_graphic_pane );
	glade_xml->get_widget ( "event_box",gui_event_box );
	glade_xml->get_widget ( "quit_button",gui_quit_button );
	glade_xml->get_widget ( "step_button",gui_step_button );
	glade_xml->get_widget ( "run_stop_button",gui_run_stop_button );
	glade_xml->get_widget ( "solar_system_checkbutton",gui_solar_system_checkbutton );
	glade_xml->get_widget ( "comets_checkbutton",gui_comets_checkbutton );
	glade_xml->get_widget ( "anaglyphic_checkbutton",gui_anaglyphic_checkbutton );
	glade_xml->get_widget ( "trails_checkbutton",gui_trails_checkbutton );
	glade_xml->get_widget ( "nice_checkbutton",gui_nice_checkbutton );

	glade_xml->get_widget ( "time_step_combobox",gui_time_step_combobox );
	glade_xml->get_widget ( "comet_combobox",gui_comet_combobox );
	glade_xml->get_widget ( "pixels_spinbutton",gui_pixels_spinbutton );

	// load comboboxes
	load_combobox ( gui_time_step_combobox, ( const string* ) anim_strings,7 );
	load_combobox ( gui_comet_combobox, ( const string* ) comet_strings,4 );

	// connect controls to actions
	gui_quit_button->signal_clicked().connect ( sigc::mem_fun ( *this, &Gravity::close ) );
	gui_run_stop_button->signal_clicked().connect ( sigc::mem_fun ( *this, &Gravity::run_stop_button_clicked ) );
	gui_step_button->signal_clicked().connect ( sigc::mem_fun ( *this, &Gravity::step_button_clicked ) );
	gui_solar_system_checkbutton->signal_toggled ().connect ( sigc::mem_fun ( *this, &Gravity::solar_system_checkbutton_toggled ) );
	gui_comets_checkbutton->signal_clicked().connect ( sigc::mem_fun ( *this, &Gravity::comets_checkbutton_toggled ) );
	gui_anaglyphic_checkbutton->signal_clicked().connect ( sigc::mem_fun ( *this, &Gravity::anaglyphic_checkbutton_toggled ) );
	gui_trails_checkbutton->signal_clicked().connect ( sigc::mem_fun ( *this, &Gravity::trails_checkbutton_toggled ) );
	gui_nice_checkbutton->signal_clicked().connect ( sigc::mem_fun ( *this, &Gravity::nice_checkbutton_toggled ) );
	gui_time_step_combobox->signal_changed().connect ( sigc::mem_fun ( *this, &Gravity::time_step_combobox_changed ) );
	gui_comet_combobox->signal_changed().connect ( sigc::mem_fun ( *this, &Gravity::comets_combobox_changed ) );
	gui_pixels_spinbutton->signal_changed().connect ( sigc::mem_fun ( *this, &Gravity::pixels_spinbutton_changed ) );
	gui_event_box->signal_button_press_event().connect ( sigc::mem_fun ( *this, &Gravity::mouse_press_event ) );
	gui_event_box->signal_button_release_event().connect ( sigc::mem_fun ( *this, &Gravity::mouse_release_event ) );
	gui_event_box->signal_motion_notify_event().connect ( sigc::mem_fun ( *this, &Gravity::mouse_move_event ) );
	gui_event_box->signal_scroll_event().connect ( sigc::mem_fun ( *this, &Gravity::wheel_event ) );
	gui_graphic_pane->signal_expose_event().connect ( sigc::mem_fun ( *this, &Gravity::expose_event ) );

}

void Gravity::load_combobox ( Gtk::ComboBox *cb,const string* array, int n )
{
	Glib::RefPtr<Gtk::ListStore> model;
	model = Gtk::ListStore::create ( m_Columns );
	cb->set_model ( model );
	int i = 0;
	while ( array[i].size() > 0 )
	{
		Gtk::TreeModel::Row row = * ( model->append() );
		row[m_Columns.m_col_name] = array[i].c_str();
		i++;
	}
	// set default value
	cb->set_active ( n );
}

string Gravity::get_combobox_selection ( Gtk::ComboBox *cb )
{
	string value = "";
	Gtk::TreeModel::iterator iter = cb->get_active();
	if ( iter )
	{
		Gtk::TreeModel::Row row = *iter;
		if ( row )
		{
			Glib::ustring v = row[m_Columns.m_col_name];
			value = ( string ) v.c_str();
		}
	}
	return value;
}

void Gravity::close()
{
	hide();
}

// remember the Gdk::Color class uses rgb
// values that range between 0 - 65535

Gdk::Color Gravity::create_color ( int r, int g, int b )
{
	Gdk::Color col = Gdk::Color::Color();
	col.set_rgb ( r * 256,g * 256,b * 256 );
	return col;
}


void Gravity::erase_planet_list()
{
	for ( unsigned int i = 0;i < planet_list.size();i++ )
	{
		delete planet_list[i];
	}
	planet_list.erase ( planet_list.begin(),planet_list.end() );
}


void Gravity::show_status()
{
	// total_time units are hours, not seconds
	unsigned long y = total_time_hours;
	unsigned long h = y % 24;
	y /= 24;
	y *= 100;
	unsigned long d = ( y % 36525 ) / 100;
	y /= 36525;
	char buf[256];
	sprintf ( buf,"Elapsed Time: %04dy %03dd %02dh",y,d,h );
	gui_status_bar->set_label ( buf );
}


void Gravity::set_time_step()
{
	string s = get_combobox_selection ( gui_time_step_combobox );
	vector<string> v = Pattern::split ( " ",s );
	double n;
	sscanf ( v[0].c_str(),"%lf",&n );
	if ( Pattern::matches ( "^minutes*",v[1] ) )
	{
		n *= 60.0;
	}
	else if ( Pattern::matches ( "^hours*",v[1] ) )
	{
		n *= 3600.0;
	}
	else if ( Pattern::matches ( "^days*",v[1] ) )
	{
		n *= 86400.0;
	}
	time_step = n;
}

void Gravity::draw_planets ( int xsize,int ysize,Glib::RefPtr<Gdk::GC>& gc,Glib::RefPtr<Gdk::Pixmap>& pm,Gdk::Color td_color,int anaglyph_flag )
{
	int pcsz = sizeof ( planet_colors ) / sizeof ( Gdk::Color );
	if ( anaglyph_flag != 0 )
	{
		gc->set_rgb_fg_color ( td_color );
	}
	for ( unsigned int i = 0;i < planet_list.size();i++ )
	{
		Planet *planet = planet_list[i];
		Cart3 v = planet->pos * drawing_scale;
		rotator->rotate ( v );
		rotator->convert_3d_to_2d ( v,anaglyph_flag );
		int sxa = ( int ) ( x_screen_center + ( v.x * screen_scale ) );
		int sya = ( int ) ( y_screen_center - ( v.y * screen_scale ) );
		if ( sxa >= 0 && sxa < xsize && sya >= 0 && sya < ysize )
		{
			// fake the sun's radius for aesthetics
			double sr = ( i == 0 ) ?4e7:planet->radius;
			Cart3 s ( sr * drawing_scale,0,-planet->pos.z * drawing_scale );
			rotator->convert_3d_to_2d ( s );
			s.x *= screen_scale * 100;
			s.x = ( s.x < min_draw_radius ) ?min_draw_radius:s.x;
			int si = ( int ) s.x;
			int sc = si / 2;
			if ( anaglyph_flag == 0 )
			{
				gc->set_rgb_fg_color ( planet_colors[i % pcsz] );
			}
			pm->draw_arc ( gc,true,sxa-sc,sya-sc, ( int ) s.x, ( int ) s.x,0,23040 );
			//printf ( "%d,%d,%d,%f\n",anaglyph_flag,sxa-sc,sya-sc,s.x );
		}
	}
}

void Gravity::draw_image ( bool erase )
{
	show_status();
	rotator->populate_matrix ( rotx,roty );
	int xSize = gui_graphic_pane->get_width();
	int ySize = gui_graphic_pane->get_height();
	Glib::RefPtr<Gdk::Window> window = gui_graphic_pane->get_window();
	if ( oldXSize != xSize || oldYSize != ySize )
	{
		pixmap = Gdk::Pixmap::create ( window,xSize,ySize,-1 );
		oldXSize = xSize;
		oldYSize = ySize;
		x_screen_center = xSize / 2;
		y_screen_center = ySize / 2;
		screen_scale = ( x_screen_center > y_screen_center ) ?y_screen_center:x_screen_center;
		//qpm->fill(QColor(Qt::black));
	}
	Glib::RefPtr<Gdk::GC> gc = Gdk::GC::create ( pixmap );
	gc->set_function ( Gdk::COPY );
	if ( erase )
	{
		gc->set_rgb_fg_color ( color_black );
		pixmap->draw_rectangle ( gc,true,0,0,xSize,ySize );
	}
	if ( anaglyph_mode )
	{
		gc->set_function ( Gdk::OR );
		// draw complete, rotated right-hand and left-hand
		// images in cyan and red for anaglyphic glasses
		draw_planets ( xSize,ySize,gc,pixmap,color_cyan,-1 ); // right eye image
		draw_planets ( xSize,ySize,gc,pixmap,color_red,1 ); // left eye image
		gc->set_function ( Gdk::COPY );
	}
	else
	{
		// draw image once
		draw_planets ( xSize,ySize,gc,pixmap,color_white,0 );
	}
	// transfer result to display
	window->draw_drawable ( gc,pixmap,0,0,0,0,xSize,ySize );
	// total_time units are hours
	total_time_hours += ( unsigned long ) ( time_step / 3600 );
}

// data table has fields "Name","OrbitRad","BodyRad","Mass","OrbitVel"

void Gravity::load_orbital_data ( string data,bool sun_only )
{
	erase_planet_list();
	vector<string> array = Pattern::split ( "\\n",data );
	array.erase ( array.begin() ); // drop header line
	for ( unsigned int i = 0;i < array.size();i++ )
	{
		string line = array[i];
		vector<string> fields = Pattern::split ( ",",line );
		string name = fields[0];
		// 0 = orbitrad, 1 = bodyrad, 2 = mass, 3 = orbitvel
		double vals[4];
		for ( unsigned int j=1;j < fields.size();j++ )
		{
			sscanf ( fields[j].c_str(),"%lf",&vals[j-1] );
		}
		Cart3 pos ( -vals[0],0,0 );
		Cart3 vel ( 0,0,vals[3] );
		Planet *planet = new Planet ( fields[0],vals[1],pos,vel,vals[2] );
		planet_list.push_back ( planet );
		if ( sun_only )
		{
			break;
		}
	}
}

void Gravity::load_comets()
{
	string s = get_combobox_selection ( gui_comet_combobox );
	int n;
	sscanf ( s.c_str(),"%d",&n );
	fflush ( stdout );
	for ( int i = 0;i < n;i++ )
	{
		stringstream ss;
		string name;
		ss << "comet" << i;
		ss >> name;
		double ca = rand() % 360; // angle in x-z plane
		double cr = ( rand() % 100000 ) + 100000; // distance from sun
		cr *= 4e6;
		Cart3 pos ( cr * sin ( ca * ToRad ),0,cr * cos ( ca * ToRad ) );
		// comet initial velocity
		double v = ( ( rand() % 200 ) + 100 ) * 50.0;
		v = ( i % 2 == 1 ) ?-v:v;
		Cart3 vel ( 0,v,0 );
		Planet *comet = new Planet ( name,1e3,pos,vel,1e9 );
		planet_list.push_back ( comet );
	}
}

void Gravity::load_objects()
{
	erase_planet_list();
	sun_only = !gui_solar_system_checkbutton->get_active();
	load_orbital_data ( SolarSystem::data,sun_only );
	if ( gui_comets_checkbutton->get_active() )
	{
		load_comets() ;
	}
	draw_image ( true );
}

void Gravity::perform_orbit_calc()
{
	OrbitalPhysics::process_planets ( planet_list,time_step,symmetric );
	draw_image ( erase_option );
}

// this function is in global namespace

void* perform_thread ( void *args )
{
	// get running instance "this" pointer
	Gravity* gp = ( Gravity* ) args;

	while ( gp->thread_active )
	{
		// optional 5 ms sleep period to reduce CPU usage
		if ( gp->nice )
		{
			usleep ( 5000 );
		}
		gdk_threads_enter();
		if ( gtk_events_pending() )
		{
			gtk_main_iteration(); // Handle unprocessed GTK events
		}
		else
		{
			g_thread_yield();     // Yield processing time
		}
		gp->perform_orbit_calc();
		gdk_flush ();
		gdk_threads_leave();
	}
	return NULL;
}

void Gravity::toggle_animation()
{
	if ( thread_active )
	{
		thread_active = false;
	}
	else
	{
		thread_active = true;
		total_time_hours = 0;
		frame_counter = 0;
		g_thread_create ( perform_thread,this,false,NULL );
	}
	gui_run_stop_button->set_label ( ( thread_active ) ?"Stop":"Run" );
}

void Gravity::step_button_clicked()
{
	if ( thread_active )
	{
		toggle_animation();
		draw_image ( true );
	}
	perform_orbit_calc();
}

void Gravity::run_stop_button_clicked()
{
	toggle_animation();
	draw_image ( true );
}

void Gravity::solar_system_checkbutton_toggled()
{
	load_objects();
	draw_image ( true );
}

void Gravity::comets_checkbutton_toggled()
{
	load_objects();
	draw_image ( true );
}

void Gravity::comets_combobox_changed()
{
	gui_comets_checkbutton->set_active ( true );
	load_objects();
	draw_image ( true );
}

void Gravity::anaglyphic_checkbutton_toggled()
{
	anaglyph_mode = gui_anaglyphic_checkbutton->get_active();
	draw_image ( true );
}

void Gravity::trails_checkbutton_toggled()
{
	erase_option = !gui_trails_checkbutton->get_active();
	draw_image ( true );
}

void Gravity::nice_checkbutton_toggled()
{
	nice = gui_nice_checkbutton->get_active();
	draw_image ( true );
}

void Gravity::time_step_combobox_changed()
{
	set_time_step();
}

void Gravity::pixels_spinbutton_changed()
{
	stringstream ss;
	ss << gui_pixels_spinbutton->get_value();
	ss >> min_draw_radius;
	draw_image ( true );
}

bool Gravity::mouse_move_event ( GdkEventMotion* e )
{
	if ( mouse_down )
	{
		double dx = ( e->y - mouse_press_x ) / 2;
		double dy = ( e->x - mouse_press_y ) / 2;
		rotx = mouse_press_rx - dx;
		roty = mouse_press_ry - dy;
		draw_image ( true );
	}
	return true;
}

bool Gravity::mouse_press_event ( GdkEventButton* e )
{
	mouse_down = true;
	// set up to control rotation
	// by dragging mouse
	mouse_press_rx = rotx;
	mouse_press_ry = roty;
	mouse_press_x = e->y;
	mouse_press_y = e->x;
	draw_image ( true );
	return true;
}

bool Gravity::mouse_release_event ( GdkEventButton* e )
{
	mouse_down = false;
	return true;
}

bool Gravity::wheel_event ( GdkEventScroll* e )
{
	double v = ( e->direction == GDK_SCROLL_UP ) ?1:-1;
	v = 1.0 + ( v/10.0 );
	drawing_scale *= v;
	draw_image ( true );
	return true;
}

bool Gravity::expose_event ( GdkEventExpose* e )
{
	draw_image ( true );
	return true;
}
