#!/usr/bin/qtrubyinit

require 'regexp_tester_ui.rb'
require 'aboutdialog_ui.rb'

PROGNAME   = "RegExpTester"
PROGVER    = "0.1"
PROGTITLE  = "#{PROGNAME} #{PROGVER}"

=begin
   The Configuration class defines program values
   to be preserved in a configuration file.
=end

class Configuration
   attr_accessor :dataFile
   attr_accessor :dataPath
   attr_accessor :search
   attr_accessor :replace
   attr_accessor :global
   attr_accessor :case
   attr_accessor :extended
   attr_accessor :multiline
   def initialize
      @dataFile = ""
      @dataPath = ENV["HOME"]
      @search = ""
      @replace = ""
      @global = true
      @case = false
      @extended = false
      @multiline = false
   end
end

=begin
   The ConfigurationHandler class reads and writes
   the program configuration file and populates a
   Configuration class instance
=end

class ConfigurationHandler
   def initialize(conf,progName)
      @conf = conf
      @progName = progName
      @confPath = File.join(ENV["HOME"], "." + @progName)
      @iniPath = @confPath + "/" + @progName + ".ini"
      Dir.mkdir(@confPath) unless FileTest.exists?(@confPath)
   end
   def writeConfig
      file = File.new(@iniPath,"w")
      unless file.nil?
         @conf.instance_variables.sort.each { |x|
            xi = @conf.instance_variable_get(x)
            # escape strings
            if(xi.class == String)
               xi.gsub!(/\\/,"\\\\\\\\")
               xi.gsub!(/"/,"\\\"")
               xi = "\"#{xi}\""
            end
            file.write("#{x}=#{xi}\n")
         }
         file.close()
      end
   end
   def readConfig
      if FileTest.exists?(@iniPath)
         file = File.new(@iniPath,"r")
         file.each { |line|
            @conf.instance_eval(line)
         }
         file.close()
      end
   end
end

=begin
   AboutDialog subclasses AboutDialogUI
   (see below for more explanation)
=end

class AboutDialog < AboutDialogUI
   def initialize
      super
      setCaption("About #{PROGNAME}")
      @groupBox3.setTitle( "#{PROGTITLE}" )
      @textLabel2.setText("#{PROGNAME} is (c) Copyright 2006, P. Lutus. All rights reserved. This program is released under the GPL." )
   end
   def closeButton_clicked(*k)
      self.close()
   end
end

=begin
   -- description of files and classes --

   regexp_tester.ui:
      user interface created by qtdesigner
      or kdevdesigner (basically the same program)
   regexp_tester_ui.rb:
      created using "rbuic" out of regexp_tester.ui,
      defines RegExpTesterUI class,
      which implements a Qt user interface in Ruby
   regexp_tester.rb:
      defines RegexpTester and several other classes

   RegExpTester subclasses RegExpTesterUI
   and provides the code body for the user interface
=end

class RegExpTester < RegExpTesterUI
   def initialize(app)
      super()
      @app = app
      # handle configuration issues
      @conf = Configuration.new
      @confHandler = ConfigurationHandler.new(@conf,PROGNAME)
      # read configuration file
      @confHandler.readConfig()
      # set checkbox values from configuration
      @globalCheckBox.setChecked(@conf.global)
      @caseCheckBox.setChecked(@conf.case)
      @extendedCheckBox.setChecked(@conf.extended)
      @multilineCheckBox.setChecked(@conf.multiline)
      @textInput.setText(@conf.search)
      @regexSearch.setText(@conf.search)
      @regexReplace.setText(@conf.replace)
      @title = PROGTITLE
      setCaption(@title)
      # load any prior data file
      load(@conf.dataFile) if @conf.dataFile.size > 0
   end

   # "choose" selects a data file

   def choose()
      fd = Qt::FileDialog.new
      fn = fd.getOpenFileName(@conf.dataPath, nil, self)
      if !fn.nil?
         load( fn )
         # preserve user selected path
         @conf.dataFile = fn
         @conf.dataPath = File.dirname(fn)
         setCaption( @title + ": " + fn )
      else
         statusBar().message( tr("Loading aborted"), 5000 )
      end
   end

   # "load" reads a chosen data file

   def load( filename )
      f = Qt::File.new( filename )
      if !f.open( Qt::IO_ReadOnly )
         return
      end

      ts = Qt::TextStream.new( f )
      @textInput.setText( ts.read() )
      @textInput.setModified( false )
      statusBar().message( tr("Loaded document %s" % filename), 5000 )
   end

   # user action for "choose" (select a file)

   def fileReadAction_activated(*k)
      choose()
   end

   # user action for about dialog

   def fileAbout_activated(*k)
      ad = AboutDialog.new
      ad.show
   end

   # carry out regular expression processing

   def regexExecute(*k)
      # get data into local variables
      data = @textInput.text()
      search = @regexSearch.text()
      replace = @regexReplace.text()
      # convert newline and tab escapes
      # in replace string
      replace.gsub!(/\\n/,"\n")
      replace.gsub!(/\\t/,"\t")
      # process options
      options = 0
      options |= Regexp::IGNORECASE unless @caseCheckBox.isChecked()
      options |= Regexp::EXTENDED if @extendedCheckBox.isChecked()
      options |= Regexp::MULTILINE if @multilineCheckBox.isChecked()
      # create regular expression
      rs = Regexp.new(search,options)
      # perform search & replace
      if (@globalCheckBox.isChecked())
         data.gsub!(rs,replace)
      else
         data.sub!(rs,replace)
      end
      # assign result data
      @textOutput.setText(data)
   end

   # user actions for regular expression processing

   def regexReplace_returnPressed(*k)
      regexExecute()
   end

   def regexSearch_returnPressed(*k)
      regexExecute()
   end

   # override handling of "close" signal from both
   # menu item and main frame

   def close(x)
      # set configuration from checkbox values
      @conf.global = @globalCheckBox.isChecked()
      @conf.case = @caseCheckBox.isChecked()
      @conf.extended = @extendedCheckBox.isChecked()
      @conf.multiline = @multilineCheckBox.isChecked()
      @conf.search = @regexSearch.text()
      @conf.replace = @regexReplace.text()
      # write configuration file
      @confHandler.writeConfig()
      @app.exit(0)
   end

   # user action for "close"

   def exitAction_activated(*k)
      close(false)
   end

end

# launch Qt application

if $0 == __FILE__
   app = Qt::Application.new(ARGV)
   w = RegExpTester.new(app)
   app.mainWidget = w
   w.show
   app.exec
end
