#!/usr/bin/env python
# -*- coding: utf-8 -*-

# ***************************************************************************
# *   Copyright (C) 2011, Paul Lutus                                        *
# *                                                                         *
# *   This program is free software; you can redistribute it and/or modify  *
# *   it under the terms of the GNU General Public License as published by  *
# *   the Free Software Foundation; either version 2 of the License, or     *
# *   (at your option) any later version.                                   *
# *                                                                         *
# *   This program is distributed in the hope that it will be useful,       *
# *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
# *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
# *   GNU General Public License for more details.                          *
# *                                                                         *
# *   You should have received a copy of the GNU General Public License     *
# *   along with this program; if not, write to the                         *
# *   Free Software Foundation, Inc.,                                       *
# *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
# ***************************************************************************

# version date 02-14-2011

VERSION = '1.1'

import os, re, sys, signal
import gobject
gobject.threads_init()
import gst
import gtk
gtk.gdk.threads_init()
import pango
import serial
import platform
import random
import math
import struct
import time,datetime
import webbrowser

class Icon:
  icon = [
    "32 32 17 1",
    "   c None",
    ".  c #1A1706",
    "+  c #3A341D",
    "@  c #675D3B",
    "#  c #61625C",
    "$  c #82691C",
    "%  c #95844D",
    "&  c #9F9C90",
    "*  c #C7A740",
    "=  c #C5AC5D",
    "-  c #B5B7B3",
    ";  c #DEC05B",
    ">  c #D9C58E",
    ",  c #CED0CF",
    "'  c #E9EBE8",
    ")  c #F2F4F1",
    "!  c #FAFCF9",
    "                                ",
    "                                ",
    "    !))))))))))))))))))))))!    ",
    "    )''''''''''''''))))))))!    ",
    "    )''''''''''')))))))))))!    ",
    "    )''''''''')))))))))))))!    ",
    "    )''''''')))))))))))))))!    ",
    "    )''''''))))))))))))))))!    ",
    "    )'''')))))))-#)))))))))!    ",
    "    )''')))))))&+,))!!!!!!!!    ",
    "    )'')))))),&=&)!!!!!!!!!!    ",
    "    )'))))))'%=%,)!!!!!!!!!!    ",
    "    )))))))'*.$&')!!!!!!!!!!    ",
    "    ))))))'*.*%,)!!!!!!!!!!!    ",
    "    )))))'*+%%,'!!!!!!!!!!!!    ",
    "    !)))'=+%%-')!!!!!!!!!!!!    ",
    "    !)))=+%=-,)!!!!!!!!!!!!!    ",
    "    !))=@%*-'!!!!!!!!!!!!!!!    ",
    "    !)=@@*-,'!!!!!!!!!!!!!!!    ",
    "    !=$@*&,'!!!!!!!!!!!!!!!!    ",
    "    =$@;&,'!!!!!!!!!!!!!!))!    ",
    "   $$@;&,)!!!!!!!!!!!!)'''''    ",
    "   $+;&-'!!!!!!!!!!!))',,,,#    ",
    "  $+;&-')!!!!!!!!!!!',-,''#     ",
    " &#*--,)!!!!!!!!!!!)'-!!'#      ",
    "+.>+,')!!!!!!!!!!!)',,!'#       ",
    " +  '')!!!!!!!!!!!)'-,'#        ",
    "    ))!!!!!!!!!!!!),-'#         ",
    "    !)!!!!!!!!!!!!),,#          ",
    "    !!!!!!!!!!!!!!),#           ",
    "                                ",
    "                                "
  ]

# this should be a temporary hack

class WidgetFinder:
  def localize_widgets(self,parent,xmlfile):
    # an unbelievable hack made necessary by
    # someone unwilling to fix a year-old bug
    with open(xmlfile) as f:
      for name in re.findall(r'(?s) id="(.*?)"',f.read()):
        if re.search(r'^k_',name):
          obj = parent.builder.get_object(name)
          setattr(parent,name,obj)

class ConfigManager:
  def __init__(self,path,dic):
    self.path = path
    self.dic = dic

  def read_config(self):
    if os.path.exists(self.path):
      with open(self.path) as f:
        for record in f.readlines():
          se = re.search('(.*?)\s*=\s*(.*)',record.strip())
          if(se):
            key,value = se.groups()
            if (key in self.dic):
              widget = self.dic[key]
              typ = type(widget)
              if(typ == list):
                widget[0] = value
              elif(typ == gtk.Entry):
                widget.set_text(value)
              elif(typ == gtk.HScale):
                widget.set_value(float(value))
              elif(typ == gtk.Window):
                w,h = value.split(',')
                widget.resize(int(w),int(h))
              elif(typ == gtk.CheckButton):
                widget.set_active(value == 'True')
              elif(typ == gtk.ComboBox):
                if(value in widget.datalist):
                  i = widget.datalist.index(value)
                  widget.set_active(i)
              else:
                print "ERROR: reading, cannot identify key %s with type %s" % (key,type(widget))

  def write_config(self):
    with open(self.path,'w') as f:
      for key,widget in sorted(self.dic.iteritems()):
        typ = type(widget)
        if(typ == list):
          value = widget[0]
        elif(typ == gtk.Entry):
          value = widget.get_text()
        elif(typ == gtk.HScale):
          value = str(widget.get_value())
        elif(typ == gtk.Window):
          _,_,w,h = widget.get_allocation()
          value = "%d,%d" % (w,h)
        elif(typ == gtk.CheckButton):
          value = ('False','True')[widget.get_active()]
        elif(typ == gtk.ComboBox):
          value = widget.get_active_text()
        else:
          print "ERROR: writing, cannot identify key %s with type %s" % (key,type(widget))
          value = "Error"
        f.write("%s = %s\n" % (key,value))

  def preset_combobox(self,box,v):
    if(v in box.datalist):
      i = box.datalist.index(v)
      box.set_active(i)
    else:
      box.set_active(0)

  def load_combobox(self,obj,data):
    if(len(obj.get_cells()) == 0):
      # Create a text cell renderer
      cell = gtk.CellRendererText ()
      obj.pack_start(cell)
      obj.add_attribute (cell, "text", 0)
    obj.get_model().clear()
    for s in data:
      obj.append_text(s.strip())
    setattr(obj,'datalist',data)

class Goertzel:
  def __init__(self,frequency,samples = 16384, thresh = 0.5):
    self.frequency = frequency
    self.samples = samples
    # normalizing factor is 4/samples^2
    # which produces unity output for a
    # unity input sine at target frequency
    self.nf = 4.0 / (samples*samples)
    self.gf = 2 * math.cos(2 * math.pi * frequency)
    self.thresh = thresh
    self.reset(True)

  def process(self,v):
    s0 = v + self.gf * self.s1 - self.s2
    self.s2, self.s1 = self.s1, s0
    self.samplecount += 1
    if(self.samplecount >= self.samples):
      self.update()

  def update(self):
    self.val = (self.s2 * self.s2 + self.s1 * self.s1
      - self.gf * self.s1 * self.s2) * self.nf
    self.reset()

  def reset(self,full = False):
    self.s1 = self.s2 = self.samplecount = 0
    if(full):
      self.val = 0

  def set_threshold(self,v):
    self.thresh = v

  def value(self):
    return self.val

  def active(self):
    return (self.val > self.thresh)

# end class Goertzel

class DecodeFax:

  def __init__(self,parent):
    self.par = parent
    # Goertzel default tone acceptance threshold
    self.goertzel_accept = 0.5
    self.machine_states = [
      self.s_waitsig,
      self.s_waitstb,
      self.s_waitste,
      self.s_waitls1,
      self.s_sync,
      self.s_waitls2,
      self.s_proc,
      self.s_end
    ]
    self.pipeline = False
    self.sink = False
    self.monitor_volume = False
    self.gstart = False
    self.gend = False

  def set_threshold(self,v):
    self.goertzel_accept = v
    if(self.gstart):
      self.gstart.set_threshold(v)
      self.gend.set_threshold(v)

  def enable_filtering(self):
    # video low-pass time constants
    # for vertical and horizontal axes
    v = (1.0,0.75)[self.par.filtering]
    self.sig_htc = v
    self.sig_vtc = v

  def setup(self):
    self.unlink_source()
    self.filemode = False
    self.reading = False
    self.state = DecodeFax.S_WAITSIG
    # precompile struct unpack operator
    self.struct_int = struct.Struct('i')
    # sync time seconds, 20 in U.S.
    self.sync_time = 20
    self.sync_interval = 0.025 # seconds
    self.lines_per_second = 2
    self.sync_lines = self.sync_time * self.lines_per_second
    self.isamplerate  = int(self.par.samplerate)
    # time constants
    self.volume_tc = 0.001
    self.afc_tc = 0.0005
    self.enable_filtering()
    self.sample_increm = self.par.samplerate / (self.par.image_width * 2)
    self.row_len = self.isamplerate / self.lines_per_second
    self.sampleinterval = 1.0 / self.par.samplerate
    self.sampleinterval_d2 = 0.5 / self.par.samplerate
    self.agc_thresh = 256
    self.cf = 1900.0
    self.mark = 2300.0
    self.space = 1500.0
    self.deviation = 400.0
    self.startf = 300.0
    self.stopf = 450.0
    self.cf_lvl = self.par.samplerate / self.cf
    self.space_lvl = self.par.samplerate / self.space
    self.mark_lvl = self.par.samplerate / self.mark
    self.sig_scale = (self.mark_lvl / self.space_lvl) * 8.0
    # Goertzel sample size
    g_size = self.par.samplerate/4.0
    self.gstart = Goertzel(
      self.startf * self.sampleinterval,
      g_size,
      self.goertzel_accept
    )
    self.gend = Goertzel(
      self.stopf * self.sampleinterval,
      g_size,
      self.goertzel_accept
    )

  def unlink_source(self):
    if(self.pipeline):
      self.pipeline.set_state(gst.STATE_NULL)
      gst.element_unlink_many(*self.chain)
      self.pipeline.remove_many(*self.chain)
      for item in self.chain:
        item = False
      self.pipeline = False
      time.sleep(0.01)

  def make_and_chain(self,name):
    target = gst.element_factory_make(name)
    self.chain.append(target)
    return target

  def setup_input(self):
    self.unlink_source()
    caps = gst.Caps(
      "audio/x-raw-int,"
      "endianness=(int)1234,"
      "channels=(int)1,"
      "width=(int)32,"
      "depth=(int)32,"
      "signed=(boolean)true,"
      "rate=(int)%d" % self.isamplerate
    )
    self.chain = []
    self.pipeline = gst.Pipeline("mypipeline")
    source = self.make_and_chain("autoaudiosrc")
    q0 = self.make_and_chain("queue")
    if(self.par.monitor_mode):
      tee = self.make_and_chain("tee")
      q1 = self.make_and_chain("queue")
      self.monitor_volume = self.make_and_chain("volume")
      self.par.set_monitor_volume()
      monitor_sink = self.make_and_chain("autoaudiosink")
    self.sink = self.make_and_chain("appsink")
    self.sink.set_property('caps', caps)
    self.sink.set_property('sync', False)
    self.sink.set_property('drop', False)
    self.sink.connect('new-buffer', self.process_data)
    self.sink.set_property('emit_signals', True)
    self.pipeline.add(*self.chain)
    if(self.par.monitor_mode):
      gst.element_link_many(source,tee,q0,self.sink)
      gst.element_link_many(tee,q1,self.monitor_volume,monitor_sink)
    else:
      gst.element_link_many(source,q0,self.sink)
    self.pipeline.set_state(gst.STATE_PLAYING)

  def init_chart_read(self):
    if(self.reading and (self.state == DecodeFax.S_WAITSIG or self.state == DecodeFax.S_WAITSTB)):
      self.state = DecodeFax.S_WAITLS2
    else:
      self.line = ""
      self.linebuf = ""
      self.samplecount = 0
      self.dcount = 0
      self.old_dcount = 0
      self.time_sec = 0
      self.state = 0
      self.sync_array = False
      self.sync_line = False
      self.image_line = 0
      self.row_index = 0
      self.row_pos = 0
      self.row_ipos = 0
      self.line_time_delta = 0
      self.sig = 0
      self.gsig = 0
      self.os = 0
      self.meter_cycles = 0
      self.linesize = 0
      self.agc_level = 1
      self.afc_level = 0
      self.par.current_chart = False
      self.state = DecodeFax.S_WAITSIG
      self.setup_input()
      self.gstart.reset(True)
      self.gend.reset(True)
      self.linetime_zero = False
      self.reading = True

  def chart_read(self,read,reset = False):
    if(reset): self.reading = False
    if(read):
      if(not self.reading):
        self.init_chart_read()
    else:
      if(self.reading):
        self.reading = False
        self.par.current_chart = False
        self.state = DecodeFax.S_END
        self.unlink_source()

  def unlock(self,*args):
    if(self.reading):
      self.state = DecodeFax.S_END

  """

  state machine states:

  S_WAITSIG : wait for audio signal

  S_WAITSTB : wait for beginning of 300 Hz start tone (5s)

  S_WAITSTE : wait for ending of start tone

  S_WAITLS1 : wait for local clock line-start mark

  S_SYNC : sync local clock with incoming signal (20s)

  S_WAITLS2 : wait for local clock line-start mark

  S_PROC : process image lines, watch for end tone (5s)

  S_END : end image

  """

  # fake enumeration for states
  S_WAITSIG,S_WAITSTB,S_WAITSTE,S_WAITLS1,S_SYNC,S_WAITLS2,S_PROC,S_END = range(8)

  state_names = ['WAITSIG','WAITSTB','WAITSTE','WAITLS1','SYNC','WAITLS2','PROC','END']

  # wait for acceptable input signal
  def s_waitsig(self):
    if(self.agc_level >= self.agc_thresh):
      self.state = DecodeFax.S_WAITSTB
      return True
    return False

  # wait for beginning of start tone
  def s_waitstb(self):
    self.image_line = 0
    if(self.agc_level < self.agc_thresh):
      self.state = DecodeFax.S_WAITSIG
      return True
    else:
      if(self.gstart.active()):
        self.state = DecodeFax.S_WAITSTE
        return True
    return False

  # wait for end of start tone
  def s_waitste(self):
    if(not self.gstart.active()):
      self.line_time_delta = 0
      self.state = DecodeFax.S_WAITLS1
      return True
    else: return False

  # wait for line start 1
  def s_waitls1(self):
    if(self.linetime_zero): # wait for line start
      self.sync_array = False
      self.state = DecodeFax.S_SYNC
      return True
    else: return False

  # synchronize with sender
  def s_sync(self):
    if(not self.sync_array):
      self.image_line = 0
      self.row_index = 0
      # zero the averaging array
      self.sync_array = [0 for i in range(self.row_len)]
      # a copy to accumulate each line
      self.sync_line = self.sync_array[:]
    # must acculumate to work with
    # noisy signals and clock errors
    self.sync_line[self.row_index] += self.sig
    self.row_index += 1
    if(self.row_index >= self.row_len):
      self.row_index = 0
      # shift result array to correct for clock error
      self.sync_line = self.adjust_line_timing (
        self.sync_line,
        self.image_line,
        self.row_len * self.par.clock_adjust
      )
      # add shifted line to accumulation array
      for i,v in enumerate(self.sync_line):
        self.sync_array[i] += v
        self.sync_line[i] = 0
      if(self.image_line >= self.sync_lines):
        # integrate and locate positive excursion
        iv = self.sync_array[0]
        os = 0
        tc = 500.0 / self.isamplerate
        for i,v in enumerate(self.sync_array):
          iv += (self.sync_array[i]-iv) * tc
          s = (-1,1)[iv > 0]
          self.sync_line[i] = s-os
          os = s
        # position sync at max positive value + 12 ms
        self.line_time_delta = int(self.sync_line.index(max(self.sync_line))
          + (self.sync_interval * 0.5 * self.isamplerate))
        self.state = DecodeFax.S_WAITLS2
        return True
    return False

  # wait for line start 2
  def s_waitls2(self):
    if(self.linetime_zero):
      # create a chart to receive data
      gtk.gdk.threads_enter()
      temp = WeatherPanel(self.par)
      self.par.add_chart(temp)
      gtk.gdk.threads_leave()
      self.row_index = 0
      self.line = ""
      self.linebuf = ""
      self.int_line = [False for i in range(self.row_len)]
      self.state = DecodeFax.S_PROC
      return True
    else: return False

  # process image lines
  def s_proc(self):
    if(self.linetime_zero):
      self.row_index = 0
      self.row_pos = 0
      self.row_ipos = 0
      update_image = False
      # detect end tone
      end_tone = self.gend.active()
      self.linesize = len(self.line)
      if(self.linesize > 0):
        if(self.par.clock_adjust != 0.0):
          # shift image line to correct for clock error
          self.line = self.adjust_line_timing(
            self.line,
            self.image_line,
            self.par.image_width * self.par.clock_adjust,
            3
          )
        self.linebuf += self.line
        update_image = ((self.image_line+1) % 10 == 0)
      if(update_image or end_tone):
        gtk.gdk.threads_enter()
        self.par.current_chart.update_from_string(self.linebuf,self.linesize)
        gtk.gdk.threads_leave()
        self.linebuf = ""
      # act on end of chart or runaway chart
      if(end_tone or self.image_line > 4000):
        self.state = DecodeFax.S_END
        return True
      self.line = ""
    if(self.row_index >= self.row_ipos):
      self.row_pos += self.sample_increm
      self.row_ipos = int(self.row_pos)
      val = self.int_line[self.row_index]
      if(val):
        val += (self.sig - val) * self.sig_vtc
      else:
        val = self.sig
      self.int_line[self.row_index] = val
      vid = (val + 1.0) * 0.5
      vid = min(vid,1.0)
      vid = max(vid,0)
      if(self.par.grayscale):
        c = chr(int(vid * 255.0))
      else:
        c = chr((0,255)[vid > 0.5])
      self.line += c * 3
    self.row_index += 1
    return False

  def s_end(self):
    if(self.par.current_chart):
      gtk.gdk.threads_enter()
      self.par.current_chart.save()
      gtk.gdk.threads_leave()
    self.par.current_chart = False
    self.state = DecodeFax.S_WAITSIG
    return False

  def process_data(self,*src):
    if(not self.reading):
      return
    buff = self.sink.emit('pull-buffer')
    if(buff != None):
      data = [self.struct_int.unpack_from(buff,i)[0] for i in range(0,len(buff),4)]
      data_len = len(data)
      for i,ms in enumerate(data):
        self.n = self.samplecount + i
        self.linetime_zero = (((self.n - self.line_time_delta) % self.row_len) == 0)
        self.agc_level += (abs(ms) - self.agc_level) * self.volume_tc
        self.agc_level = max(self.agc_level,1)
        ms /= self.agc_level
        # detect positive zero crossings
        s = (ms > 0.0)
        if(s > self.os):
          self.meter_cycles += 1
          self.dcount, self.old_dcount = self.n - self.old_dcount, self.n
          self.dcount = max(self.dcount,.1)
          # need high-bandwidth signal for detecting start & stop tones
          self.gsig = ((self.cf_lvl/self.dcount) - 1.0) * self.sig_scale
          # automatic frequency control
          if(self.par.afc):
            self.afc_level += (self.gsig - self.afc_level) * self.afc_tc
            self.gsig -= self.afc_level
          # limit amplitude excursions
          self.gsig = max(-2.0,self.gsig)
          self.gsig = min(2.0,self.gsig)
          # create a low-pass version of the signal for nice video
          self.sig += (self.gsig - self.sig) * self.sig_htc
        self.os = s
        self.gstart.process(self.gsig)
        self.gend.process(self.gsig)
        # now execute machine states
        while self.machine_states[self.state](): pass
        if(self.linetime_zero):
          self.image_line += 1
          if(self.par.debug):
            self.time_sec = self.n * self.sampleinterval
            gtk.gdk.threads_enter()
            print "%f %s dcount %f sig %f agclvl %e start %e stop %e vol tc %f" % \
            (self.time_sec,DecodeFax.state_names[self.state],self.dcount, self.sig, self.agc_level,
            self.gstart.value(),self.gend.value(),self.volume_tc)
            gtk.gdk.threads_leave()
      self.samplecount += data_len

  def adjust_line_timing(self,line,line_count,delta,mult = 1):
    x = int(line_count * delta)
    if(x != 0):
      ll = len(line)
      # avoid wrap artifact
      w = (ll,-ll)[x < 0]
      x %= w
      # varying line lengths
      x *= mult
      line = line[x:] + line[:x]
    return line

class HelpPanel(gtk.Frame):
  def __init__(self,parent):
    gtk.Frame.__init__(self)
    self.par = parent
    self.old_entry = ''
    self.data = ''
    self.old_pos = 0
    self.xmlfile = 'helppanel_gui.glade'
    self.builder = gtk.Builder()
    self.builder.add_from_file(self.xmlfile)
    WidgetFinder().localize_widgets(self,self.xmlfile)
    self.k_helppanel.show()
    self.k_textview.modify_bg(gtk.STATE_NORMAL,self.par.white)
    self.k_textview.modify_font(self.par.mono_font)
    self.k_helppanel.connect('destroy',self.close)
    self.k_close_button.connect('clicked',self.close)
    #self.k_helppanel.connect('expose-event',self.focus_gained)
    self.k_search_entry.connect('key-release-event',self.search)
    self.k_search_entry.set_tooltip_text('Instant search:\n  enter a search string,\n  press Enter for next case')
    self.add(self.k_helppanel)
    self.filepath = self.par.program_name + "_help.txt"
    # must specify unicode string
    data = u''
    with open(self.filepath) as f:
      data += f.read()
    data = re.sub('<NAME>',self.par.program_name,data)
    data = re.sub('<NAMELC>',self.par.program_name.lower(),data)
    data = re.sub('<VERSION>',VERSION,data)
    self.tbuf = self.k_textview.get_buffer()
    self.tbuf.set_text(data)
    self.data = data.lower()
    self.show()
    self.k_search_entry.set_text(self.par.search_entry[0])
    gobject.timeout_add(250,lambda: self.k_search_entry.grab_focus())
    gobject.timeout_add(500,lambda: self.search())

  def search(self,*args):
    entry = self.k_search_entry.get_text().lower()
    fail = True
    iend = False
    pos = 0
    ll = len(entry)
    if(ll > 0):
      if(entry != self.old_entry):
        # find first case
        pos = self.data.find(entry)
      else:
        # find next case
        pos = self.data.find(entry,self.old_pos+1)
      # if found
      if(pos >= 0):
        fail = False
        ipos = self.tbuf.get_iter_at_offset(pos)
        iend = self.tbuf.get_iter_at_offset(pos+ll)
    if(fail):
      if(self.old_pos > 0):
        self.old_pos = 0
        self.old_entry = ''
        self.search()
        return
      else:
        ipos = self.tbuf.get_iter_at_offset(0)
        iend = self.tbuf.get_iter_at_offset(0)
    if(iend):
      self.k_textview.scroll_to_iter(iend,0.0,True,0.5,0.5)
      self.tbuf.select_range(ipos,iend)
    self.old_pos = pos
    self.old_entry = entry

  def close(self,*args):
    self.par.search_entry[0] = self.k_search_entry.get_text()
    self.par.help_pane = False
    self.par.help_index = -1
    self.destroy()

class WeatherPanel(gtk.Frame):
  def __init__(self,parent):
    gtk.Frame.__init__(self)
    self.par = parent
    self.imagebuf = ""
    self.undo_stack = []
    self.height = 0
    self.width = 0
    self.ratio = 1.0
    self.old_width = -1
    self.label = False
    self.changed = False
    self.savepath = False
    self.pixbuf = False
    self.adjust_mode = False
    self.xmlfile = 'weatherpanel_gui.glade'
    self.builder = gtk.Builder()
    self.builder.add_from_file(self.xmlfile)
    WidgetFinder().localize_widgets(self,self.xmlfile)
    self.k_weatherpanel.show()
    self.add(self.k_weatherpanel)
    self.k_undo_button.set_tooltip_text('Undo a recent image editing action')
    self.connect('destroy', self.save)
    self.k_save_button.connect('clicked', self.save)
    self.k_close_button.connect('clicked', self.close)
    self.k_invert_button.connect('clicked', self.invert_image)
    self.k_rot_cw_button.connect('clicked', lambda w: self.rot(gtk.gdk.PIXBUF_ROTATE_CLOCKWISE))
    self.k_rot_ccw_button.connect('clicked', lambda w: self.rot(gtk.gdk.PIXBUF_ROTATE_COUNTERCLOCKWISE))
    self.k_undo_button.connect('clicked', self.undo_pop)
    self.k_eventbox.connect('motion-notify-event', self.track_mouse_position)
    self.k_eventbox.connect('button-press-event', self.mouse_button_press)
    self.k_weatherpanel.connect('expose-event',self.focus_gained)
    self.par.k_mainwindow.connect('check-resize',self.check_resize)
    self.update_undo()
    self.show()
    self.focus_gained()

  def active(self,message = False):
    if(self.par.current_chart == self):
      if(message):
        self.k_advise_label.set_text(" Error: Cannot adjust active chart.")
      return True
    else:
      return False

  def undo_push(self):
    self.undo_stack.append(self.pixbuf)
    while(len(self.undo_stack) > 16):
      self.undo_stack.pop(0)
    self.update_undo()

  def undo_pop(self,*args):
    if(len(self.undo_stack) > 0):
      self.pixbuf = self.undo_stack.pop()
      self.redim_pixbuf()
    self.update_undo()

  def update_undo(self):
    self.k_undo_button.set_sensitive(len(self.undo_stack) > 0)

  def rot(self,d):
    if(not self.active(True)):
      self.undo_push()
      if(self.pixbuf):
        self.pixbuf = self.pixbuf.rotate_simple(d)
        self.redim_pixbuf()

  def redim_pixbuf(self):
        self.width = self.pixbuf.get_rowstride()
        self.height = self.pixbuf.get_height()
        self.reload_pixbuf()

  def reload_pixbuf(self,*args):

    self.set_changed(True)
    self.old_width = -1
    self.ratio = 1.0
    if(self.par.full_scale):
      self.k_image.set_from_pixbuf(self.pixbuf)
    else:
      self.check_resize()

  def check_resize(self,*args):
    if((not self.par.full_scale) and self.pixbuf):
      _,_,sw,_ = self.k_scrolledwindow.get_allocation()
      vsb   = self.k_scrolledwindow.get_vscrollbar()
      if(vsb != None):
        _,_,scw,_ = vsb.get_allocation()
        ww = sw - (scw+4)
        if(ww > 5):
          w = self.pixbuf.get_width()
          h = self.pixbuf.get_height()
          r = float(ww) / w
          self.scaled_w = int(w * r)
          self.scaled_h = int(h * r)
          if(self.old_width != self.scaled_w and self.scaled_h > 0):
            self.old_width = self.scaled_w
            scaled = self.pixbuf.scale_simple(self.scaled_w,self.scaled_h,gtk.gdk.INTERP_BILINEAR)
            self.ratio = float(w) / scaled.get_width()
            self.k_image.set_from_pixbuf(scaled)

  def clock_correct_image(self,cf):
    if(not self.active(True) and self.pixbuf):
      self.undo_push()
      array = self.pixbuf_to_array(self.pixbuf)
      # in rotating lines, must take the difference
      # between width and rowstride into account
      w = self.pixbuf.get_width() * 3
      rs = self.pixbuf.get_rowstride()
      extra = chr(0) * (rs-w)
      res = ""
      for y,line in enumerate(array):
        x = int(y * cf)
        if(x != 0):
          # anticipate a full-wrap possibility
          sw = (w,-w)[x < 0]
          x %= sw
          # three bytes per pixel
          x *= 3
          line = line[x:] + line[:x]
        res += line + extra
      self.imagebuf = res
      self.update_image()

  def focus_gained(self,*args):
    if(self.savepath):
      self.label.set_tooltip_text(self.savepath)
      self.par.k_status_label.set_tooltip_text(self.savepath)
    else:
      self.par.k_status_label.set_tooltip_text("New chart")
    self.par.displayed_chart = self

  # this function intentionally removes the extra byte(s)
  # that make width and rowstride different
  # this difference must be made up
  # as the array is used to construct a pixbuf
  def pixbuf_to_array(self,pb):
    array = []
    if(pb):
      pixels = pb.get_pixels()
      w = self.pixbuf.get_width() * 3
      rs = self.pixbuf.get_rowstride()
      for h in range(self.height):
        p = h * rs
        array.append(pixels[p:p+w])
      return array

  def mouse_button_press(self,widget,evt):
    if(not self.active(True) and self.pixbuf):
      x,y = self.get_position(widget,evt)
      if(evt.button == 3):
        self.par.current_image = self
        self.par.calibrate(x,y)
      elif(x >= 0 and x < self.width and y >= 0 and y < self.height):
        self.par.cancel_calibrate()
        self.undo_push()
        array = self.pixbuf_to_array(self.pixbuf)
        # in rotating lines, must take the difference
        # between width and rowstride into account
        w = self.pixbuf.get_width() * 3
        rs = self.pixbuf.get_rowstride()
        extra = chr(0) * (rs-w)
        res = ""
        wx = x * 3
        for line in array:
          line = line[wx:] + line[:wx]
          res += line + extra
        self.imagebuf = res
        self.update_image()

  def track_mouse_position(self,w,evt):
    if(self.par.current_chart != self and self.pixbuf):
      x,y = self.get_position(w,evt)

  def get_position(self,w,evt):
    xp = yp = 0
    try:
      w1 = w.get_allocation()[2]
      pb = self.k_image.get_pixbuf()
      w2 = pb.get_width()
      delta = (w1-w2) / 2
      xp = int(evt.x - delta)
      xp = int(xp * self.ratio)
      yp = int(evt.y * self.ratio)
      pos = "Mouse {%d,%d}" % (xp,yp)
      self.k_advise_label.set_text(pos)
    except:
      pass
    return (xp,yp)

  def close(self,*args):
    self.par.close_window(self)

  def set_savepath(self,path):
    self.savepath = path
    self.k_advise_label.set_tooltip_text(self.savepath)
    if(self.label):
      self.label.set_tooltip_text(self.savepath)

  def set_changed(self,state):
    self.changed = state
    self.k_save_button.set_sensitive(state)
    self.update_controls()

  def update_controls(self):
    active = self.active()
    self.k_rot_cw_button.set_sensitive(not active)
    self.k_rot_ccw_button.set_sensitive(not active)
    self.k_invert_button.set_sensitive(not active)
    self.k_close_button.set_sensitive(not active)
    if(self.par.displayed_chart == self):
      tt = (self.par.default_tag,"Receiving chart, no actions available.")[active]
      if(tt != self.par.k_status_label.get_text() and self.par.cal_mode == 0):
        self.par.show_status_tag(tt)

  def save(self,*args):
    if(len(self.imagebuf) > 0 and self.pixbuf and self.changed):
      if(not self.savepath):
        ts = str(datetime.datetime.now())
        ts = re.sub('(\.\d)(\d+)','\\1',ts)
        self.set_savepath(os.path.join(self.par.configpath, re.sub(' ','_',"chart " + ts + ".jpg")))
      self.pixbuf.save(self.savepath,"jpeg")
      self.set_changed(False)

  def invert_image(self,*args):
    if(not self.active(True)):
      self.imagebuf = self.pixbuf.get_pixels().translate(self.par.inversion_string)
      self.update_image()

  def update_from_string(self,line,w):
    if(len(line) > 0):
      self.imagebuf += line
      self.width = int(w)
      self.height = len(self.imagebuf) / self.width
      self.update_image()
    if(self.par.scroll):
      self.delayed_scroll()

  def delayed_scroll(self):
    gobject.timeout_add(500, lambda: self.scroll_to_bottom(self.k_scrolledwindow))

  def scroll_to_bottom(self,sw):
    va = sw.get_vadjustment()
    delta = va.get_property('upper') - va.get_property('page_size')
    va.set_value(delta)

  def update_image(self,*args):
    self.pixbuf = gtk.gdk.pixbuf_new_from_data(
      self.imagebuf,
      gtk.gdk.COLORSPACE_RGB,
      False,
      8,
      self.width / 3,
      self.height,
      self.width
    )
    self.reload_pixbuf()

  def load_file(self,path):
    self.set_savepath(path)
    try:
      self.pixbuf = gtk.gdk.pixbuf_new_from_file(path)
      imwidth = self.pixbuf.get_width()
      self.width = self.pixbuf.get_rowstride()
      self.height = self.pixbuf.get_height()
    except:
      return False
    self.reload_pixbuf()
    self.set_changed(False)
    return True

class WeatherReader:
  # pretend enumeration
  S_STANDBY,S_RECEIVE = range(2)
  # sample rates must be even numbers
  sample_rates = ('100000','88200','44100','22050','11024','8500','6000')
  def __getitem__(self, key):
    return self.builder.get_object(key)
  def __init__(self):
    self.debug = False
    # exit correctly on keyboard and other signals
    signal.signal(signal.SIGTERM, self.close)
    signal.signal(signal.SIGINT, self.close)
    self.program_name = self.__class__.__name__
    self.program_title = self.program_name + ' ' + VERSION
    self.configpath = os.path.expanduser("~/." + self.program_name)
    self.first_run = (not os.path.exists(self.configpath))
    if(self.first_run):
      os.mkdir(self.configpath)
    self.config_file = os.path.join(self.configpath, self.program_name + ".ini")
    self.fax_decoder = DecodeFax(self)
    self.status_color = gtk.gdk.color_parse("#80e0ff")
    self.white = gtk.gdk.color_parse("white")
    self.gray = gtk.gdk.color_parse("#e0e0e0")
    self.black = gtk.gdk.color_parse("black")
    self.red = gtk.gdk.color_parse("red")
    self.yellow = gtk.gdk.color_parse("yellow")
    self.green = gtk.gdk.color_parse("green")
    self.magenta = gtk.gdk.color_parse("magenta")
    self.darkgreen = gtk.gdk.color_parse("#004000")
    self.orange = gtk.gdk.color_parse("#ffe080")
    self.default_rate = '22050'
    self.default_thresh = '50%'
    self.default_volume = '0%'
    self.search_entry = ['']
    # overall width of 1728 pixels seems to be standard
    # with 1641.6 image pixels + 86.4 sync pixels
    self.image_width = 1728
    self.max_displayed_charts = 16
    self.samplerate = 0.0
    self.timer_interval = 500 # milliseconds
    self.help_pane = False
    self.help_index  = -1
    self.displayed_chart = False
    self.grayscale = False
    self.filering = False
    self.full_scale = False
    self.scroll = True
    self.afc = False
    self.monitor_mode = False
    self.tab_num = 1
    self.volume = 0.0
    self.cal_mode = 0
    self.old_audio_color = False
    self.clock_adjust = 0
    self.page = 0
    self.inversion_string = ''.join([chr(c) for c in range(255,-1,-1)])
    self.operating_system = platform.system()
    self.xmlfile = 'weatherreader_gui.glade'
    self.builder = gtk.Builder()
    self.builder.add_from_file(self.xmlfile)
    WidgetFinder().localize_widgets(self,self.xmlfile)
    self.k_mainwindow.set_title(self.program_title)
    self.default_tag = "Actions: left-click: sync bar alignment, right-click: clock calibration."
    self.tooltips = {
      self.k_audio_level_label : 'Relative Audio Level',
      self.k_sig_label : 'Frequency:\n  Left: 1500 Hz\n  Center: 1900 Hz\n  Right: 2300 Hz',
      self.k_monitor_checkbutton : 'Enable input monitoring',
      self.k_monitor_volume_combobox : 'Monitor relative volume',
      self.k_cal_button : 'Start calibration procedure',
      self.k_cal_entry : 'Present calibration value',
      self.k_receive_button : 'Enable receive mode',
      self.k_standby_button : 'Disable receive mode',
      self.k_lock_button : 'Force receiver lock without synchronization',
      self.k_unlock_button : 'Unlock receiver, abandon chart',
      self.k_load_file_button : 'Load a previously received chart file',
      self.k_defaults_button : 'Set all default values',
      self.k_help_button : 'Read %s help' % self.program_title,
      self.k_full_scale_checkbutton : 'Show images at full scale',
      self.k_scroll_checkbutton : 'Scroll to image bottom as data is received',
      self.k_afc_checkbutton : 'Enable Automatic Frequency Control mode\n(more careful tuning is preferred)',
      self.k_grayscale_checkbutton : 'Grayscale mode for satellite images and pictures',
      self.k_filtering_checkbutton : 'Enable image noise filtering',
      self.k_thresh_combobox : 'Set start/stop tone sensitivity\n(if you don\'t know what this is, set it to 50%)',
      self.k_rate_combobox : 'Set data rate (only during standby)',
      self.k_quit_button : 'Exit %s' % self.program_title,
      self.k_website_button : 'Visit the %s home page' % self.program_title,
    }
    for key,value in self.tooltips.iteritems():
      self.set_tooltip(key,value)
    self.config_data = {
      'cal' : self.k_cal_entry,
      'appwindow' : self.k_mainwindow,
      'grayscale' : self.k_grayscale_checkbutton,
      'filtering' : self.k_filtering_checkbutton,
      'AFC' : self.k_afc_checkbutton,
      'rate' : self.k_rate_combobox,
      'threshold' : self.k_thresh_combobox,
      'full_scale' : self.k_full_scale_checkbutton,
      'scroll_to_bottom' : self.k_scroll_checkbutton,
      'monitor_mode' : self.k_monitor_checkbutton,
      'volume' : self.k_monitor_volume_combobox,
      'search_entry' : self.search_entry
    }
    self.cm = ConfigManager(self.config_file,self.config_data)
    self.cm.load_combobox(self.k_rate_combobox,WeatherReader.sample_rates)
    self.cm.preset_combobox(self.k_rate_combobox,self.default_rate)
    self.cm.load_combobox(self.k_thresh_combobox,["%d%%" % x for x in range(200,0,-1)])
    self.cm.preset_combobox(self.k_thresh_combobox,self.default_thresh)
    self.cm.load_combobox(self.k_monitor_volume_combobox,["%d%%" % x for x in range(500,-1,-1)])
    self.cm.preset_combobox(self.k_monitor_volume_combobox,self.default_volume)
    self.mono_font = pango.FontDescription("Monospace")
    self.k_sig_label.modify_fg(gtk.STATE_NORMAL,self.magenta)
    self.k_sig_label.modify_font(self.mono_font)
    self.k_sig_viewport.modify_bg(gtk.STATE_NORMAL,self.black)
    self.k_audio_viewport.modify_bg(gtk.STATE_NORMAL,self.black)
    self.k_audio_level_label.modify_fg(gtk.STATE_NORMAL,self.green)
    self.k_audio_level_label.modify_font(self.mono_font)
    self.state = WeatherReader.S_RECEIVE
    self.k_state_viewport.modify_bg(gtk.STATE_NORMAL,self.black)
    self.k_state_label.modify_fg(gtk.STATE_NORMAL,self.green)
    self.k_status_viewport.modify_bg(gtk.STATE_NORMAL,None)
    self.k_notebook = gtk.Notebook()
    self.k_notebook_viewport.add(self.k_notebook)
    self.k_notebook.set_scrollable(True)
    self.k_notebook.show()
    self.current_chart = False
    self.k_mainwindow.set_icon(gtk.gdk.pixbuf_new_from_xpm_data(Icon.icon))
    self.k_mainwindow.connect('destroy', self.close)
    self.k_mainwindow.connect('delete-event', self.check_active)
    self.k_cal_button.connect('clicked', self.calibrate)
    self.k_quit_button.connect('clicked', self.close)
    self.k_website_button.connect('clicked', self.visit_website)
    self.k_receive_button.connect('clicked', lambda w: self.set_mode(True))
    self.k_standby_button.connect('clicked', lambda w: self.set_mode(False))
    self.k_lock_button.connect('clicked', lambda w: self.fax_decoder.init_chart_read())
    self.k_unlock_button.connect('clicked', self.fax_decoder.unlock)
    self.k_load_file_button.connect('clicked', self.load_file_dialog)
    self.k_help_button.connect('clicked', self.launch_help)
    self.k_notebook.connect('switch-page', self.changed_page)
    self.k_cal_entry.connect('key-release-event',self.cal_keyboard)
    self.k_grayscale_checkbutton.connect('toggled',self.set_grayscale)
    self.k_afc_checkbutton.connect('toggled',self.set_afc)
    self.k_full_scale_checkbutton.connect('toggled',self.set_full_scale)
    self.k_scroll_checkbutton.connect('toggled',self.set_scroll)
    self.k_filtering_checkbutton.connect('toggled',self.set_filtering)
    self.k_rate_combobox.connect('changed', self.set_rate)
    self.k_monitor_checkbutton.connect('toggled', self.set_monitor_mode)
    self.k_monitor_volume_combobox.connect('changed', self.set_monitor_volume)
    self.k_thresh_combobox.connect('changed', self.set_threshold)
    self.k_defaults_button.connect('clicked', self.set_defaults)
    self.chartlist = []
    self.cm.read_config()
    self.retrieve_settings()
    self.fax_decoder.setup()
    self.set_mode(True) # default receive mode
    if(self.first_run):
      self.launch_help()
    # periodic GUI refresh
    gobject.timeout_add(self.timer_interval,self.process)

  def process(self):
    self.update_state()
    self.process_freq_meter()
    self.set_audio_level()
    return True

  def process_freq_meter(self):
    s = ''
    cycles = 0
    if(self.fax_decoder.agc_level > self.fax_decoder.agc_thresh):
      cycles = self.fax_decoder.meter_cycles
      self.fax_decoder.meter_cycles = 0
      cycles *= 1000.0 / self.timer_interval
      x = int(1 + ((cycles-1475)/60))
      x = max(0,x)
      x = min(15,x)
      s = (' ' * x) + '|'
    self.k_sig_label.set_label(s)
    self.k_freq_label.set_label("Freq: %04.0f Hz" % cycles)

  def set_audio_level(self):
    v = math.sqrt(self.fax_decoder.agc_level)
    lv = int(v / 1024)
    lv = min(lv,16)
    s = '|' * lv
    self.k_audio_level_label.set_text(s)
    label = "Audio "
    suff = "OK"
    col = self.darkgreen
    if(v > 16384):
      col = self.red
      suff = "High"
    if(v < 100): suff = "Low"
    self.k_audio_label.set_text(label + suff)
    if(col != self.old_audio_color):
      self.k_audio_label.modify_fg(gtk.STATE_NORMAL,col)
      self.old_audio_color = col

  def check_active(self,*args):
    active = (self.current_chart != False)
    if(active):
      self.tell_user("Cannot close while receiving chart --\n"
        + "press \"Unlock\", then close.")
    return active

  def set_defaults(self,*args):
    if(self.ask_user("Reset all user settings (except calibration) to defaults?")):
      self.cm.preset_combobox(self.k_rate_combobox,self.default_rate)
      self.cm.preset_combobox(self.k_thresh_combobox,self.default_thresh)
      self.cm.preset_combobox(self.k_monitor_volume_combobox,self.default_volume)
      self.k_grayscale_checkbutton.set_active(True)
      self.k_full_scale_checkbutton.set_active(False)
      self.k_afc_checkbutton.set_active(False)
      self.k_scroll_checkbutton.set_active(True)
      self.k_monitor_checkbutton.set_active(False)
      self.k_filtering_checkbutton.set_active(False)
      self.retrieve_settings()

  def retrieve_settings(self):
    self.set_rate()
    self.set_afc()
    self.set_grayscale()
    self.set_threshold()
    self.set_full_scale()
    self.set_current_cal()
    self.set_scroll()
    self.set_monitor_mode()
    self.set_monitor_volume()
    self.set_filtering()

  def show_status_tag(self,tag,color = None):
    if(color == None):
      color = self.status_color
    self.k_status_label.set_text(tag)
    self.k_status_viewport.modify_bg(gtk.STATE_NORMAL,color)

  def update_state(self):
    receive = (self.mode == WeatherReader.S_RECEIVE)
    tag = ('Standby','Receive')[receive]
    col = (self.green,self.yellow)[receive]
    if(receive):
      lock = (self.current_chart != False)
      stag = ('WAIT','LOCK')[lock]
      tag += "|" + stag
      col = (self.yellow,self.red)[lock]
    else:
      lock = False
    self.k_color_label.set_text(tag)
    self.k_color_viewport.modify_bg(gtk.STATE_NORMAL,col)
    self.k_receive_button.set_sensitive(not receive and not lock)
    self.k_standby_button.set_sensitive(receive and not lock)
    self.k_lock_button.set_sensitive(receive and (not lock))
    self.k_unlock_button.set_sensitive(receive and lock)
    self.k_quit_button.set_sensitive(not lock)
    self.k_cal_entry.set_sensitive(self.cal_mode != 0)
    self.k_rate_combobox.set_sensitive(not receive)
    self.k_defaults_button.set_sensitive(not receive)
    self.k_monitor_checkbutton.set_sensitive(not receive)
    self.k_monitor_volume_label.set_sensitive(self.monitor_mode)
    self.k_monitor_volume_combobox.set_sensitive(self.monitor_mode)
    s = ""
    if(self.fax_decoder.reading):
      s = "%s" % (DecodeFax.state_names[self.fax_decoder.state])
    self.k_state_label.set_text(s)
    for chart in self.chartlist:
      chart.update_controls()
    if(lock and self.current_chart == self.displayed_chart):
      self.current_chart.update_controls()
    return True

  def set_scroll(self,*args):
    self.scroll = self.k_scroll_checkbutton.get_active()
    if(self.scroll and self.displayed_chart):
      self.displayed_chart.delayed_scroll()

  def set_filtering(self,*args):
    self.filtering = self.k_filtering_checkbutton.get_active()
    self.fax_decoder.enable_filtering()

  def set_monitor_mode(self,*args):
    self.monitor_mode = self.k_monitor_checkbutton.get_active()

  def set_monitor_volume(self,*args):
    s = self.k_monitor_volume_combobox.get_active_text()
    self.volume = float(s.replace('%','')) / 100.0
    if(self.fax_decoder.monitor_volume):
      self.fax_decoder.monitor_volume.set_property('volume',self.volume)

  def set_rate(self,*args):
    s = self.k_rate_combobox.get_active_text()
    try:
      self.samplerate = float(s)
      self.fax_decoder.setup()
    except:
      pass

  def set_threshold(self,*args):
    s = self.k_thresh_combobox.get_active_text()
    v = float(s.replace('%',''))
    self.fax_decoder.set_threshold(v/100.0)

  def set_grayscale(self,*args):
    self.grayscale = self.k_grayscale_checkbutton.get_active()

  def set_full_scale(self,*args):
    self.full_scale = self.k_full_scale_checkbutton.get_active()
    if(self.displayed_chart):
      self.displayed_chart.reload_pixbuf()

  def set_afc(self,*args):
    self.afc = self.k_afc_checkbutton.get_active()

  def launch_help(self,*args):
    if(not self.help_pane):
      self.help_pane = HelpPanel(self)
      self.help_pane.label = gtk.Label("Help")
      self.k_notebook.append_page(self.help_pane, self.help_pane.label)
      self.help_index = self.k_notebook.get_n_pages() - 1
    self.k_notebook.set_current_page(self.help_index)

  def set_tooltip(self,widget,tip):
    if(widget): widget.set_tooltip_text(tip)

  def get_current_cal(self):
    try:
      s = self.k_cal_entry.get_text()
      v = float(s)
    except:
      v = 0
    return v

  def set_current_cal(self,v = None):
    if(v == None):
      v = self.get_current_cal()
    s = "%.4e" % (v)
    self.k_cal_entry.set_text(s)
    self.clock_adjust = v

  def cal_keyboard(self,w,evt):
    c = evt.keyval
    cn = gtk.gdk.keyval_name(c)
    if(cn == "Return"):
      self.cal_update(self.get_current_cal(),True)

  def calibrate(self,*args):
    self.cal_mode += 1
    if(self.cal_mode == 1):
      if(len(args) > 1):
        # mouse input
        self.cal_mode += 1
      else:
        # calibrate button press
        self.k_cal_button.set_label("Cancel")
        self.show_status_tag("Clock calibrate mode.",self.green)
        self.set_tooltip(self.displayed_chart,
        "Clock calibrate mode: right-click one extreme of a\n"
        "vertical feature, or enter a number below, or press \"Cancel\".")
        return
    if(self.cal_mode == 2):
      if(len(args) < 2):
        self.cancel_calibrate()
      else:
        x,y = args
        self.calsave = (x,y)
        self.k_cal_button.set_label("Cancel")
        self.show_status_tag("Clock calibrate mode.",self.orange)
        self.set_tooltip(self.displayed_chart,
        "Clock calibrate mode: now right-click the other extreme of a\n"
        "vertical feature, or enter a number below, or press \"Cancel\".")
    elif(self.cal_mode == 3):
      if(len(args) < 2):
        self.cancel_calibrate()
      else:
        x,y = args
        dx = float(x - self.calsave[0])
        dy = float(y - self.calsave[1])
        v  = dx / (dy * self.image_width)
        self.cal_update(v,False)
        self.update_state()
    else:
      self.cancel_calibrate()

  def cancel_calibrate(self):
    self.cal_update(None,False)

  def cal_update(self,v,keyboard):
    self.cal_mode = 0
    self.show_status_tag(self.default_tag)
    self.k_cal_button.set_label("Calibrate")
    self.set_tooltip(self.displayed_chart,"")
    if(keyboard):
      self.set_current_cal(v)
    else:
      valid = (v != None)
      if(valid and self.ask_user("Preserve this calibration value\nfor future chart reception?")):
        ov = self.get_current_cal()
        nv = v
        if(ov != 0 and self.ask_user("Add new value to old or replace old value\nYes = add, No = replace?")):
          nv += ov
        self.set_current_cal(nv)
      if(self.displayed_chart and (self.current_chart != self.displayed_chart)):
        if(valid and self.ask_user("Apply this calibration\nto the current image?")):
          self.displayed_chart.clock_correct_image(v * self.image_width)

  def ask_user(self,s):
    md = gtk.MessageDialog(
      self.k_mainwindow,
      0,
      gtk.MESSAGE_WARNING,
      gtk.BUTTONS_YES_NO,
      s
    )
    reply = md.run()
    md.destroy()
    return reply == gtk.RESPONSE_YES

  def tell_user(self,s):
    md = gtk.MessageDialog(
      self.k_mainwindow,
      0,
      gtk.MESSAGE_INFO,
      gtk.BUTTONS_OK,
      s
    )
    reply = md.run()
    md.destroy()
    return reply == gtk.RESPONSE_YES

  def load_file_dialog(self,*args):
    chooser = gtk.FileChooserDialog(title="Open Chart File",action=gtk.FILE_CHOOSER_ACTION_OPEN, \
    buttons=(gtk.STOCK_CANCEL,gtk.RESPONSE_CANCEL,gtk.STOCK_OPEN,gtk.RESPONSE_OK))
    chooser.set_current_folder(self.configpath)
    chooser.set_select_multiple(True)
    resp = chooser.run()
    if(resp == gtk.RESPONSE_OK):
      filenames = chooser.get_filenames()
      for path in filenames:
        if not self.already_loaded(path):
          self.load_chart(path)
    chooser.destroy()

  def already_loaded(self,path):
    for item in self.chartlist:
      if(item.savepath and item.savepath == path): return True
    return False

  def load_chart(self,path):
    temp = WeatherPanel(self)
    if (temp.load_file(path)):
      self.add_chart(temp,path)

  def close_window(self,w):
    if (w in self.chartlist):
      self.chartlist.remove(w)
      if(self.displayed_chart == w):
        self.displayed_chart = False
      w.destroy()

  def set_mode(self,mode):
    self.mode = (WeatherReader.S_STANDBY,WeatherReader.S_RECEIVE)[mode]
    self.fax_decoder.chart_read(mode,True)

  def increment_page(self):
    self.page = ((self.page + 1) % self.k_notebook.get_n_pages())
    self.k_notebook.set_current_page(self.page)

  def changed_page(self,w,x,n):
    self.page = n

  def add_chart(self,chart,path = False):
    if(not path):
      self.current_chart = chart
      path = "New chart"
    self.chartlist.append(chart)
    chart.label = gtk.Label("Chart %d" % self.tab_num)
    chart.label.set_tooltip_text(path)
    self.tab_num += 1
    self.k_notebook.append_page(chart, chart.label)
    self.page = self.k_notebook.get_n_pages() - 1
    self.k_notebook.set_current_page(self.page)
    # avoid too many open tabs
    while(len(self.chartlist) > self.max_displayed_charts):
      self.chartlist[0].close()

  def visit_website(self,*args):
    webbrowser.open("http://arachnoid.com/python/weatherreader")

  def close(self,*args):
    if(self.help_pane):
      self.help_pane.close()
    if(not self.check_active()):
      self.fax_decoder.unlink_source()
      self.cm.write_config()
      for chart in self.chartlist:
        if(chart):
          chart.save()
      gtk.main_quit()

app=WeatherReader()
gtk.main()
