// ***************************************************************************
// *   Copyright (C) 2013 by Paul Lutus                                      *
// *   lutusp@arachnoid.com                                                  *
// *                                                                         *
// *   This program is free software; you can redistribute it and/or modify  *
// *   it under the terms of the GNU General Public License as published by  *
// *   the Free Software Foundation; either version 2 of the License, or     *
// *   (at your option) any later version.                                   *
// *                                                                         *
// *   This program is distributed in the hope that it will be useful,       *
// *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
// *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
// *   GNU General Public License for more details.                          *
// *                                                                         *
// *   You should have received a copy of the GNU General Public License     *
// *   along with this program; if not, write to the                         *
// *   Free Software Foundation, Inc.,                                       *
// *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
// ***************************************************************************

import java.applet.*;
import java.awt.*;
import java.util.*;
import java.net.*;
//import orbitFrame;

public class orbit extends Applet implements Runnable
{
	private Thread	 m_orbit = null;

	// STANDALONE APPLICATION SUPPORT:
	//		m_fStandAlone will be set to true if applet is run standalone
	//--------------------------------------------------------------------------
	public boolean m_fStandAlone = false;

	public String AppName = "Gravitation 3.8";
	public String noClicks = AppName + " Copyright \251 P. Lutus -- http://www.arachnoid.com";
	public String oneClick = AppName + " is CareWare. Please visit http://www.arachnoid.com";
	public String twoClicks = "http://www.arachnoid.com";

	public static void main(String args[])
	{
		orbit applet_orbit = new orbit();
		orbitFrame frame = new orbitFrame(applet_orbit.AppName);

		// Must show Frame before we size it so insets() will return valid values
		//----------------------------------------------------------------------
		frame.show();
        frame.hide();
		frame.resize(frame.insets().left + frame.insets().right  + 620,
					 frame.insets().top  + frame.insets().bottom + 460);

		frame.add("Center", applet_orbit);
		applet_orbit.m_fStandAlone = true;
		applet_orbit.init();
		applet_orbit.start();
        frame.show();
	}

	public orbit()
	{
	}

	public String getAppletInfo()
	{
		return "Name: " + AppName + "\r\n" +
		       "Author: Paul Lutus\r\n" +
		       "Created with Microsoft Visual J++ Version 1.1";
	}

	// ************** my favorite place for globals **************************

	SpaceCanvas orbCanvas;
	boolean drawline = false;
	boolean dataMode = false;
	boolean newScene = false;
	boolean mouseMoving = false;
	boolean rescaling = false;
	AudioClip collideSound = null;
	numberField gravConst = new numberField("6.67e-8",12); // dyne cm^2 gm^-2
	numberField massConst = new numberField("256",12); // multiplier for radius to mass
	numberField timeInterval = new numberField("150",12);
	numberField dispScale = new numberField("16",6);
	numberField minimumRadius = new numberField("5",12);
	Color spaceColor = Color.black;
	Vector choiceList;
	int choiceIndex = 0;

	String setupString = AppName + 
		" provides a real-time simulation of multiple orbiting bodies under the influence\n" +
		"of gravity. Although the program contains predefined scenarios, you can change the\n" +
		"program's behavior in many ways. The easiest way to change the orbital behavior\n" +
		"is to click one of the orbiting bodies and drag it to a new position. Another way\n" +
		"is to click the \"data\" button and edit position and velocity data. Yet another way\n" +
		"is to change the values on this panel:\n\n" +
		"The Gravitational Constant default value corresponds to the current estimate of this value\n" +
		"in nature.\n\n" +
		"The Mass Multiplier decides how dense the orbiting bodies are -- the default value is\n" +
		"rather dense. The mass of the orbiting bodies is equal to this multiplier times the cube\n" +
		"of the selected radius.\n\n" +
		"The Time Interval is the number of seconds each calculation represents, although the\n" +
		"program runs much faster than real time. Small values for this parameter result in\n" +
		"more accurate simulations, but these take longer to compute.\n\n" +
		"The Minimum Display Radius setting chooses the minimum display size for bodies\n" + 
		"regardless of their actual computation radius.\n\n" +
		"The \"Highest Computer Speed\" checkbox runs the simulation at the highest possible\n" +
		"computer speed. This setting may not work with some computers and browsers.\n" +
		"It generally succeeds with Microsoft Internet Explorer and generally fails with Netscape.\n\n" +
		"The \"Merge on Collision\" decides how to handle a collision between two bodies. If this\n" +
		"option is set, colliding bodies merge and the resultant body has the mass and momentum\n" +
		"of the two original bodies. If this option is not set, two close-approaching bodies\n" +
		"generally fly through one another and depart the viewport at high speed, a feat not\n" +
		"possible in nature.\n\n" +
		"The \"Scale\" setting (main panel) sets the magnification of the \"viewport\". Set it\n" +
		"using the scroll control to its right or type in a new number.\n\n" +
		"Choose scenarios with the drop-down list. The \"Three Mutual Orbit\" scenario shows how\n" +
		"unstable three mutually interacting gravitationally bodies can be, although some of\n" +
		"the instability of this scenario results from the numerical approximation that is used.\n\n" +
		"Each of the scenarios can be customized by dragging the bodies on the display and by\n" +
		"editing the data values.\n\n" +
		"To see some interesting patterns, select the \"Lines\" option on the main panel.\n\n" +
		noClicks + "\n\n" +
		oneClick + "\n";

	Color defaultColor[] = {
		Color.yellow,
		Color.green,
		new Color(128,128,255),
		Color.cyan,
		Color.red,
		Color.magenta,
		Color.orange,
		Color.white,
		Color.pink
	};

	double set_solarSystem[][] = {
		{0,0,0,0,192,1},
		{-500,0,0,-.48,8,0},
		{-1000,0,0,-.34,8,0},
		{-1500,0,0,-.28,8,0},
		{-2000,0,0,-.24,8,0},
		{-2500,0,0,-.214,8,0},
		{-3000,0,0,-.195,8,0},
		{-6000,0,0,-.055,8,0}
	};

	double set_chaosSystem[][] = {
		{0,0,0,0,192,1},
		{-500,0,0,-.48,12,0},
		{-1000,0,0,.34,12,0},
		{-1500,0,0,-.28,12,0},
		{-2000,0,0,.24,12,0},
		{-2500,0,0,-.214,12,0},
		{-3000,0,0,.195,12,0},
		{750,0,0,-.48,12,0},
		{1250,0,0,.34,12,0},
		{1750,0,0,-.28,12,0},
		{2250,0,0,.24,12,0},
		{2750,0,0,-.214,12,0},
		{3250,0,0,.195,12,0},
		{-6000,0,0,-.045,12,0},
		{6000,0,0,-.055,12,0},
		{0,6000,-.045,0,12,0},
		{0,-6000,.055,0,12,0}
	};

	double set_figureEight[][] = {
		{-1550,0,0,0,192,1},
		{1550,0,0,0,192,1},
		{-3000,0,0,-.268,12,0}
	};

	double set_threeOrbit[][] = {
		{0,-1000,.15,0,128,0},
		{-866,500,-.075,-.1299,128,0},
		{866,500,-.075,.1299,128,0}
	};

	double set_fourOrbit[][] = {
		{-1500,0,0,.2,96,0},
		{-1000,0,0,-.04,96,0},
		{1550,0,0,.04,96,0},
		{950,0,0,-.2,96,0}
	};

	double set_ellipse[][] = {
		{0,0,0,0,288,1},
		{-2000,0,0,-.3,10,0},
		{2000,0,0,.3,10,0}
	};

	double set_simple[][] = {
		{0,0,0,0,192,1},
		{-1000,0,0,-.35,12,0}
	};

	String choiceName[] = {
		"Planetary System + Comet",
		"Chaotic Planetary System",
		"Three-Body Figure Eight",
		"Three Mutual Orbit",
		"Four Mutual Orbit",
		"Three-Body Elliptical Orbit",
		"Simple Orbit"
	};

	Choice setChoice;
	Checkbox lineBox;
	Checkbox highSpeed;
	Checkbox mergeOnCollision;
	CardLayout cardSet;
	Panel dataPanel;
	Panel setupPanel,topSetup,bottomSetup;
	Panel rootPanel;
	Panel controlP1,controlP2,controlPanel;
	TextArea setupHelp;

	String docBase;
	boolean atHome;

	Color controlBG = new Color(128,128,128);

	Scrollbar setScale;

	// ************* end global declarations ****************************************

	public void init()
	{

		if(m_fStandAlone == false) {
			docBase = getDocumentBase().toString().toLowerCase();
			atHome = (docBase.indexOf("arachnoid.com") >= 0);
		}

		setLayout(new BorderLayout());

		orbCanvas = new SpaceCanvas(this);
		dataPanel = new Panel();
		setupPanel = new Panel();
		setupPanel.setLayout(new GridLayout(2,1));
		topSetup = new Panel();
		bottomSetup = new Panel();
		dataPanel.setLayout(new GridLayout(12,1,5,5));
		rootPanel = new Panel();
		cardSet = new CardLayout();
		rootPanel.setLayout(cardSet);
		rootPanel.add("Space",orbCanvas);
		rootPanel.add("Data",dataPanel);
		setupPanel.add(topSetup);
		setupPanel.add(bottomSetup);
		rootPanel.add("Setup",setupPanel);

		setupHelp = new TextArea(setupString);
		bottomSetup.setLayout(new BorderLayout());
		bottomSetup.add("Center",setupHelp);

		topSetup.setLayout(new GridLayout(6,2));

		highSpeed = new Checkbox();
		mergeOnCollision = new Checkbox();
		mergeOnCollision.setState(true);

		sysAdd(new Label("Gravitational Constant (dyne cm^2 gm^-2) "),gravConst);
		sysAdd(new Label("Mass Multiplier "),massConst);
		sysAdd(new Label("Time Interval seconds "),timeInterval);
		sysAdd(new Label("Minimum Display Radius"),minimumRadius);
		sysAdd(new Label("Highest Computer Speed"),highSpeed);
		sysAdd(new Label("Merge on Collision"),mergeOnCollision);

		setChoice = new Choice();
		setChoice.setBackground(Color.white);
		lineBox = new Checkbox("Lines");

		choiceList = new Vector();

		choiceList.addElement(set_solarSystem);
		choiceList.addElement(set_chaosSystem);
		choiceList.addElement(set_figureEight);
		choiceList.addElement(set_threeOrbit);
		choiceList.addElement(set_fourOrbit);
		choiceList.addElement(set_ellipse);
		choiceList.addElement(set_simple);

		for(int i = 0;i < choiceName.length;i++) {
			setChoice.addItem(choiceName[i]);
		}

		controlPanel = new Panel();
		controlPanel.setLayout(new GridLayout(2,1));

		add("Center",rootPanel);
		add("South",controlPanel);

		orbCanvas.setBackground(spaceColor);

		controlP1 = new Panel();
		controlP1.setBackground(controlBG);

		controlP2 = new Panel();
		controlP2.setBackground(controlBG);

		controlPanel.add(controlP1);
		controlPanel.add(controlP2);

		controlP1.setLayout(new FlowLayout(FlowLayout.CENTER));
		controlP2.setLayout(new FlowLayout(FlowLayout.CENTER));

		controlP1.add(setChoice);
		controlP1.add(new Button("Go"));
		controlP1.add(new Button("Stop"));
		controlP1.add(new Button("New"));
		controlP1.add(lineBox);


		controlP2.add(new Label("Scale "));
		controlP2.add(dispScale);
		dispScale.setBackground(Color.white);

		setScale = new Scrollbar(Scrollbar.HORIZONTAL,4,0,0,32);

		controlP2.add(setScale);

		controlP2.add(new Button("View"));
		controlP2.add(new Button("Data"));
		controlP2.add(new Button("Setup"));

	}

	public void playCollideSound()
	{
		if(!m_fStandAlone) {
			play(getCodeBase(),"collide.au");
		}
	}
	
	public void resetScrollbar()
	{
		dispScale.setText("16");
		setScale.setValue(4);
	}

	public void sysAdd(Label lb,Object nf)
	{
		Panel pn1 = new Panel();
		pn1.setLayout(new FlowLayout(FlowLayout.LEFT));
		Panel pn2 = new Panel();
		pn2.setLayout(new FlowLayout(FlowLayout.LEFT));
		pn1.add(lb);
		if(nf instanceof numberField)
			pn2.add((numberField)nf);
		else
			pn2.add((Checkbox)nf);
		topSetup.add(pn1);
		topSetup.add(pn2);
	}

	public void destroy()
	{
	}

	public void start()
	{
		if (m_orbit == null)
		{
			m_orbit = new Thread(this);
			m_orbit.start();
		}
	}
	
	public void stop()
	{
		if (m_orbit != null)
		{
			m_orbit.stop();
			m_orbit = null;
		}
	}

	public void run()
	{
		show();
		orbCanvas.start();
	}


	String oldLabel = "";

	public boolean action(Event event,Object obj)
	{
		boolean r = false;
		String lbl;
		Button b;
		Object targ = event.target;
		if(targ instanceof Button)
		{
			if(oldLabel.equals("Setup")) {
				if((massConst.testValue()) || (timeInterval.testValue()))
					orbCanvas.resetAll();
			}
			b = (Button)targ;
			lbl = b.getLabel();
			if(lbl.equals("Stop")) {
				orbCanvas.running = false;
			}
			else if(lbl.equals("Go")) {
				orbCanvas.running = true;
			}
			else if(lbl.equals("New")) {
				orbCanvas.running = false;
				resetScrollbar();
				orbCanvas.resetAll();
				newScene = true;
			}
			else if(lbl.equals("Data")) {
				dataMode = true;
				// orbCanvas.updateData();
				cardSet.show(rootPanel,"Data");
			}
			else if(lbl.equals("Setup")) {
				cardSet.show(rootPanel,"Setup");
				dataMode = false;
			}
			else if(lbl.equals("View")) {
				cardSet.show(rootPanel,"Space");
				dataMode = false;
			}
			orbCanvas.updateData();
			r = true;
			orbCanvas.paint(orbCanvas.getGraphics());
			validate();
			oldLabel = lbl;
		}
		else if(targ instanceof Choice) {
			orbCanvas.running = false;
			Choice q = (Choice)targ;
			choiceIndex = q.getSelectedIndex();
			orbCanvas.setDefault((double[][])choiceList.elementAt(choiceIndex));
			resetScrollbar();
			orbCanvas.imageUpdate(false);
			orbCanvas.paint(orbCanvas.getGraphics());
			validate();
		}
		else if(targ == lineBox) {
			Checkbox q = (Checkbox)targ;
			if(q.getLabel().equals("Lines")) {
				drawline = q.getState();
				orbCanvas.paint(orbCanvas.getGraphics());
			}
		}
		return r;
	}

	public boolean handleEvent(Event e)
	{
		Object targ = e.target;
		if(targ == setScale) // react to change of scale
		{
			Scrollbar q = (Scrollbar)targ;
			double v = Math.pow(2,(double)q.getValue());
			dispScale.setText("" + v);
			dispScale.refresh();
			rescaling = true;
			orbCanvas.paint(orbCanvas.getGraphics());
			return true;
		}
		return super.handleEvent(e);
	}

	public void msg(String s)
	{
		if(m_fStandAlone)
			System.out.println(s);
		else {
			showStatus(s);
			delay(5000);
		}
	}

	public void delay(long w)
	{
		try {
			Thread.sleep(w);
		}
		catch (InterruptedException e)
		{
			stop();
		}
	}

	public void showStat(String s)
	{
		if(m_fStandAlone)
			System.out.println(s);
		else
			showStatus(s);
			
	}

	public orbitBody testBody(int x,int y)
	{
		orbitBody b = null;
		Rectangle r;
		boolean found = false;
		int i = 0;
		do
		{
			if(i < orbCanvas.bodyList.size()) {
				b = (orbitBody)orbCanvas.bodyList.elementAt(i);
				int px = orbCanvas.px + (int)(b.x.dvalue/dispScale.dvalue),py = orbCanvas.py + (int)(b.y.dvalue/dispScale.dvalue);
				r = new Rectangle(px-b.irad,py-b.irad,b.idia,b.idia);
				found = r.inside(x,y);
			}
		}
		while((!found) && (++i < orbCanvas.bodyList.size()));
		if(!found)
			b = null;
		return b;
	}

	String makeFix(double v,int n)
	{
		String sv = "" + v;
		int i = sv.lastIndexOf(".");
		if(i >= 0) {
			i += n + 1;
			if(i < sv.length())
				sv = sv.substring(0,i);
		}
		return sv;
	}

	int startx,starty;
	orbitBody mouseBody = null;

	public boolean mouseDown(Event evt, int x, int y)
	{
		mouseBody = testBody(x,y);
		if(mouseBody == null) {
			if(m_fStandAlone == false) {	
				if((!atHome) && ((evt.target instanceof Panel) || (evt.target instanceof SpaceCanvas))) { // off site access	
					if(evt.clickCount == 1) {
						showStat(oneClick);
					}
					else if (evt.clickCount == 2) {// this actually works
						try {
							getAppletContext().showDocument(new URL(twoClicks));
						}
						catch (MalformedURLException e) {
						}
					}
				}
			}
		}
		else {
			mouseMoving = true;
		}
		startx = x;
		starty = y;
		return true;
	}

	public boolean mouseUp(Event evt, int x, int y)
	{
		mouseBody = null;
		mouseMoving = false;
		return true;
	}

	public boolean mouseDrag(Event evt, int x, int y)
	{
		if(mouseBody != null) {
			mouseBody.x.dvalue = (x-orbCanvas.px) * dispScale.dvalue;
			mouseBody.y.dvalue = (y-orbCanvas.py) * dispScale.dvalue;
			mouseBody.updatePanel();
			orbCanvas.imageUpdate(false); // special case -- no label
			orbCanvas.dynamic = true;
		}
		return true;
	}

	public boolean mouseMove(Event evt, int x, int y)
	{
		return true;
	}

	public boolean mouseEnter(Event evt, int x, int y)
	{
		showStat(noClicks);
		return true;
	}
	
	public boolean mouseExit(Event evt, int x, int y)
	{
		showStat("");
		return true;
	}

}

class SpaceCanvas extends Canvas implements Runnable
{
	int w = -1,h = -1,px,py;
	Vector bodyList = null;
	orbit owner;
	boolean dynamic = false;
	boolean running = false;
	private Thread	 m_space = null;

	public void start()
	{
		if (m_space == null)
		{
			m_space = new Thread(this);
			m_space.start();
		}
	}
	
	public void stop()
	{
		if (m_space != null)
		{
			m_space.stop();
			m_space = null;
		}
	}

	public void run()
	{
		running = true;
		while (true)
		{
			if(running) {
				//try
			//	{
					imageUpdate(true);
					getToolkit().sync();
					delay((owner.highSpeed.getState()?2:5));
				//}
				//catch (InterruptedException e)
				//{
				//	stop();
				//}
			}
			else {
				delay(100);
			}
		}
	}

	private void delay(long w)
	{
		try {
			Thread.sleep(w);
		}
		catch (InterruptedException e) {
			stop();
		}
	}
	
	String fieldLabel[] = {
		"Color",
		"x",
		"y",
		"vx",
		"vy",
		"Radius",
		"Fixed",
		"Alive"
	};

	public void msg(String s)
	{
		System.out.println(s);
	}
	
	SpaceCanvas(orbit o)
	{
		super();
		owner = o;
	}

	public void setDefault(double q[][])
	{
		if(bodyList != null) {
			bodyList.removeAllElements();
		}
		bodyList = new Vector();
		owner.dataPanel.removeAll();
		Panel labelPanel = new Panel();
		labelPanel.setLayout(new GridLayout(1,fieldLabel.length));
		for(int i = 0; i < fieldLabel.length;i++) {
			labelPanel.add(new Label(fieldLabel[i]));
		}
		owner.dataPanel.add(labelPanel);

		for(int i = 0; i < q.length;i++) {
			orbitBody ob = new orbitBody(q[i][0],q[i][1],q[i][2],q[i][3],q[i][4],(q[i][5] != 0),owner.defaultColor[i % owner.defaultColor.length],owner);
			bodyList.addElement(ob);
			owner.dataPanel.add(ob.dataPanel);
		}
		owner.newScene = true;
	}

	public void updateData()
	{
		for(int i = 0;i < bodyList.size();i++) {
			((orbitBody)bodyList.elementAt(i)).updatePanel();
		}
	}

	public void resetAll()
	{
		for(int i = 0;i < bodyList.size();i++) {
			((orbitBody)bodyList.elementAt(i)).resetBody();
		}
		imageUpdate(false);
		paint(getGraphics());
	}

	public void imageUpdate(boolean dyna)
	{
		int nw,nh;
		nw = size().width;
		nh = size().height;
		orbitBody a,b;
		dynamic = dyna;
		if((h != nh) || (w != nw)) // new size or init
		{
			h = nh;
			px = nw/2;
			w = nw;
			py = nh/2;
			if(bodyList == null) {
				setDefault((double[][])owner.choiceList.elementAt(owner.choiceIndex));
				owner.newScene = false;
			}
		}
		if(dynamic) {
			for(int i = 0;i < bodyList.size();i++) {
				a = (orbitBody)bodyList.elementAt(i);
				for(int j = 0;j < bodyList.size();j++) {
					if(i != j) {
						b = (orbitBody)bodyList.elementAt(j);
						a.calcg(b,dynamic);
					}
				}
			}
		}
		for(int i = 0;i < bodyList.size();i++) {
			((orbitBody)bodyList.elementAt(i)).velUpdate(dynamic);
		}
		myupdate(getGraphics());
	}

	public void resetBodies()
	{
		Graphics gg = getGraphics(); // don't use system graphics context!
		gg.setPaintMode();
		gg.setColor(owner.spaceColor);
		gg.fillRect(0,0,w,h);
		for(int i = 0;i < bodyList.size();i++) {
			((orbitBody)bodyList.elementAt(i)).setNewDraw();
		}
	}

	public void paint(Graphics g)
	{
		resetBodies();
		myupdate(g);
	}

	public void myupdate(Graphics g)
	{
		orbitBody ob;
		Graphics gg = getGraphics(); // don't use system graphics context!
		for(int i = 0;i < bodyList.size();i++) {
			ob = (orbitBody)bodyList.elementAt(i);
			if(!owner.dataMode)
				ob.myUpdate(gg,px,py);
		}
		if(owner.newScene) {
			g.setColor(Color.yellow);
			String s1 = "Press \"Go\" to see this simulation,";
			String s2 = "or use your mouse to move a body first.";
			g.drawString(s1,(w/2)-s1.length()*2,h-40);
			g.drawString(s2,(w/2)-s2.length()*2,h-20);
			owner.newScene = false;
		}
		owner.rescaling = false;
	}

}

class orbitBody extends Canvas {
	double initx,inity,initvx,initvy,initrad;
	boolean initfixed;
	double diameter,mass,gravConst,massConst,timeIntSq;
	int oldx,oldy,irad,idia,oldDia,oldRad;
	Color bodyColor,initBodyColor,lastColor;
	orbit owner;
	Panel dataPanel;
	numberField[] textArray;
	int maxText = 5;
	boolean oldActive = false;

	numberField x = new numberField();
	numberField y = new numberField();
	numberField vx = new numberField();
	numberField vy = new numberField();
	numberField radius = new numberField();

	Checkbox fixed;
	Checkbox alive;

	public void msg(String s)
	{
		System.out.println(s);
	}

	orbitBody(double xx, double yy, double vxx, double vyy, double rr,boolean fix,Color bc,orbit o)
	{
		owner = o;
		initx = xx;
		inity = yy;
		initvx = vxx;
		initvy = vyy;
		initrad = rr;
		radius.dvalue = initrad;
		newRadius();
		initfixed = fix;
		fixed = new Checkbox("Yes");
		alive = new Checkbox("Yes");
		fixed.setState(initfixed);
		initBodyColor = bc;
		bodyColor = bc;
		lastColor = bc;
		resetBody();
		makePanel();
	}

	public Panel makePanel()
	{
		dataPanel = new Panel();
		dataPanel.setLayout(new GridLayout(1,maxText+3,5,0));
		Panel colorDot = new Panel();
		colorDot.setBackground(bodyColor);
		dataPanel.add(colorDot);
		textArray = new numberField[maxText];
		for(int i = 0;i < maxText;i++) {
			switch(i) {
			case 0:
				textArray[i] = x;
				break;
			case 1:
				textArray[i] = y;
				break;
			case 2:
				textArray[i] = vx;
				break;
			case 3:
				textArray[i] = vy;
				break;
			case 4:
				textArray[i] = radius;
				break;
			}
			dataPanel.add(textArray[i]);
		}
		Panel fixedH = new Panel(); // extra panels for centering
		Panel aliveH = new Panel();
		dataPanel.add(fixedH);
		dataPanel.add(aliveH);
		fixedH.add(fixed);
		aliveH.add(alive);
		updatePanel();
		return dataPanel;
	}

	public void updatePanel()
	{
		for(int i = 0;i < maxText;i++) {
			textArray[i].refresh();
		}
	}
	
	public void resetBody()
	{
		alive.setState(true);
		x.dvalue = initx;
		y.dvalue = inity;
		bodyColor = initBodyColor;
		lastColor = initBodyColor;
		radius.dvalue = initrad;
		newRadius();
		oldDia = idia;
		oldRad = irad;
		setNewDraw();
		vx.dvalue = initvx * owner.timeInterval.dvalue;
		vy.dvalue = initvy * owner.timeInterval.dvalue;
		gravConst = owner.gravConst.dvalue;
		timeIntSq = owner.timeInterval.dvalue * owner.timeInterval.dvalue;
		fixed.setState(initfixed);
	}

	public void newRadius()
	{
		massConst = owner.massConst.dvalue;
		diameter = radius.dvalue * 2;
		irad = (int)(radius.dvalue/owner.dispScale.dvalue);
		irad = ((irad < owner.minimumRadius.ivalue)?owner.minimumRadius.ivalue:irad);
		idia = irad * 2;
		mass = massConst * radius.dvalue * radius.dvalue * radius.dvalue;
		radius.newValue = false;
	}

	public void setNewDraw()
	{
		oldActive = false;
		oldx = -100000;
		oldy = -100000;
	}

	public void handleCollision(orbitBody a,orbitBody b) // a lives, b dies
	{
		double sumMass = a.mass + b.mass;
		a.vx.dvalue = (a.vx.dvalue * (a.mass/sumMass)) + (b.vx.dvalue * (b.mass/sumMass));
		a.vy.dvalue = (a.vy.dvalue * (a.mass/sumMass)) + (b.vy.dvalue * (b.mass/sumMass));
		a.mass = sumMass;
		a.radius.setText("" + Math.pow((a.mass/a.massConst),1.0/3.0));
		a.newRadius();
		b.alive.setState(false);
		a.bodyColor = sumColor(a.bodyColor,b.bodyColor);
		owner.playCollideSound();
	}

	public Color sumColor(Color x,Color y)
	{
		int r,g,b;
		r = (x.getRed() + y.getRed()) /2;
		g = (x.getGreen() + y.getGreen()) /2;
		b = (x.getBlue() + y.getBlue()) /2;
		return new Color(r,g,b);
	}

	// surprise -- no trig required!

	public void calcg(orbitBody q,boolean dynamic)
	{
		boolean collision = false;
		if(alive.getState() && q.alive.getState()) {
			double dx = x.dvalue - q.x.dvalue;
			double dy = y.dvalue - q.y.dvalue;
			double m = ((dx * dx) + (dy * dy)); // need this value in two places
			double sqm = Math.sqrt(m); // sqm is the actual radial distance
			
			if(owner.mergeOnCollision.getState()) {
				if(((radius.dvalue + q.radius.dvalue) >= sqm) && (!q.fixed.getState())) {// if bodies have collided and other is not fixed
					handleCollision(this,q);
					collision = true;
				}
			}
			if(!collision) {
				if((!fixed.getState()) && (dynamic)) {	
					if(m != 0.0) { // don't bother if this value is zero
						m = gravConst * q.mass * timeIntSq / m;
						vx.dvalue -= m * (dx/sqm); // equivalent to sin(atan2(dy,dx))
						vy.dvalue -= m * (dy/sqm); // equivalent to cos(atan2(dy,dx))
					}
				}
			}
		}
	}

	public void velUpdate(boolean dynamic)
	{
		if(alive.getState() && (!fixed.getState()) && (dynamic)) {
			x.dvalue += vx.dvalue;
			y.dvalue += vy.dvalue;
			if(owner.dataMode)
				updatePanel();
		}
	}

	public void myUpdate(Graphics g,int cx,int cy)
	{
		if((radius.newValue) || (owner.rescaling)) {
			newRadius();
		}
		int qrad = ((owner.drawline)?2:irad);
		int qdia = ((owner.drawline)?4:idia);
		int px = cx + (int)(x.dvalue/owner.dispScale.dvalue),py = cy + (int)(y.dvalue/owner.dispScale.dvalue);
		int ovx = px-qrad,ovy = py-qrad;
		if((!owner.drawline) || (owner.mouseMoving)) {
			g.setXORMode(owner.spaceColor);
		}
		g.setColor(bodyColor);
		if((oldx != px) || (oldy != py) || (oldActive != alive.getState()) || (oldRad != qrad) || (!lastColor.equals(bodyColor)) ) {
			if(oldActive) {
				g.setColor(lastColor);
				g.fillOval(oldx-oldRad,oldy-oldRad,oldDia,oldDia);
				g.setColor(bodyColor);
				if((alive.getState()) && (owner.drawline) && (!owner.mouseMoving) && (!owner.rescaling)) {
					g.drawLine(oldx,oldy,px,py);
				}				
			}
			if(alive.getState()) {
				g.fillOval(ovx,ovy,qdia,qdia);
			}
		}
		oldx = px;
		oldy = py;
		oldDia = qdia;
		oldRad = qrad;
		oldActive = alive.getState();
		lastColor = bodyColor;
	}

}

class numberField extends TextField
{
	double dvalue = 0;
	int ivalue = 0;
	long lvalue = 0;
	public boolean newValue = false;

	numberField()
	{
		super ();
	}

	numberField(String s)
	{
		super(s);
		convert();
	}

	numberField(String s,int w)
	{
		super(s,w);
		convert();
	}

	numberField(int w)
	{
		super(w);
	}

	public void setText(String s)
	{
		super.setText(s);
		convert();
	}

	boolean testValue() // test and reset flag
	{
		boolean r = newValue;
		newValue = false;
		return r;
	}

	private void convert()
	{
		try {
			dvalue = Double.valueOf(this.getText()).doubleValue();
			ivalue = (int)dvalue;
			lvalue = (long)dvalue;
		}
		catch (NumberFormatException e) {
		}
	}

	public boolean handleEvent(Event e)
	{
		convert();
		newValue = true;
		return false; // allow keystrokes to be handled by super
	}

	public void refresh()
	{
		super.setText(makeFix(dvalue,2));
	}

	String makeFix(double v,int n)
	{
		String sv = "" + v;
		int i = sv.lastIndexOf(".");
		if(i >= 0) {
			i += n + 1;
			if(i < sv.length())
				sv = sv.substring(0,i);
		}
		return sv;
	}

}
