// ***************************************************************************
// *   Copyright (C) 2018 by Paul Lutus                                      *
// *   lutusp@arachnoid.com                                                  *
// *                                                                         *
// *   This program is free software; you can redistribute it and/or modify  *
// *   it under the terms of the GNU General Public License as published by  *
// *   the Free Software Foundation; either version 2 of the License, or     *
// *   (at your option) any later version.                                   *
// *                                                                         *
// *   This program is distributed in the hope that it will be useful,       *
// *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
// *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
// *   GNU General Public License for more details.                          *
// *                                                                         *
// *   You should have received a copy of the GNU General Public License     *
// *   along with this program; if not, write to the                         *
// *   Free Software Foundation, Inc.,                                       *
// *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
// ***************************************************************************

/*
 * SpellCheckPanel.java
 *
 * Created on February 26, 2002, 7:47 AM
 */
package SpellCheck;

import Arachnophilia.*;
import java.awt.*;
import java.io.*;
import java.util.*;
import java.util.zip.*;
import javax.swing.*;
import javax.swing.text.*;

/**
 *
 * @author Administrator
 */
final public class SpellCheckPanel extends javax.swing.JPanel {

    Arachnophilia main;
    SpellCheckFrame parent;
    TreeSet<String> mainDict, customDict, ignoreWordList;
    String[] mainDictArray = null;
    String dictPath;
    String skipDataPath;
    HashMap<String, String> skipTagList;
    String currentWord = "";
    boolean panelLocked = false;
    boolean selectionValid = false;
    boolean validWordDisplay = false;
    ArachDocument currentDocument = null;
    String skipTagFileName = "SpellSkipTags.txt";
    DocContentHandler docHandler;
    SearchReplace srch;
    boolean stopSearch = false;
    int defaultDocPosition = 0;
    int docPosition = 0;
    int endPosition = 0;

    /**
     * Creates new form SpellCheckPanel
     */
    public SpellCheckPanel(Arachnophilia m, SpellCheckFrame p) {
        main = m;
        parent = p;
        initComponents();
        docHandler = new DocContentHandler(main);
        srch = new SearchReplace();
        isHTMLCheckBox.setSelected(main.configValues.spellIsHTML);
        ignoreUCCheckBox.setSelected(main.configValues.spellIgnoreUCWords);
        customDict = new TreeSet<String>(new CaseInsensitiveComparator());
        mainDict = new TreeSet<String>(new CaseInsensitiveComparator());
        ignoreWordList = new TreeSet<String>();
        dictPath = main.basePath + "/" + Arachnophilia.spellCheckDataDirName;
        skipDataPath = dictPath + "/" + skipTagFileName;
        skipTagList = ArachComp.loadParserData(skipDataPath);
        loadDictionariesThread(dictPath, mainDict, customDict);

    }
    /*
     private HashMap loadData(String fileName) {
     String path = main.basePath
     + ArachConstants.SYSTEM_FILESEP
     + main.htmlBeautifyDataDirName
     + ArachConstants.SYSTEM_FILESEP
     + fileName;
     HashMap hash = ArachComp.loadParserData(path);
     if(hash == null) {
     validSetup = false;
     }
     return hash;
     }*/

    private void combineDictionaries(TreeSet<String> dict, TreeSet<String> cust) {
        // it could be done this way instead
        // dict.addAll(stand);
        // dict.addAll(cust);
        int totalLen = cust.size();
        addDictionary(dict, cust, totalLen);
    }

    private void addDictionary(TreeSet<String> dict, TreeSet<String> add, int totalLen) {
        int n = 0;
        if (add != null) {
            Iterator<String> it = add.iterator();
            while (it.hasNext()) {
                dict.add(it.next());
                n++;
                if ((n % 20) == 0) {
                    setProgressBar("Adding to dictionary", 0, n, totalLen);
                }
            }
        }
    }

    private void loadDictionariesThread(final String path, final TreeSet<String> dict, final TreeSet<String> cust) {
        if (!panelLocked) {
            panelLocked = true;
            Thread t = new Thread() {
                @Override
                public void run() {
                    loadDictionaries(path, dict, cust);
                    combineDictionaries(dict, cust);
                    mainDictArray = mainDict.toArray(new String[]{});
                    setProgressBar("Ready to start.", 0, 0, 0);
                }
            };
            t.start();
        }
    }

    private void loadDictionaries(String path, TreeSet<String> stand, TreeSet<String> cust) {
        // add desired comparator

        File[] list = new File(path).listFiles();
        if (list != null) {
            for (int i = 0; i < list.length; i++) {
                File f = list[i];
                String name = f.getName();
                //System.out.println("load: " + name);
                if (name.endsWith(".zip")) {
                    readZipFile(f, stand, cust);
                } else if (name.endsWith(".txt") && !name.equals(skipTagFileName)) {
                    if (name.toLowerCase().startsWith("custom")) {
                        readTextFile(f, cust);
                    } else {
                        readTextFile(f, stand);
                    }
                }
            }
        }
        panelLocked = false;
    }

    private void readZipFile(File f, TreeSet<String> stand, TreeSet<String> cust) {
        try {
            //System.out.println("readzip: " + f.getName());
            ZipFile zip = new ZipFile(f.getPath());
            Enumeration entries = zip.entries();
            while (entries.hasMoreElements()) {
                ZipEntry entry = (ZipEntry) entries.nextElement();
                InputStream is = zip.getInputStream(entry);
                InputStreamReader ir = new InputStreamReader(zip.getInputStream(entry));
                String name = entry.toString();
                //System.out.println("read: " + name);
                if (name.endsWith(".txt")) {
                    if (name.toLowerCase().startsWith("custom")) {
                        readStream(ir, cust, is.available(), name);
                    } else {
                        readStream(ir, stand, is.available(), name);
                    }
                }
                ir.close();
            }
            zip.close();
        } catch (Exception e) {
            e.printStackTrace(System.out);
        }
    }

    private void readStream(InputStreamReader ir, TreeSet<String> dict, int totalLen, String name) {
        if (totalLen > 0) {
            try {
                BufferedReader bis = new BufferedReader(ir);
                String line;
                //progressBar.setMinimum(0);
                //progressBar.setMaximum(totalLen);
                int len = 0;
                int n = 0;
                while ((line = bis.readLine()) != null) {
                    dict.add(line);
                    len += line.length();
                    if ((n++ % 20) == 0) {
                        //System.out.println(len + "," + totalLen);
                        setProgressBar("Reading " + name, 0, len, totalLen);
                    }
                }
                setProgressBar("Done.", 0, 0, 0);
                bis.close();
            } catch (Exception e) {
                e.printStackTrace(System.out);
            }
        }
    }

    private void writeStream(OutputStreamWriter ow, TreeSet<String> dict) {
        try {
            BufferedWriter bw = new BufferedWriter(ow);
            String line;
            Iterator<String> it = dict.iterator();
            while (it.hasNext()) {
                line = it.next();
                bw.write(line + ArachConstants.SYSTEM_EOL);
            }
            bw.close();
        } catch (Exception e) {
            e.printStackTrace(System.out);
        }
    }

    private void readTextFile(File f, TreeSet<String> dict) {
        //System.out.println("readtext: " + f.getName());
        try {
            InputStream is = new FileInputStream(f);
            int len = is.available();
            InputStreamReader ir = new InputStreamReader(is);
            readStream(ir, dict, len, f.getName());
            ir.close();
            is.close();
        } catch (Exception e) {
            e.printStackTrace(System.out);
        }
    }

    private void saveTextFile(File f, TreeSet<String> dict) {
        if (dict != null && dict.size() > 0) {
            try {
                OutputStreamWriter ow = new OutputStreamWriter(new FileOutputStream(f));
                writeStream(ow, dict);
                ow.close();
            } catch (Exception e) {
                e.printStackTrace(System.out);
            }
        }
    }

    public void saveCustomDict() {
        if (customDict != null && customDict.size() > 0) {
            StringBuilder path = new StringBuilder();
            path.append(dictPath).append("/" + Arachnophilia.customDictName);
            saveTextFile(new File(path.toString()), customDict);
        }
    }

    // the case-insensitive comparator
    // that (I think) places UC and lc cases adjacent
    class CaseInsensitiveComparator implements Comparator<String> {

        @Override
        public int compare(String a, String b) {
            int i = a.compareToIgnoreCase(b);
            if (i == 0) {
                return (a.compareTo(b));
            } else {
                return i;
            }
        }

        public boolean equals(java.lang.Object a, java.lang.Object b) {
            return ((String) a).equals((String) b);
        }
    }

    public void spellCheckEntry() {
        ArachDocument doc = main.currentSelectedDocument;
        if (doc != null) {
            int htmlFileType = main.fileTypes.getFileTypeForName("HTML");
            if (htmlFileType == doc.getFileType()) {
                isHTMLCheckBox.setSelected(true);
                resetIsHTMLFlag(isHTMLCheckBox);
            }
            // wipe out temporary ignore list
            ignoreWordList = new TreeSet<String>();
            dictList.setListData(new String[]{});
            setProgressBar("Ready to start.", 0, 0, 0);
        }
    }

    private void spellCheck() {
        ArachDocument doc = main.currentSelectedDocument;
        if (doc == null || panelLocked) {
            main.beep();
        } else {
            if (currentDocument != doc) {
                ignoreWordList = new TreeSet<String>();
                currentDocument = doc;
            }
            dictList.setListData(new String[]{});
            wordField.setText("");
            int a = doc.textComp.getSelectionStart();
            int b = doc.textComp.getSelectionEnd();
            if (b > (a + 32)) {
                defaultDocPosition = a;
                endPosition = b;
                selectionValid = true;
            } else {
                defaultDocPosition = 0;
                endPosition = doc.textComp.getLength() - 1;
                selectionValid = false;
            }
        }
        docPosition = defaultDocPosition;
        scanForNextWordThread();
    }

    private void addToCustomDict() {
        if (validWordDisplay && !panelLocked) {
            if (currentDocument != null && currentDocument == main.currentSelectedDocument) {

                customDict.add(currentWord);
                combineDictionaries(mainDict, customDict);
                mainDictArray = mainDict.toArray(new String[]{});
                scanForNextWordThread();
            } else { // not valid
                spellCheck();
            }
        } else {
            main.beep();
        }
    }

    private boolean nonDictWordWarning(String word) {
        int reply = JOptionPane.OK_OPTION;
        if (!wordOK(word, mainDict)) {
            reply = JOptionPane.showConfirmDialog(this, "The replacement word you have chosen --\n"
                    + "\"" + word + "\" -- is not in\n"
                    + "the dictionary. OK to proceed?", "Word not in dictionary", JOptionPane.YES_NO_OPTION);
        }
        return reply == JOptionPane.YES_OPTION;
    }

    private void replaceWord() {
        if (validWordDisplay && !panelLocked) {
            if (currentDocument != null && currentDocument == main.currentSelectedDocument) {

                String replace = replaceField.getText();
                boolean r = nonDictWordWarning(replace);
                if (r) {
                    currentDocument.undoPush();
                    int a = currentDocument.textComp.getSelectionStart();
                    int b = currentDocument.textComp.getSelectionEnd();
                    currentDocument.textComp.replaceSelection(replace);
                    docPosition += replace.length() - (b - a);
                    scanForNextWordThread();
                }
            } else { // not valid
                spellCheck();
            }
        } else {
            main.beep();
        }
    }

    private void ignoreAll() {

        if (validWordDisplay && !panelLocked) {
            if (currentDocument != null && currentDocument == main.currentSelectedDocument) {

                ignoreWordList.add(currentWord);
                scanForNextWordThread();
            } else { // not valid
                spellCheck();
            }
        } else {
            main.beep();
        }

    }

    private void scanForNextWordThread() {
        if (!panelLocked) {
            panelLocked = true;
            Thread t1 = new Thread() {
                @Override
                public void run() {
                    stopSearch = false;
                    scanForNextWord();
                    panelLocked = false;
                }
            };
            t1.start();
            //System.out.println("thread started.");
        } else {
            main.beep();
        }
    }

    private void scanForNextWord() {
        boolean doMore = true;
        if (currentDocument != null && currentDocument == main.currentSelectedDocument) {
            String source = currentDocument.textComp.getText();
            int top = source.length();
            boolean endOfDocument = false;
            validWordDisplay = false;
            while (doMore && !stopSearch) {
                StringBuilder word = new StringBuilder();
                int start = 0;
                int end;
                int lastValidWordPos = 0;
                boolean inWord = false;
                boolean found = false;
                boolean inTag = false;
                boolean inScript = false;
                int n = 0;
                while (!stopSearch && !found && docPosition <= endPosition + 1 && docPosition <= top) {
                    char c = ' ';
                    if (docPosition < top && docPosition <= endPosition) {
                        c = source.charAt(docPosition);
                    }
                    //System.out.println("[" + c + "]");
                    if (c == '<' && main.configValues.spellIsHTML) {
                        inTag = true;
                        String tag = isolateTag(source, docPosition, top);
                        if (skipTagList.containsKey(tag)) {
                            String endMark = skipTagList.get(tag);
                            int p = srch.findText(source, endMark, docPosition, false);
                            if (p != -1) {
                                docPosition = p + endMark.length() - 1;
                                //skip = true;
                            }
                        }
                        /*if(tag.equals("script")) {
                         inScript = true;
                         }
                         else if(tag.equals("/script")) {
                         inScript = false;
                         }*/
                    } else if (c == '&' && main.configValues.spellIsHTML) {
                        String entity = isolateEntity(source, docPosition, top);
                        if (entity.length() > 0) {

                            if (entity.equals("nbsp")) { // handle this specially
                                c = ' ';
                            } else {
                                try {
                                    c = (char) main.entityProcessor.entityCharToNumber(entity, true);
                                } catch (Exception e) {
                                    e.printStackTrace(System.out);
                                    c = 0;
                                }
                            }
                            docPosition += entity.length() + 1;
                        }

                    }
                    if (!inTag && !inScript) {
                        if (inWord) {
                            if (!wordCharacter(c)) {
                                found = true;
                            } else {
                                word.append(c);
                                lastValidWordPos = docPosition;
                            }
                        } else { // not in word
                            if (wordCharacter(c)) {
                                start = docPosition;
                                word.append(c);
                                lastValidWordPos = docPosition;
                                inWord = true;
                            }
                        }
                    } else { // in tag
                        if (c == '>') {
                            inTag = false;
                        }
                    }
                    docPosition++;
                }
                if (docPosition >= top || docPosition >= endPosition + 1 || stopSearch) {
                    setProgressBar("Done.", 0, 0, 100);
                    doMore = false;
                    endOfDocument = true;
                    main.beep();

                } else {
                    if ((n++ % 20) == 0) {
                        setProgressBar("Scanning", defaultDocPosition, docPosition - 1, top);
                    }
                }
                if (found) {
                    end = lastValidWordPos;
                    String sword = word.toString();
                    if (!wordOK(sword, mainDict)) {
                        if (!wordOK(sword, ignoreWordList)) {
                            processFind(sword, start, end, top);
                            doMore = false;
                        }
                    }
                }

            }
        } else { // doc references don't match
            //doMore = false;
            main.beep();
        }
        if (stopSearch) {
            setProgressBar("Stopped.", 0, 0, 0);
        }
    }

    private void processFind(final String word, final int start, final int end, final int top) {
        Runnable update = new Runnable() {
            @Override
            public void run() {
                processFindDispatch(word, start, end, top);
            }
        };
        SwingUtilities.invokeLater(update);
    }

    private void processFindDispatch(String word, int start, int end, int top) {
        setProgressBar("Scanning", defaultDocPosition, docPosition - 1, top);
        setList(word, mainDict);
        showFind(currentDocument, start, end + 1);
        //currentDocument.centerCaretOnPage();
        wordField.setText(word);
        currentWord = word;
        validWordDisplay = true;
    }

    private void setList(String word, TreeSet<String> dict) {
        SortedSet head = dict.headSet(word);
        int index = head.size();
        String nearestWord = "";
        int sz = 32;
        int max = mainDictArray.length;
        //System.out.println( max);
        int listIndex = -1;
        int n = 0;
        ArrayList<String> listContent = new ArrayList<String>();

        for (int i = -sz; i < sz; i++) {
            int j = (index + i);
            if (j >= 0 && j < max) {
                if (i == 0) {
                    listIndex = n;
                    nearestWord = mainDictArray[j];
                }
                listContent.add(mainDictArray[j]);
                n++;
            }
        }
        replaceField.setText(nearestWord);
        dictList.setListData(listContent.toArray(new String[]{}));
        if (listIndex >= 0 && listIndex < n) {
            dictList.setSelectedIndex(listIndex);
            Point p = dictList.indexToLocation(listIndex);
            Dimension d = listScrollPane.getSize();
            int value = p.y - d.height / 2;
            // this won't work the "first time"
            // unless the enclosed component has a
            // considerable default preferred size
            listScrollPane.getVerticalScrollBar().setValue(value);
        }
    }

    private void setProgressBar(final String prompt, final int start, final int v, final int top) {
        Runnable update = new Runnable() {
            @Override
            public void run() {
                setProgressBarDispatch(prompt, start, v, top);
            }
        };
        SwingUtilities.invokeLater(update);
    }

    private void setProgressBarDispatch(String prompt, int start, int v, int top) {
        if (prompt.endsWith(".")) {
            progressBar.setValue(0);
            progressBar.setString(prompt);
        } else if (top > 0) {
            progressBar.setMinimum(start);
            progressBar.setMaximum(top);
            progressBar.setValue(v);
            progressBar.setString(prompt + " " + ((100 * v) / top) + "%");
            //System.out.println("pb: "+ start + " , " + v + " , " + top);
        }
    }

    private void showFind(ArachDocument doc, int start, int end) {
        doc.centerCaret = true;
        doc.textComp.select(start, end);
        //doc.textComp.setCaretPosition(start);
        //doc.textComp.moveCaretPosition(end);
        Caret car = doc.textComp.getCaret();
        if (car != null) {
            car.setSelectionVisible(true);
        }
    }

    private boolean wordOK(String word, TreeSet dict) {
        //System.out.println("testing [" + word + "]: " + dict.contains(word));
        // test default
        if (dict.contains(word)) {
            return true;
        }
        // all uppercase?
        if (main.configValues.spellIgnoreUCWords) {
            if (word.toUpperCase().equals(word)) {
                return true;
            }
        }
        // test lowercase
        if (dict.contains(word.toLowerCase())) {
            return true;
        }
        // don't test single charcacters
        if (word.length() < 2) {
            return true;
        }

        // trivial possessive case
        if (word.length() > 2 && word.toLowerCase().endsWith("\'s")) {
            return wordOK(word.substring(0, word.length() - 2), dict);
        }
        // not-so-trivial possessive case
        if (word.length() > 1 && word.toLowerCase().endsWith("s")) {
            return wordOK(word.substring(0, word.length() - 1), dict);
        }
        // starts with apostrophe
        if (word.length() > 1 && word.toLowerCase().endsWith("\'")) {
            return wordOK(word.substring(0, word.length() - 1), dict);
        }
        // ends with apostrophe
        if (word.length() > 1 && word.toLowerCase().startsWith("\'")) {
            return wordOK(word.substring(word.length() - 1), dict);
        }

        return false;
    }

    // basically, only clear the flag if character
    // is not uppercase
    private boolean testUpperCase(char c, boolean isUC) {
        if (!Character.isUpperCase(c)) {
            isUC = false;
        }
        return isUC;
    }

    private boolean wordCharacter(char c) {
        // letter, apostrope, hyphen
        return Character.isLetter(c) || c == '\'';
    }

    private String isolateTag(String s, int i, int len) {
        StringBuilder tag = new StringBuilder();
        char c;
        while (i < len) {
            c = s.charAt(i);
            if (c != '<') {
                break;
            }
            i++;
        }
        while (i < len) {
            c = s.charAt(i);
            if (Character.isWhitespace(c) || c == '>') {
                break;
            } else {
                tag.append(c);
            }
            i++;
        }
        return tag.toString().toLowerCase();
    }

    private String isolateEntity(String s, int i, int len) {
        int maxLen = 6;
        int n = 0;
        char c;
        StringBuilder tag = new StringBuilder();
        while (i < len && n < maxLen) {
            c = s.charAt(i);
            if (c != '&') {
                break;
            }
            i++;
            n++;
        }
        while (i < len && n < maxLen) {
            c = s.charAt(i);
            if (c == ';') {
                break;
            }
            tag.append(c);
            i++;
            n++;
        }
        if (n >= maxLen) {
            return "";
        }
        return tag.toString();
    }

    private void selectDictionaryWord() {
        String s = (String) dictList.getSelectedValue();
        if (s != null) {
            replaceField.setText(s);
        } else {
            main.beep();
        }
    }

    private void startOrStop() {
        if (!panelLocked) {
            spellCheck();
        } else {
            stopSearch = true;
        }
    }

    private void resetIsHTMLFlag(Object box) {
        main.configValues.spellIsHTML =
                ((JCheckBox) box).getSelectedObjects() != null;
    }

    private void resetIgnoreUCFlag(Object box) {
        main.configValues.spellIgnoreUCWords =
                ((JCheckBox) box).getSelectedObjects() != null;
    }

    /**
     * This method is called from within the constructor to initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is always
     * regenerated by the Form Editor.
     */
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {
        java.awt.GridBagConstraints gridBagConstraints;

        rightPane = new javax.swing.JPanel();
        startStopButton = new MyJButton();
        ignoreButton = new MyJButton();
        ignoreAllButton = new MyJButton();
        addButton = new MyJButton();
        changeButton = new MyJButton();
        hideButton = new MyJButton();
        centerPane = new javax.swing.JPanel();
        jLabel1 = new javax.swing.JLabel();
        wordField = new javax.swing.JTextField();
        jLabel2 = new javax.swing.JLabel();
        listScrollPane = new javax.swing.JScrollPane();
        dictList = new JList<String>();
        jLabel4 = new javax.swing.JLabel();
        replaceField = new javax.swing.JTextField();
        bottomPanel = new javax.swing.JPanel();
        progressBar = new javax.swing.JProgressBar();
        isHTMLCheckBox = new javax.swing.JCheckBox();
        ignoreUCCheckBox = new javax.swing.JCheckBox();

        setLayout(new java.awt.BorderLayout());

        rightPane.setLayout(new java.awt.GridBagLayout());

        startStopButton.setText("Start/Stop");
        startStopButton.setToolTipText("Start from beginning/Stop scan");
        startStopButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                startStopButtonMouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.insets = new java.awt.Insets(4, 4, 4, 4);
        rightPane.add(startStopButton, gridBagConstraints);

        ignoreButton.setText("Ignore once");
        ignoreButton.setToolTipText("Skip this word, move on");
        ignoreButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                ignoreButtonActionPerformed(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.insets = new java.awt.Insets(4, 4, 4, 4);
        rightPane.add(ignoreButton, gridBagConstraints);

        ignoreAllButton.setText("Ignore all");
        ignoreAllButton.setToolTipText("Ignore all cases of this word in this document");
        ignoreAllButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                ignoreAllButtonActionPerformed(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.insets = new java.awt.Insets(4, 4, 4, 4);
        rightPane.add(ignoreAllButton, gridBagConstraints);

        addButton.setText("Add");
        addButton.setToolTipText("Add this word to your custom dictionary");
        addButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                addButtonActionPerformed(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.insets = new java.awt.Insets(4, 4, 4, 4);
        rightPane.add(addButton, gridBagConstraints);

        changeButton.setText("Replace");
        changeButton.setToolTipText("Replace word in document");
        changeButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                changeButtonActionPerformed(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 4;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.insets = new java.awt.Insets(4, 4, 4, 4);
        rightPane.add(changeButton, gridBagConstraints);

        hideButton.setText("Hide");
        hideButton.setToolTipText("Hide this dialog");
        hideButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                hideButtonActionPerformed(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 5;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.insets = new java.awt.Insets(4, 4, 4, 4);
        rightPane.add(hideButton, gridBagConstraints);

        add(rightPane, java.awt.BorderLayout.EAST);

        centerPane.setLayout(new java.awt.GridBagLayout());

        jLabel1.setText("Not found:");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(4, 8, 4, 4);
        centerPane.add(jLabel1, gridBagConstraints);

        wordField.setToolTipText("This word was not found in the dctionary");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(4, 4, 4, 4);
        centerPane.add(wordField, gridBagConstraints);

        jLabel2.setText("Nearest match in dictionary:");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(4, 8, 4, 4);
        centerPane.add(jLabel2, gridBagConstraints);

        dictList.setToolTipText("Nearest match list");
        dictList.setMinimumSize(new java.awt.Dimension(10, 1000));
        dictList.setPreferredSize(new java.awt.Dimension(10, 1000));
        dictList.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                dictListMouseClicked(evt);
            }
        });
        listScrollPane.setViewportView(dictList);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.weighty = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(4, 8, 4, 4);
        centerPane.add(listScrollPane, gridBagConstraints);

        jLabel4.setText("Replace with:");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.insets = new java.awt.Insets(4, 8, 4, 4);
        centerPane.add(jLabel4, gridBagConstraints);

        replaceField.setToolTipText("<html>Select a dictionary word<br>or type in a replacement word</html>");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(4, 4, 4, 4);
        centerPane.add(replaceField, gridBagConstraints);

        add(centerPane, java.awt.BorderLayout.CENTER);

        bottomPanel.setLayout(new java.awt.GridBagLayout());

        progressBar.setStringPainted(true);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(4, 4, 4, 4);
        bottomPanel.add(progressBar, gridBagConstraints);

        isHTMLCheckBox.setText("Is HTML");
        isHTMLCheckBox.setToolTipText("Use special HTML document processing");
        isHTMLCheckBox.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                isHTMLCheckBoxActionPerformed(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(4, 4, 4, 4);
        bottomPanel.add(isHTMLCheckBox, gridBagConstraints);

        ignoreUCCheckBox.setText("Skip Uppercase");
        ignoreUCCheckBox.setToolTipText("Disregard ALL-UPPERCASE words");
        ignoreUCCheckBox.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                ignoreUCCheckBoxActionPerformed(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(4, 4, 4, 4);
        bottomPanel.add(ignoreUCCheckBox, gridBagConstraints);

        add(bottomPanel, java.awt.BorderLayout.SOUTH);
    }// </editor-fold>//GEN-END:initComponents

    private void hideButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_hideButtonActionPerformed
        // Add your handling code here:
        parent.saveAndHide();
    }//GEN-LAST:event_hideButtonActionPerformed

    private void changeButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_changeButtonActionPerformed
        // Add your handling code here:
        replaceWord();
    }//GEN-LAST:event_changeButtonActionPerformed

    private void startStopButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_startStopButtonMouseClicked
        // Add your handling code here:
        startOrStop();
    }//GEN-LAST:event_startStopButtonMouseClicked

    private void dictListMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_dictListMouseClicked
        // Add your handling code here:
        selectDictionaryWord();
    }//GEN-LAST:event_dictListMouseClicked

    private void addButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_addButtonActionPerformed
        // Add your handling code here:
        addToCustomDict();
    }//GEN-LAST:event_addButtonActionPerformed

    private void ignoreAllButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_ignoreAllButtonActionPerformed
        // Add your handling code here:
        ignoreAll();
    }//GEN-LAST:event_ignoreAllButtonActionPerformed

    private void ignoreUCCheckBoxActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_ignoreUCCheckBoxActionPerformed
        // Add your handling code here:
        resetIgnoreUCFlag(evt.getSource());
    }//GEN-LAST:event_ignoreUCCheckBoxActionPerformed

    private void ignoreButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_ignoreButtonActionPerformed
        // Add your handling code here:
        scanForNextWord();
    }//GEN-LAST:event_ignoreButtonActionPerformed

    private void isHTMLCheckBoxActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_isHTMLCheckBoxActionPerformed
        // Add your handling code here:
        resetIsHTMLFlag(evt.getSource());
    }//GEN-LAST:event_isHTMLCheckBoxActionPerformed
    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JButton addButton;
    private javax.swing.JPanel bottomPanel;
    private javax.swing.JPanel centerPane;
    private javax.swing.JButton changeButton;
    private javax.swing.JList<String> dictList;
    private javax.swing.JButton hideButton;
    private javax.swing.JButton ignoreAllButton;
    private javax.swing.JButton ignoreButton;
    private javax.swing.JCheckBox ignoreUCCheckBox;
    private javax.swing.JCheckBox isHTMLCheckBox;
    private javax.swing.JLabel jLabel1;
    private javax.swing.JLabel jLabel2;
    private javax.swing.JLabel jLabel4;
    private javax.swing.JScrollPane listScrollPane;
    private javax.swing.JProgressBar progressBar;
    private javax.swing.JTextField replaceField;
    private javax.swing.JPanel rightPane;
    private javax.swing.JButton startStopButton;
    private javax.swing.JTextField wordField;
    // End of variables declaration//GEN-END:variables
}
