// ***************************************************************************
// *   Copyright (C) 2018 by Paul Lutus                                      *
// *   lutusp@arachnoid.com                                                  *
// *                                                                         *
// *   This program is free software; you can redistribute it and/or modify  *
// *   it under the terms of the GNU General Public License as published by  *
// *   the Free Software Foundation; either version 2 of the License, or     *
// *   (at your option) any later version.                                   *
// *                                                                         *
// *   This program is distributed in the hope that it will be useful,       *
// *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
// *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
// *   GNU General Public License for more details.                          *
// *                                                                         *
// *   You should have received a copy of the GNU General Public License     *
// *   along with this program; if not, write to the                         *
// *   Free Software Foundation, Inc.,                                       *
// *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
// ***************************************************************************

package SourceBeautifiers;

/**
 *
 * @author lutusp
 */
import Arachnophilia.*;
import java.io.*;
import java.util.*;
import java.util.regex.*;
import javax.swing.*;

/**
 *
 * @author lutusp
 */
final public class RubyBeautifier {

    boolean DEBUG = false;
    String tabStr = ArachComp.getTabString();
    Arachnophilia main;
    
    final String indentStr[] = {
        "^module\\b.*",
        "^class\\b.*",
        "^if\\b.*",
        ".*(=\\s*|^)until\\b.*",
        ".*(=\\s*|^)for\\b.*",
        "^unless\\b.*",
        ".*(=\\s*|^)while\\b.*",
        ".*(=\\s*|^)begin\\b.*",
        ".*(^| )case\\b.*",
        ".*\\bthen\\b.*",
        "^rescue\\b.*",
        "^def\\b.*",
        ".*\\bdo\\b.*",
        "^else\\b.*",
        "^elsif\\b.*",
        "^ensure\\b.*",
        ".*\\bwhen\\b.*",
        ".*\\{[^\\}]*$",
        ".*\\[[^\\]]*$"
    };
    final String outdentStr[] = {
        "^rescue\\b.*",
        "^ensure\\b.*",
        "^elsif\\b.*",
        "^end\\b.*",
        "^else\\b.*",
        ".*\\bwhen\\b.*",
        "^[^\\{]*\\}.*",
        "^[^\\[]*\\].*"
    };
    final String filterStr[] = {
        "\\{[^\\{]*?\\}",
        "\\[[^\\[]*?\\]",
        "'.*?'",
        "\".*?\"",
        "\\`.*?\\`",
        "\\([^\\(]*?\\)",
        "\\/.*?\\/",
        "%r(.).*?\\1"
    };
    ArrayList<Pattern> indentPat;
    ArrayList<Pattern> outdentPat;

    public RubyBeautifier(Arachnophilia m) {
        main = m;
        indentPat = new ArrayList<Pattern>();
        for (String s : indentStr) {
            indentPat.add(Pattern.compile(s));
        }
        outdentPat = new ArrayList<Pattern>();
        for (String s : outdentStr) {
            outdentPat.add(Pattern.compile(s));
        }
    }

    private String readFile(String path) {
        String out = "";
        try {
            File f = new File(path);
            char[] buff = new char[(int) f.length()];
            FileReader fin = new FileReader(f);
            fin.read(buff);
            fin.close();
            out = new String(buff);
        } catch (Exception e) {
            e.printStackTrace(System.out);
        }
        return out;
    }

    private void writeFile(String data, String path) {
        try {
            File out = new File(path);
            FileWriter fw = new FileWriter(out);
            fw.write(data);
            fw.close();
        } catch (Exception e) {
            e.printStackTrace(System.out);
        }
    }

    private int countMatches(String data, String pattern) {
        int count = 0;
        Pattern p = Pattern.compile(pattern);
        Matcher m = p.matcher(data);
        while (m.find()) {
            count++;
        }
        return count;
    }

    private String makeTab(int count) {
        String out = "";
        while (count-- > 0) {
            out += tabStr;
        }
        return out;
    }

    private String addLine(String line, int tab) {
        line = line.trim();
        if (line.length() > 0) {
            line = makeTab(tab) + line.trim();
        }
        return line + "\n";
    }

    public String beautify(String path, String data) {
        StringBuilder output = new StringBuilder();
        boolean commentBlock = false;
        boolean inHereDoc = false;
        boolean programEnd = false;
        boolean commentLine;
        String hereDocTerm = "";
        Stack<String> multiLineStack = new Stack<String>();
        String multiLineString = "";
        int tab = 0;
        String[] array = data.split("\n");
        for (String line : array) {
            String tline = "";
            line = line.trim();
            if (!programEnd) {
                // detect program end mark
                if (line.matches("^__END__$")) {
                    programEnd = true;
                } else {
                    // combine continuing lines
                    if (!line.matches("^\\s*#.*") && line.matches(".*[^\\\\]\\\\\\s*$")) {
                        multiLineStack.push(line);
                        multiLineString += line.replaceFirst("^(.*)\\\\\\s*$", "$1");
                    } else {
                        // add final line
                        if (multiLineString.length() > 0) {
                            multiLineStack.push(line);
                            multiLineString += line.replaceFirst("^(.*)\\\\\\s*$", "$1");
                        }
                    }
                    tline = (multiLineString.length() > 0) ? multiLineString : line;
                    if (tline.matches("^=begin.*")) {
                        commentBlock = true;
                    }
                    if (inHereDoc) {
                        if (tline.matches(".*" + hereDocTerm + ".*")) {
                            inHereDoc = false;
                        }
                    } else { // not in hereDoc
                        if (tline.matches(".*=\\s*<<.*")) {
                            hereDocTerm = tline.replaceFirst(".*=\\s*<<-?\\s*([_\\w]+).*", "$1");
                            inHereDoc = hereDocTerm.length() > 0;
                        }
                    }
                }
            }
            if (commentBlock || programEnd || inHereDoc) {
                // append trimmed but not indended
                output.append(line).append("\n");
            } else {
                commentLine = tline.matches("^#.*");
                if (!commentLine) {
                    // throw out sequences that will
                    // only sow confusion
                    for (String s : filterStr) {
                        String r = "";
                        while (!r.equals(tline)) {
                            r = tline;
                            tline = tline.replaceAll(s, "");
                        }
                    }
                    // delete end-of-line comments
                    tline = tline.replaceFirst("#[^\"]+$", "");
                    // convert quotes
                    tline = tline.replaceAll("\\\\\"", "'");
                    for (Pattern p : outdentPat) {
                        if (p.matcher(tline).matches()) {
                            tab -= 1;
                            break;
                        }
                    }
                }
                if (multiLineStack.size() > 0) {
                    for (String s : multiLineStack) {
                        output.append(addLine(s, tab));
                    }
                    multiLineStack.clear();
                    multiLineString = "";
                } else {
                    output.append(addLine(line, tab));
                }
                if (!commentLine) {
                    for (Pattern p : indentPat) {
                        if (p.matcher(tline).matches() && !tline.matches(".*\\s+end\\s*$")) {
                            tab += 1;
                            break;
                        }
                    }
                }
            }
            if (tline.matches("^=end.*")) {
                commentBlock = false;
            }
        }
        if (tab != 0) {
            JOptionPane.showMessageDialog(main, "Error in " + path + ": indent/outdent mismatch:" + tab, ArachConstants.APPNAME + " Code Beautifier", JOptionPane.INFORMATION_MESSAGE);

            // System.err.println("File:" + path + ", error: indent/outdent mismatch: " + tab);
        }
        return output.toString();
    }

}
