// ***************************************************************************
// *   Copyright (C) 2018 by Paul Lutus                                      *
// *   lutusp@arachnoid.com                                                  *
// *                                                                         *
// *   This program is free software; you can redistribute it and/or modify  *
// *   it under the terms of the GNU General Public License as published by  *
// *   the Free Software Foundation; either version 2 of the License, or     *
// *   (at your option) any later version.                                   *
// *                                                                         *
// *   This program is distributed in the hope that it will be useful,       *
// *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
// *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
// *   GNU General Public License for more details.                          *
// *                                                                         *
// *   You should have received a copy of the GNU General Public License     *
// *   along with this program; if not, write to the                         *
// *   Free Software Foundation, Inc.,                                       *
// *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
// ***************************************************************************

 package RTFConverter;

/*
 * RTFPanel.java
 *
 * Created on April 8, 2002, 6:44 PM
 */
import Arachnophilia.*;
import java.awt.*;
import java.io.*;
import javax.swing.*;
import javax.swing.text.*;
import javax.swing.text.rtf.*;

class HTMLStateMachine {

    String[] alignNames = {
        "left",
        "center",
        "right"
    };
    public boolean acceptFonts = true;
    private String fontName = "";
    private int alignment = -1;
    private boolean bold = false;
    private boolean italic = false;
    private boolean underline = false;
    private Color color = null;
    private int size = -1;
    private double firstLineIndent = 0;
    private double oldLeftIndent = 0;
    private double oldRightIndent = 0;
    private double leftIndent = 0;
    private double rightIndent = 0;
    private boolean firstLine = false;

    public void updateState(AttributeSet as, StringBuilder sb, Element el) {
        String name = el.getName();
        if (name.equalsIgnoreCase("paragraph")) {
            firstLine = true;
        }
        leftIndent = updateDouble(as, leftIndent, StyleConstants.LeftIndent);
        rightIndent = updateDouble(as, rightIndent, StyleConstants.RightIndent);
        if (leftIndent != oldLeftIndent || rightIndent != oldRightIndent) {
            // close old table
            closeIndentTable(sb, oldLeftIndent, oldRightIndent);
        }
        bold = updateBoolean(as, StyleConstants.Bold, "b", bold, sb);
        italic = updateBoolean(as, StyleConstants.Italic, "i", italic, sb);
        underline = updateBoolean(as, StyleConstants.Underline, "u", underline, sb);
        size = updateFontSize(as, size, sb);
        color = updateFontColor(as, color, sb);
        if (acceptFonts) {
            fontName = updateFontName(as, fontName, sb);
        }
        alignment = updateAlignment(as, alignment, sb);
        firstLineIndent = updateDouble(as, firstLineIndent, StyleConstants.FirstLineIndent);

        if (leftIndent != oldLeftIndent || rightIndent != oldRightIndent) {
            // open new table
            openIndentTable(sb, leftIndent, rightIndent);
            oldLeftIndent = leftIndent;
            oldRightIndent = rightIndent;
        }
    }

    private void openIndentTable(StringBuilder sb, double leftIndent, double rightIndent) {
        if (leftIndent != 0 || rightIndent != 0) {
            closeSubsetTags(sb);
            sb.append("<table><tr>");
            String leftTab = getSpaceTab((int) (leftIndent / 4));
            if (leftTab.length() > 0) {
                sb.append("<td>").append(leftTab).append("</td>");
            }
            sb.append("<td>");
        }
    }

    private void closeIndentTable(StringBuilder sb, double leftIndent, double rightIndent) {
        if (leftIndent != 0 || rightIndent != 0) {
            closeSubsetTags(sb);
            sb.append("</td>");
            String rightTab = getSpaceTab((int) (rightIndent / 4));
            if (rightTab.length() > 0) {
                sb.append("<td>").append(rightTab).append("</td>");
            }
            sb.append("</tr></table>");
        }
    }

    public void closeTags(StringBuilder sb) {
        closeSubsetTags(sb);
        closeTag(alignment, -1, "div", sb);
        alignment = -1;
        closeIndentTable(sb, oldLeftIndent, oldRightIndent);
    }

    private void closeSubsetTags(StringBuilder sb) {
        closeTag(bold, "b", sb);
        closeTag(italic, "i", sb);
        closeTag(underline, "u", sb);
        closeTag(color, "font", sb);
        closeTag(fontName, "font", sb);
        closeTag(size, -1, "font", sb);
        bold = false;
        italic = false;
        underline = false;
        color = null;
        fontName = "";
        size = -1;
    }

    private void closeTag(boolean v, String tag, StringBuilder sb) {
        if (v) {
            sb.append("</").append(tag).append(">");
        }
    }

    private void closeTag(Color c, String tag, StringBuilder sb) {
        if (c != null) {
            sb.append("</").append(tag).append(">");
        }
    }

    private void closeTag(String s, String tag, StringBuilder sb) {
        if (s.length() > 0) {
            sb.append("</").append(tag).append(">");
        }
    }

    private void closeTag(int v, int limit, String tag, StringBuilder sb) {
        if (v > limit) {
            sb.append("</").append(tag).append(">");
        }
    }

    private int updateAlignment(AttributeSet as, int oldAlignment, StringBuilder sb) {
        Object o = as.getAttribute(StyleConstants.Alignment);
        if (o != null) {

            int newAlignment = ((Integer) o).intValue();
            // can't do justified
            if (newAlignment == 3) {
                newAlignment = 0;
            }
            if (newAlignment != oldAlignment) {
                //System.out.println(oldAlignment + "," + newAlignment);
                if (newAlignment >= 0 && newAlignment <= 2) {
                    if (oldAlignment > -1) {
                        sb.append("</div>");
                    }
                    sb.append("<div align=\"").append(alignNames[newAlignment]).append("\">");
                    oldAlignment = newAlignment;
                }
            }
        }
        return oldAlignment;
    }

    private Color updateFontColor(AttributeSet as, Color oldCol, StringBuilder sb) {
        Object o = as.getAttribute(StyleConstants.Foreground);
        if (o != null) {
            Color newCol = (Color) o;
            if (newCol != oldCol) {
                if (oldCol != null) {
                    sb.append("</font>");
                }
                if (newCol != null) {
                    sb.append("<font color=\"#").append(makeColorString(newCol)).append("\">");
                }
            }
            oldCol = newCol;
        }
        return oldCol;
    }

    private String updateFontName(AttributeSet as, String oldFontName, StringBuilder sb) {
        Object o = as.getAttribute(StyleConstants.FontFamily);
        if (o != null) {
            String newFontName = (String) o;
            if (!newFontName.equals(oldFontName)) {
                if (!oldFontName.equals("")) {
                    sb.append("</font>");
                }
                sb.append("<font face=\"").append(newFontName).append("\">");
            }
            oldFontName = newFontName;
        }
        return oldFontName;
    }

    private double updateDouble(AttributeSet as, double oldVal, Object key) {
        Object o = as.getAttribute(key);
        if (o != null) {
            oldVal = (double) ((Float) o).floatValue();
        }
        return oldVal;
    }

    private int updateFontSize(AttributeSet as, int oldSize, StringBuilder sb) {
        Object o = as.getAttribute(StyleConstants.FontSize);
        if (o != null) {
            int newSize = ((Integer) o).intValue();
            if (newSize != oldSize) {
                if (oldSize != -1) {
                    sb.append("</font>");
                }
                sb.append("<font size=\"").append(newSize / 4).append("\">");
            }
            oldSize = newSize;
        }
        return oldSize;
    }

    private boolean updateBoolean(
            AttributeSet as, Object key, String tag, boolean oldVal, StringBuilder sb) {
        //System.out.println("looking for [" + key + "]");
        Object o = as.getAttribute(key);
        if (o != null) {
            //System.out.println(key + " not null!");
            boolean newVal = ((Boolean) o).booleanValue();
            if (newVal != oldVal) {
                if (newVal) {
                    sb.append("<").append(tag).append(">");
                } else {
                    sb.append("</").append(tag).append(">");
                }
            }
            oldVal = newVal;
        }
        return oldVal;
    }

    private String makeColorString(Color c) {
        String result = Long.toString(c.getRGB() & 0xffffff, 16);
        if (result.length() < 6) {
            StringBuilder sb = new StringBuilder();
            for (int i = result.length(); i < 6; i++) {
                sb.append("0");
            }
            sb.append(result);
            result = sb.toString();
        }
        return result;
    }

    public String performFirstLineIndent(String s) {
        if (firstLine) {
            if (firstLineIndent != 0.0) {
                int n = (int) (firstLineIndent / 4);
                s = getSpaceTab(n) + s;
            }
            firstLine = false;
        }
        return s;
    }

    public String getSpaceTab(int n) {
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < n; i++) {
            sb.append("&nbsp;");
        }
        return sb.toString();
    }
}

/**
 *
 * @author  Administrator
 */
public class RTFConverter {

    public void convertRTFToHTML(Arachnophilia main) {
        ArachDocument doc = main.currentSelectedDocument;
        if (doc != null) {
            int rtfType = main.fileTypes.getFileTypeForName("RTF");
            if (doc.getFileType() == rtfType) {
                HTMLStateMachine state = new HTMLStateMachine();
                String data = doc.textComp.getText();
                data = data.replaceAll("&", "&amp;");
                data = data.replaceAll("<", "&lt;");
                data = data.replaceAll(">", "&gt;");
                data = convertRTFStringToHTML(data, state);
                main.createHTMLDocForRTFConversion(data);
            } else {
                JOptionPane.showMessageDialog(main, "The source for this conversion must be an RTF-formatted document.", "Source not RTF Document", JOptionPane.WARNING_MESSAGE);
            }
        }
    }

    private String convertRTFStringToHTML(String data, HTMLStateMachine state) {
        RTFEditorKit kit = new RTFEditorKit();
        DefaultStyledDocument doc = new DefaultStyledDocument();
        readString(data, doc, kit);
        data = scanDocument(doc, state);
        return data;
    }

    private void readString(String data, Document doc, RTFEditorKit rtf) {

        try {
            // ByteArrayInputStream ba = new ByteArrayInputStream(data.getBytes());
            // rtf.read( ba, doc,0);
            rtf.read(new StringReader(data), doc, 0);
        } catch (Exception e) {
            e.printStackTrace(System.out);
        }

    }

    private String scanDocument(Document doc, HTMLStateMachine state) {
        String out = "";
        try {
            StringBuilder sb = new StringBuilder();
            Element def = doc.getDefaultRootElement();
            recurseElements(def, doc, sb, state);
            state.closeTags(sb);
            out = sb.toString();
        } catch (Exception e) {
            e.printStackTrace(System.out);
        }
        return out;
    }

    private void recurseElements(Element parent, Document doc, StringBuilder sb, HTMLStateMachine state) {
        for (int i = 0; i < parent.getElementCount(); i++) {
            Element child = parent.getElement(i);
            scanAttributes(child, doc, sb, state);
            recurseElements(child, doc, sb, state);
        }
    }

    private void scanAttributes(Element el, Document doc, StringBuilder sb, HTMLStateMachine state) {
        try {
            int a = el.getStartOffset();
            int b = el.getEndOffset();
            String content = doc.getText(a, b - a);
            AttributeSet as = el.getAttributes();
            state.updateState(as, sb, el);
            String name = el.getName();
            if (name.equalsIgnoreCase("content")) {
                StringBuilder temp = new StringBuilder();
                int i = 0;
                while (i < content.length()) {
                    int c = content.charAt(i);
                    if (c == 1027 && i < content.length() - 1) {
                        int d = content.charAt(++i);
                        switch (d) {
                            case 101:
                                temp.append("&lsquo;");
                                break;
                            case 102:
                                temp.append("&rsquo;");
                                break;
                            case 103:
                                temp.append("&ldquo;");
                                break;
                            case 104:
                                temp.append("&rdquo;");
                                break;
                            default:
                                temp.append((char) c);
                                temp.append((char) d);
                        }
                    } else {
                        temp.append((char) c);
                    }
                    i++;
                }
                content = temp.toString();
                content = content.replaceAll("\\t", state.getSpaceTab(8));
                content = content.replaceAll("\\n", "<br/>\n");
                content = state.performFirstLineIndent(content);
                sb.append(content);
            }
        // debugging section
            /*
        //System.out.println("------------ " + el.getName() + " = \"" + content + "\"");
        Enumeration e = as.getAttributeNames();
        while(e.hasMoreElements()) {
        Object o = e.nextElement();
        Object at = as.getAttribute(o);
        //System.out.println(o + " = " + at.toString());
        }*/
        } catch (BadLocationException e) {
            e.printStackTrace(System.out);
        }
    }
}
