// ***************************************************************************
// *   Copyright (C) 2018 by Paul Lutus                                      *
// *   lutusp@arachnoid.com                                                  *
// *                                                                         *
// *   This program is free software; you can redistribute it and/or modify  *
// *   it under the terms of the GNU General Public License as published by  *
// *   the Free Software Foundation; either version 2 of the License, or     *
// *   (at your option) any later version.                                   *
// *                                                                         *
// *   This program is distributed in the hope that it will be useful,       *
// *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
// *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
// *   GNU General Public License for more details.                          *
// *                                                                         *
// *   You should have received a copy of the GNU General Public License     *
// *   along with this program; if not, write to the                         *
// *   Free Software Foundation, Inc.,                                       *
// *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
// ***************************************************************************

 /*
 * TableWizardPanel.java
 *
 * Created on February 24, 2002, 10:11 AM
 */

package ListTableWizards;

import Arachnophilia.*;
import java.awt.*;
import java.util.*;
import javax.swing.*;

/**
 *
 * @author  Administrator
 */
final public class TableWizardPanel extends javax.swing.JPanel {
    
    Arachnophilia main;
    TableWizardFrame parent;
    DocContentHandler docHandler = null;
    boolean bareTableMode = false;
    int tableRows = -1,tableColumns = -1;
    ArrayList<String> rowList = null;
    String delimiter = null;
    String docContent = null;
    TableWizData wizData = null;
    
    final String[] delimiters = {",","\t",";",":","/","|",".","+"," ","\0"};
    final String[] delimText = {",","[Tab]",";",":","/","|",".","+","[Space]","[None]"};
    
    /** Creates new form TableWizardPanel */
    public TableWizardPanel(Arachnophilia m,TableWizardFrame p) {
        main = m;
        parent = p;
        initComponents();
        loadControls();
    }
    
    public void launch() {
        setVisible(true);
        analyzeDocSelection(true);
    }
    
    public void analyzeDocSelection(boolean findDelim) {
        docHandler = new DocContentHandler(main);
        //int rows = 0,columns = 0;
        rowList = null;
        delimiter = null;
        docContent = docHandler.getContent();
        bareTableMode = !docHandler.textSelected();
        wizData = new TableWizData();
        retrieveControls(wizData);
        if(bareTableMode) {
            tableRows = readTextField(rowsField,0);
            tableColumns = readTextField(columnsField,0);
            
            scanResultsArea.setText("No document selection, using defaults of " + tableRows + " rows and " + tableColumns + " columns.");
        }
        else {
            rowList = ArachComp.parseDelimLine(docContent,"\n",true);
            delimiter = getSelectedDelim();
            if(findDelim || delimiter == null) {
                String testContent = ArachComp.mergeDelimLine(rowList,"\n");
                delimiter = findMostLikelyDelim(testContent);
                if(delimiter == null) {
                   main.beep();
                    JOptionPane.showMessageDialog(main
                    ,"Cannot find any delimiters. Please format\n"
                    +"your text as a table, with field delimiters\n"
                    + "between fields, and try again."
                    ,"Searching for delimiters"
                    ,JOptionPane.OK_OPTION);
                }
                else {
                    setDataDelimiterComboBox(dataDelimiterComboBox,delimiter);
                }
            }
            if(delimiter != null) {
                tableRows = rowList.size();
                setTextField(rowsField,tableRows);
                tableColumns = findMaxColumns(rowList,delimiter);
                setTextField(columnsField,tableColumns);
            }
            scanResultsArea.setText("Scanned document text, " + ((findDelim)?"detected":"using") + " delimiter of \"" + hashDelims(delimiter,false) + "\" and parsed table of " + tableRows + " rows and " + tableColumns + " columns.");
        } // not bare table mode
        
    }
    
    private void executeBuild() {
        docHandler = new DocContentHandler(main);
        // must make this call
        docContent = docHandler.getContent();
        bareTableMode = !docHandler.textSelected();
        if(bareTableMode) {
            docContent = "";
        }
        wizData = new TableWizData();
        retrieveControls(wizData);
        rowList = null;
        if(!bareTableMode) {
            rowList = ArachComp.parseDelimLine(docContent,"\n",true);
        }
        delimiter = getSelectedDelim();
        if(delimiter != null) {
            String newContent = buildTable(tableRows,tableColumns,rowList,delimiter,wizData);
            if(newContent != null && newContent.length() > 0) {
                newContent = main.beautifyUtils.beautifyHTML(newContent,false);
                // insert the result if no selection
                //System.out.println("[" + newContent + "]");
                docHandler.setContent(newContent,true);
            }
        }
        else {
           main.beep();
        }
        
    }
    
    private String buildTable(int rows, int columns,ArrayList<String> localRowList,String delimiter,TableWizData wizData) {
        StringBuilder sb = new StringBuilder();
        int rowVecLen = (localRowList != null)?localRowList.size():0;
        for(int row = 0;row < rows;row++) {
            ArrayList<String> fields = null;
            if(localRowList != null && row < rowVecLen) {
                String line = localRowList.get(row);
                fields = ArachComp.parseDelimLine(line,delimiter,true);
            }
            StringBuilder sbr = new StringBuilder();
            for(int column = 0;column < columns;column++) {
                String cell = "";
                if(fields != null ) { // there's data
                    if( column < fields.size()) {
                        cell = fields.get(column);
                    }
                    else { // but this cell has none :(
                        cell = "&nbsp;";
                    }
                }
                else if(wizData.placeholderData) {
                    cell = "R"+(row+1)+"C"+(column+1);
                }
                cell += "\n";
                if(wizData.commentMarkers) {
                    sbr.append("<!-- Row ").append(row+1).append(" Column ").append(column+1).append(" -->\n");
                }
                String mods = (wizData.centerData)?"align=\"center\"":"";
                if((column == 0 && wizData.titlecolorFirstColumn) || (row == 0 && wizData.titlecolorFirstRow)) {
                    mods += " " + "bgcolor=\"#" + ArachComp.colorIntToString(wizData.titleRowColor) + "\"";
                }
                if((column == 0 && wizData.boldFirstColumn) || (row == 0 && wizData.boldFirstRow)) {
                    cell = wrapTag("b",cell);
                }
                sbr.append(wrapTag("td",cell,mods));
            }
            sb.append(wrapTag("tr",sbr.toString()));
        }
        String tableMods = "cellspacing=\"" + wizData.cellSpacing + "\"";
        tableMods += " cellpadding=\"" + wizData.cellPadding + "\"";
        tableMods += " border=\"" + wizData.borderWidth + "\"";
        if(wizData.tableWidth > 0) {
            tableMods += " width=\"" + wizData.tableWidth + "%\"";
        }
        if(wizData.colorBackground) {
            tableMods += " bgColor=\"#" + ArachComp.colorIntToString(wizData.backgroundColor) + "\"";
        }
        if(wizData.borderWidth > 0) {
            if(wizData.colorBorder) {
                tableMods += " borderColor=\"#" + ArachComp.colorIntToString(wizData.borderColor) + "\"";
            }
        }
        
        String result = wrapTag("table",sb.toString(),tableMods);
        return result;
    }
    
    private String wrapTag(String tag,String v) {
        return wrapTag(tag,v,"");
    }
    
    private String wrapTag(String tag,String v,String mods) {
        if(mods.length() > 0) {
            mods = " " + mods;
        }
        String result = "<" + tag
        + mods + ">\n"
        + v
        + "</" + tag + ">\n";
        return result;
    }
    
    private int findMaxColumns(ArrayList<String> localRowList,String delim) {
        int i = 0;
        int max = 0;
        int len = localRowList.size();
        while(i < len) {
            ArrayList<String> v = ArachComp.parseDelimLine(localRowList.get(i),delim);
            int n = v.size();
            max = (max < n)?n:max;
            i++;
        }
        return max;
    }
    
    private String findMostLikelyDelim(String content) {
        String ml = null;
        int max = 0;
        int i = 0;
        int len = delimiters.length;
        while(i < len) {
            String s = delimiters[i];
            if(!s.equals("\0")) {
                int n = countOccurrences(content,s);
                if(max < n) {
                    ml = s;
                    max = n;
                }
            }
            i++;
        }
        return ml;
    }
    
    private int countOccurrences(String content,String s) {
        int n = 0;
        int p = 0;
        while((p = content.indexOf(s,p)) != -1) {
            n++;
            p += s.length();
        }
        return n;
    }
    
    private String getSelectedDelim() {
        String v = (String)dataDelimiterComboBox.getSelectedItem();
        if(v != null) {
            v = hashDelims(v,true);
        }
        return v;
    }
    
    private void setTitleColor() {
        setChipColor(titleColorChip,"table title row/column");
        parent.requestFocusInWindow();
    }
    
    private void setBackgroundColor() {
        setChipColor(backgroundColorChip,"table background");
        parent.requestFocusInWindow();
    }
    
    private void setBorderColor() {
        setChipColor(borderColorChip,"table border");
        parent.requestFocusInWindow();
    }
    
    private void setChipColor(JButton b,String title) {
        Color col = b.getBackground();
        col = JColorChooser.showDialog(main,"Choose a color for the " + title,col);
        if(col != null) {
            b.setBackground(col);
        }
    }
    
    public void quit() {
        TableWizData data = new TableWizData();
        retrieveControls(data);
        // don't save the rows and columns
        // from an automatic table generation
        if(bareTableMode) {
            data.defaultColumns = main.configValues.tableWizData.defaultColumns;
            data.defaultRows = main.configValues.tableWizData.defaultRows;
        }
        main.configValues.tableWizData = data;
    }
    
    private void loadControls() {
        TableWizData d = main.configValues.tableWizData;
        setTextField(rowsField,d.defaultRows);
        setTextField(columnsField,d.defaultColumns);
        setTextField(tableWidthField,d.tableWidth);
        setTextField(borderWidthField,d.borderWidth);
        setTextField(cellPaddingField,d.cellPadding);
        setTextField(cellSpacingField,d.cellSpacing);
        setDataDelimiterComboBox(dataDelimiterComboBox,d.dataDelimiter);
        setCheckBox(commentMarkersCheckbox,d.commentMarkers);
        setCheckBox(centerDataCheckbox,d.centerData);
        setCheckBox(placeHolderCheckBox,d.placeholderData);
        setCheckBox(boldFirstRowCheckBox,d.boldFirstRow);
        setCheckBox(titleColorRowCheckBox,d.titlecolorFirstRow);
        setCheckBox(boldFirstColumnCheckBox,d.boldFirstColumn);
        setCheckBox(titleColorColumnCheckbox,d.titlecolorFirstColumn);
        setCheckBox(colorBackgroundCheckBox,d.colorBackground);
        setCheckBox(colorBorderCheckBox,d.colorBorder);
        borderColorChip.setBackground(new Color(d.borderColor));
        backgroundColorChip.setBackground(new Color(d.backgroundColor));
        titleColorChip.setBackground(new Color(d.titleRowColor));
    }
    
    private void retrieveControls(TableWizData d) {
        d.defaultRows = readTextField(rowsField,d.defaultRows);
        d.defaultColumns = readTextField(columnsField,d.defaultColumns);
        d.tableWidth = readTextField(tableWidthField,d.tableWidth);
        d.borderWidth = readTextField(borderWidthField,d.borderWidth);
        d.cellPadding = readTextField(cellPaddingField,d.cellPadding);
        d.cellSpacing = readTextField(cellSpacingField,d.cellSpacing);
        d.dataDelimiter = getDataDelimiterComboBox(dataDelimiterComboBox,d.dataDelimiter);
        d.commentMarkers = readCheckBox(commentMarkersCheckbox);
        d.centerData = readCheckBox(centerDataCheckbox);
        d.placeholderData = readCheckBox(placeHolderCheckBox);
        d.boldFirstRow = readCheckBox(boldFirstRowCheckBox);
        d.titlecolorFirstRow = readCheckBox(titleColorRowCheckBox);
        d.boldFirstColumn = readCheckBox(boldFirstColumnCheckBox);
        d.titlecolorFirstColumn = readCheckBox(titleColorColumnCheckbox);
        d.colorBackground = readCheckBox(colorBackgroundCheckBox);
        d.colorBorder = readCheckBox(colorBorderCheckBox);
        d.borderColor = borderColorChip.getBackground().getRGB();
        d.backgroundColor = backgroundColorChip.getBackground().getRGB();
        d.titleRowColor = titleColorChip.getBackground().getRGB();
    }
    
    private boolean readCheckBox(JCheckBox f) {
        return f.getSelectedObjects() != null;
    }
    
    private int readTextField(JTextField f,int v) {
        try {
            v = Integer.parseInt(f.getText());
        }
        catch (Exception e) {
        }
        return v;
    }
    
    private int readTextField(JTextField f,char v) {
        String sv = f.getText();
        if(sv.length() > 0) {
            v = sv.charAt(0);
        }
        return (int) v;
    }
    
    private int getDataDelimiterComboBox(JComboBox f,int defVal) {
        String v = (String)f.getSelectedItem();
        if(v != null) {
            v = hashDelims(v,true);
            if(v != null) {
                defVal = v.charAt(0);
            }
        }
        return defVal;
    }
    
    private void setDataDelimiterComboBox(JComboBox f,int v) {
        String sv = hashDelims(""+(char)v,false);
        if(sv != null) {
            f.setSelectedItem(sv);
        }
    }
    
    private void setDataDelimiterComboBox(JComboBox f,String v) {
        String sv = hashDelims(v,false);
        if(sv != null) {
            f.setSelectedItem(sv);
        }
    }
    
    private String hashDelims(String v,boolean reverse) {
        if(reverse) {
            v = hashDelimArrays(v,delimText,delimiters);
        }
        else {
            v = hashDelimArrays(v,delimiters,delimText);
        }
        return v;
    }
    
    private String hashDelimArrays(String v,String[] a,String[] b) {
        int i = 0;
        int len = a.length;
        while(i < len && !v.equals(a[i])) {
            i++;
        }
        if(i < len) {
            v = b[i];
        }
        else {
            v = null;
        }
        return v;
    }
    
    private void setTextField(JTextField f,int v) {
        f.setText("" + v);
    }
    
    private void setTextField(JTextField f,char v) {
        f.setText("" + v);
    }
    
    private void setCheckBox(JCheckBox f,boolean v) {
        f.setSelected(v);
    }
    
    private void performUndo() {
        if(main.currentSelectedDocument != null) {
            main.currentSelectedDocument.undo();
        }
    }
    
    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {
        java.awt.GridBagConstraints gridBagConstraints;

        middlePanel = new javax.swing.JPanel();
        jLabel1 = new javax.swing.JLabel();
        jLabel2 = new javax.swing.JLabel();
        jLabel3 = new javax.swing.JLabel();
        jLabel6 = new javax.swing.JLabel();
        jLabel7 = new javax.swing.JLabel();
        jLabel8 = new javax.swing.JLabel();
        jLabel9 = new javax.swing.JLabel();
        rowsField = new javax.swing.JTextField();
        columnsField = new javax.swing.JTextField();
        tableWidthField = new javax.swing.JTextField();
        borderWidthField = new javax.swing.JTextField();
        cellPaddingField = new javax.swing.JTextField();
        cellSpacingField = new javax.swing.JTextField();
        scanResultPanel = new javax.swing.JPanel();
        jScrollPane1 = new javax.swing.JScrollPane();
        scanResultsArea = new javax.swing.JTextArea();
        dataDelimiterComboBox = new JComboBox<String>(delimText);
        bottomPanel = new javax.swing.JPanel();
        colorSelectPanel = new javax.swing.JPanel();
        backgroundColorButton = new MyJButton();
        borderColorButton = new MyJButton();
        titleColorButton = new MyJButton();
        backgroundColorChip = new javax.swing.JButton();
        borderColorChip = new javax.swing.JButton();
        titleColorChip = new javax.swing.JButton();
        rescanButton = new MyJButton();
        doneButton = new MyJButton();
        undoButton = new javax.swing.JButton();
        createButton = new javax.swing.JButton();
        eastpanel = new javax.swing.JPanel();
        commentMarkersCheckbox = new javax.swing.JCheckBox();
        centerDataCheckbox = new javax.swing.JCheckBox();
        placeHolderCheckBox = new javax.swing.JCheckBox();
        boldFirstRowCheckBox = new javax.swing.JCheckBox();
        titleColorRowCheckBox = new javax.swing.JCheckBox();
        boldFirstColumnCheckBox = new javax.swing.JCheckBox();
        titleColorColumnCheckbox = new javax.swing.JCheckBox();
        colorBackgroundCheckBox = new javax.swing.JCheckBox();
        colorBorderCheckBox = new javax.swing.JCheckBox();

        setLayout(new java.awt.BorderLayout());

        middlePanel.setBorder(javax.swing.BorderFactory.createTitledBorder(javax.swing.BorderFactory.createEtchedBorder(), "Table Characteristics"));
        middlePanel.setLayout(new java.awt.GridBagLayout());

        jLabel1.setText("Rows");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 0, 8);
        middlePanel.add(jLabel1, gridBagConstraints);

        jLabel2.setText("Columns");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 0, 8);
        middlePanel.add(jLabel2, gridBagConstraints);

        jLabel3.setText("Table Width %");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 0, 8);
        middlePanel.add(jLabel3, gridBagConstraints);

        jLabel6.setText("Border Width Pixels");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 4;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 0, 8);
        middlePanel.add(jLabel6, gridBagConstraints);

        jLabel7.setText("Cell Padding Pixels");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 5;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 0, 8);
        middlePanel.add(jLabel7, gridBagConstraints);

        jLabel8.setText("Cell Spacing Pixels");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 6;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 0, 8);
        middlePanel.add(jLabel8, gridBagConstraints);

        jLabel9.setText("Data Delimiter");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 7;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.weighty = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 0, 8);
        middlePanel.add(jLabel9, gridBagConstraints);

        rowsField.setPreferredSize(new java.awt.Dimension(40, 21));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(0, 0, 0, 4);
        middlePanel.add(rowsField, gridBagConstraints);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.insets = new java.awt.Insets(0, 0, 0, 4);
        middlePanel.add(columnsField, gridBagConstraints);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.insets = new java.awt.Insets(0, 0, 0, 4);
        middlePanel.add(tableWidthField, gridBagConstraints);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 4;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.insets = new java.awt.Insets(0, 0, 0, 4);
        middlePanel.add(borderWidthField, gridBagConstraints);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 5;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.insets = new java.awt.Insets(0, 0, 0, 4);
        middlePanel.add(cellPaddingField, gridBagConstraints);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 6;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.insets = new java.awt.Insets(0, 0, 0, 4);
        middlePanel.add(cellSpacingField, gridBagConstraints);

        scanResultPanel.setBorder(javax.swing.BorderFactory.createTitledBorder(javax.swing.BorderFactory.createEtchedBorder(), "Scan Results"));
        scanResultPanel.setLayout(new java.awt.BorderLayout());

        jScrollPane1.setMinimumSize(new java.awt.Dimension(70, 70));
        jScrollPane1.setPreferredSize(new java.awt.Dimension(70, 70));

        scanResultsArea.setEditable(false);
        scanResultsArea.setLineWrap(true);
        scanResultsArea.setWrapStyleWord(true);
        jScrollPane1.setViewportView(scanResultsArea);

        scanResultPanel.add(jScrollPane1, java.awt.BorderLayout.CENTER);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTH;
        gridBagConstraints.insets = new java.awt.Insets(4, 4, 4, 4);
        middlePanel.add(scanResultPanel, gridBagConstraints);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 7;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTH;
        gridBagConstraints.insets = new java.awt.Insets(0, 0, 0, 4);
        middlePanel.add(dataDelimiterComboBox, gridBagConstraints);

        add(middlePanel, java.awt.BorderLayout.CENTER);

        bottomPanel.setLayout(new java.awt.GridBagLayout());

        colorSelectPanel.setBorder(javax.swing.BorderFactory.createTitledBorder(javax.swing.BorderFactory.createEtchedBorder(), "Choose Colors"));
        colorSelectPanel.setLayout(new java.awt.GridBagLayout());

        backgroundColorButton.setText("Background");
        backgroundColorButton.setToolTipText("Choose a color for the table's background");
        backgroundColorButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                backgroundColorButtonActionPerformed(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 3;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(5, 5, 5, 5);
        colorSelectPanel.add(backgroundColorButton, gridBagConstraints);

        borderColorButton.setText("Border");
        borderColorButton.setToolTipText("Choose a color for the table's border");
        borderColorButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                borderColorButtonActionPerformed(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(5, 5, 5, 5);
        colorSelectPanel.add(borderColorButton, gridBagConstraints);

        titleColorButton.setText("Title");
        titleColorButton.setToolTipText("Choose a color for the title regions");
        titleColorButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                titleColorButtonActionPerformed(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 5;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(5, 5, 5, 5);
        colorSelectPanel.add(titleColorButton, gridBagConstraints);

        backgroundColorChip.setBackground(new java.awt.Color(255, 51, 102));
        backgroundColorChip.setText(" ");
        backgroundColorChip.setToolTipText("Sample of background color");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.insets = new java.awt.Insets(5, 5, 5, 5);
        colorSelectPanel.add(backgroundColorChip, gridBagConstraints);

        borderColorChip.setBackground(new java.awt.Color(255, 51, 102));
        borderColorChip.setText(" ");
        borderColorChip.setToolTipText("Sample of border color");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.insets = new java.awt.Insets(5, 5, 5, 5);
        colorSelectPanel.add(borderColorChip, gridBagConstraints);

        titleColorChip.setBackground(new java.awt.Color(255, 51, 102));
        titleColorChip.setText(" ");
        titleColorChip.setToolTipText("Sample of title color");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 4;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.insets = new java.awt.Insets(5, 5, 5, 5);
        colorSelectPanel.add(titleColorChip, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridwidth = 4;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        bottomPanel.add(colorSelectPanel, gridBagConstraints);

        rescanButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/Icons/Options.gif"))); // NOI18N
        rescanButton.setText("Rescan");
        rescanButton.setToolTipText("Re-evaluate the document selection using your specifications");
        rescanButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                rescanButtonActionPerformed(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(5, 5, 5, 5);
        bottomPanel.add(rescanButton, gridBagConstraints);

        doneButton.setText("Done");
        doneButton.setToolTipText("Close this dialog");
        doneButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                doneButtonActionPerformed(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 3;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(5, 5, 5, 5);
        bottomPanel.add(doneButton, gridBagConstraints);

        undoButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/Icons/Undo.gif"))); // NOI18N
        undoButton.setText("Undo");
        undoButton.setToolTipText("Undo table creation");
        undoButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                undoButtonActionPerformed(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(5, 5, 5, 5);
        bottomPanel.add(undoButton, gridBagConstraints);

        createButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/Icons/NewBCard.gif"))); // NOI18N
        createButton.setText("Create");
        createButton.setToolTipText("Undo table creation");
        createButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                createButtonActionPerformed(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(5, 5, 5, 5);
        bottomPanel.add(createButton, gridBagConstraints);

        add(bottomPanel, java.awt.BorderLayout.SOUTH);

        eastpanel.setBorder(javax.swing.BorderFactory.createTitledBorder(javax.swing.BorderFactory.createEtchedBorder(), "Options"));
        eastpanel.setLayout(new java.awt.GridBagLayout());

        commentMarkersCheckbox.setText("Comment markers");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 0, 0);
        eastpanel.add(commentMarkersCheckbox, gridBagConstraints);

        centerDataCheckbox.setText("Center Data");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 0, 0);
        eastpanel.add(centerDataCheckbox, gridBagConstraints);

        placeHolderCheckBox.setText("Placeholder Data");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 0, 0);
        eastpanel.add(placeHolderCheckBox, gridBagConstraints);

        boldFirstRowCheckBox.setText("Bold First Row");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 0, 0);
        eastpanel.add(boldFirstRowCheckBox, gridBagConstraints);

        titleColorRowCheckBox.setText("Titlecolor First Row");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 4;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 0, 0);
        eastpanel.add(titleColorRowCheckBox, gridBagConstraints);

        boldFirstColumnCheckBox.setText("Bold First Column");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 5;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 0, 0);
        eastpanel.add(boldFirstColumnCheckBox, gridBagConstraints);

        titleColorColumnCheckbox.setText("Titlecolor First Column");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 6;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 0, 0);
        eastpanel.add(titleColorColumnCheckbox, gridBagConstraints);

        colorBackgroundCheckBox.setText("Color Background");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 7;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 0, 0);
        eastpanel.add(colorBackgroundCheckBox, gridBagConstraints);

        colorBorderCheckBox.setText("Color Border");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 8;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.weighty = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 0, 0);
        eastpanel.add(colorBorderCheckBox, gridBagConstraints);

        add(eastpanel, java.awt.BorderLayout.EAST);
    }// </editor-fold>//GEN-END:initComponents
    
    private void undoButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_undoButtonActionPerformed
        // Add your handling code here:
        performUndo();
    }//GEN-LAST:event_undoButtonActionPerformed
        
    private void rescanButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_rescanButtonActionPerformed
        // Add your handling code here:
        analyzeDocSelection(false);
    }//GEN-LAST:event_rescanButtonActionPerformed
    
    private void titleColorButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_titleColorButtonActionPerformed
        // Add your handling code here:
        setTitleColor();
    }//GEN-LAST:event_titleColorButtonActionPerformed
    
    private void backgroundColorButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_backgroundColorButtonActionPerformed
        // Add your handling code here:
        setBackgroundColor();
    }//GEN-LAST:event_backgroundColorButtonActionPerformed
    
    private void borderColorButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_borderColorButtonActionPerformed
        // Add your handling code here:
        setBorderColor();
    }//GEN-LAST:event_borderColorButtonActionPerformed
    
    private void doneButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_doneButtonActionPerformed
        // Add your handling code here:
        parent.quit();
    }//GEN-LAST:event_doneButtonActionPerformed

    private void createButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_createButtonActionPerformed
        // TODO add your handling code here:
        executeBuild();
    }//GEN-LAST:event_createButtonActionPerformed
    
    
    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JButton backgroundColorButton;
    private javax.swing.JButton backgroundColorChip;
    private javax.swing.JCheckBox boldFirstColumnCheckBox;
    private javax.swing.JCheckBox boldFirstRowCheckBox;
    private javax.swing.JButton borderColorButton;
    private javax.swing.JButton borderColorChip;
    private javax.swing.JTextField borderWidthField;
    private javax.swing.JPanel bottomPanel;
    private javax.swing.JTextField cellPaddingField;
    private javax.swing.JTextField cellSpacingField;
    private javax.swing.JCheckBox centerDataCheckbox;
    private javax.swing.JCheckBox colorBackgroundCheckBox;
    private javax.swing.JCheckBox colorBorderCheckBox;
    private javax.swing.JPanel colorSelectPanel;
    private javax.swing.JTextField columnsField;
    private javax.swing.JCheckBox commentMarkersCheckbox;
    private javax.swing.JButton createButton;
    private javax.swing.JComboBox<String> dataDelimiterComboBox;
    private javax.swing.JButton doneButton;
    private javax.swing.JPanel eastpanel;
    private javax.swing.JLabel jLabel1;
    private javax.swing.JLabel jLabel2;
    private javax.swing.JLabel jLabel3;
    private javax.swing.JLabel jLabel6;
    private javax.swing.JLabel jLabel7;
    private javax.swing.JLabel jLabel8;
    private javax.swing.JLabel jLabel9;
    private javax.swing.JScrollPane jScrollPane1;
    private javax.swing.JPanel middlePanel;
    private javax.swing.JCheckBox placeHolderCheckBox;
    private javax.swing.JButton rescanButton;
    private javax.swing.JTextField rowsField;
    private javax.swing.JPanel scanResultPanel;
    private javax.swing.JTextArea scanResultsArea;
    private javax.swing.JTextField tableWidthField;
    private javax.swing.JButton titleColorButton;
    private javax.swing.JButton titleColorChip;
    private javax.swing.JCheckBox titleColorColumnCheckbox;
    private javax.swing.JCheckBox titleColorRowCheckBox;
    private javax.swing.JButton undoButton;
    // End of variables declaration//GEN-END:variables
    
}
