// ***************************************************************************
// *   Copyright (C) 2018 by Paul Lutus                                      *
// *   lutusp@arachnoid.com                                                  *
// *                                                                         *
// *   This program is free software; you can redistribute it and/or modify  *
// *   it under the terms of the GNU General Public License as published by  *
// *   the Free Software Foundation; either version 2 of the License, or     *
// *   (at your option) any later version.                                   *
// *                                                                         *
// *   This program is distributed in the hope that it will be useful,       *
// *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
// *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
// *   GNU General Public License for more details.                          *
// *                                                                         *
// *   You should have received a copy of the GNU General Public License     *
// *   along with this program; if not, write to the                         *
// *   Free Software Foundation, Inc.,                                       *
// *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
// ***************************************************************************

/*
 * FTPUtils.java
 *
 * Created on February 20, 2002, 1:08 AM
 */
package FTPService;

import Arachnophilia.*;
import java.io.*;
import java.net.*;
import java.util.*;

/**
 *
 * @author Administrator
 * @version
 */
final public class FTPUtils {

    Arachnophilia main;
    public int portNumber = 21;
    private Socket comSocket;
    private Socket dataSocket = null;
    private BufferedReader bReader;
    private PrintWriter pWriter;
    boolean waitForReply = false;
    boolean uploadAll = false;
    boolean syncLog = false;
    private TreeMap<String, Long> fileTable;
    String server;
    String currentDirectory = "";
    private String user;
    private String password;
    FTPPanel panel;
    int totalFilesScanned;
    int totalFilesTransferred;
    int totalBytesScanned;
    int totalBytesTransferred;
    int totalDirs;
    ArrayList<String> errorFileList;
    String currentFile = "";

    /**
     * Creates new FTPUtils
     */
    public FTPUtils(String portNumber, String server, String user, String password, FTPPanel panel, Arachnophilia m, boolean syncLog) {
        this.syncLog = syncLog;
        errorFileList = new ArrayList<String>();
        this.panel = panel;
        main = m;
        this.user = user;
        this.password = password;
        this.server = server;
        try {
            this.portNumber = Integer.parseInt(portNumber);
        } catch (Exception e) {
            this.portNumber = 21;
        }
        if (!syncLog) {
            initializeConnection();
        }
    }

    private void initializeConnection() {
        try {
            panel.processTrace("* Establishing connection with " + server, 1);
            connect(server);
            login(user, password);
        } catch (IOException ioe) {
            ioe.printStackTrace(System.out);
        }
    }

    private void connect(String server)
            throws IOException {

        comSocket = new Socket(server, portNumber);

        InputStream is = comSocket.getInputStream();
        bReader = new BufferedReader(new InputStreamReader(is));
        OutputStream os = comSocket.getOutputStream();
        pWriter = new PrintWriter(os, true); // auto flush true.

        String reply = sendCommand("", "220",false);

        if (reply != null && reply.length() > 0) { // ftp server alive
            panel.processTrace("Connected to FTP server " + server, 2);
        } else {
            panel.processTrace("Error connecting to FTP server " + server, 2);
        }
    }

    private void login(String user, String password)
            throws IOException {
        sendCommand("USER " + user, "",false);
        sendCommand("PASS " + password, "230",false);
        currentDirectory = "???";
    }

    private boolean changeDir(String dir)
            throws IOException {
        boolean ok = true;
        if (!dir.equals(currentDirectory)) {
            ok = sendCommand("CWD " + dir, "",false).length() > 0;
            panel.processTrace("CWD " + dir + " " + ((ok) ? "successful" : "failed"), 3);
            if (ok) {
                currentDirectory = dir;
            }
        }
        return ok;
    }

    private String getList(String dir)
            throws IOException {
        if (sendCommand("LIST " + dir, "",false).length() > 0) {
            return getDataAsString();
        }
        return null;
    }

    private boolean fileExists(String name)
            throws IOException {
        return (getList(name)) != null;
    }

    private String getDataAsString()
            throws IOException {
        dataSocket = getDataSocket();
        InputStream is = dataSocket.getInputStream();
        String data = getAsString(is);
        is.close();
        dataSocket.close();
        return data;
    }

    private byte[] getDataAsByteArray()
            throws IOException {
        dataSocket = getDataSocket();
        InputStream is = dataSocket.getInputStream();
        byte[] data = getAsByteArray(is);
        is.close();
        dataSocket.close();
        return data;
    }

    private boolean makeDir(String dir, boolean suppressWarning)
            throws IOException {
        boolean outcome = sendCommand("MKD " + dir, "",false).length() > 0;
        panel.processTrace("MKD " + dir + " " + ((outcome | suppressWarning) ? "successful" : "failed"), 3);
        return outcome;
    }

    private void changeMakeDir(String dir)
            throws IOException {
        if (!changeDir(dir)) { // unable to create
            int p = 0;
            // must make parent directories
            // before creating target
            while ((p = dir.indexOf("/", p)) != -1) {
                String part = dir.substring(0, p);
                if (part.length() > 1) {
                    makeDir(part, true);
                }
                p = p + 1;
            }
            // make he full path
            makeDir(dir, true);
        }
        // try again :)
        changeDir(dir);
    }

    private void setTransferType(boolean asc)
            throws IOException {
        // set file transfer type
        String ftype = (asc ? "A" : "I");
        sendCommand("TYPE " + ftype, "200",false);
    }

    public boolean createDirUploadFile(String localBaseDir, String remoteBaseDir, String dir, String fileName, boolean asc)
            throws IOException {
        String destPath = smartDelim(remoteBaseDir, dir);
        panel.processTrace("Processing " + destPath + "/" + fileName, 2);
        changeMakeDir(destPath);

        byte[] data = readLocalFile(localBaseDir, dir, fileName, asc);

        boolean done = false;
        boolean needReInit = false;
        while (!done) {
            if (needReInit) {
                panel.processTrace("Error: must reinitialize.", 3);
                initializeConnection();
                changeMakeDir(destPath);
                needReInit = false;
            }
            try {

                uploadByteArray(fileName, data, asc);
                done = true;

            } catch (Exception e) {
                panel.processTrace("*** error in createDirUploadFile: " + e.getMessage(), 3);
                needReInit = true;
            }
        }
        return done;
    }

    public void uploadByteArray(String file, byte[] data, boolean asc)
            throws IOException {
        setTransferType(asc);
        dataSocket = getDataSocket();
        OutputStream os = dataSocket.getOutputStream();
        sendCommand("STOR " + file, "",true);
        int max = data.length;
        int offset = 0;
        int len = 2048;
        while (offset < max) {
            int q = len;
            if (q > max - offset) {
                q = max - offset;
            }
            os.write(data, offset, q);
            os.flush();
            offset += q;
            panel.updateProgressBar(0, max, offset, file);
        }
        os.close();
        dataSocket.close();
        sendCommand("", "226",false);
        panel.updateProgressBar(0, max, 0, "");
    }

    public byte[] readLocalFile(String basePath, String dir, String file, boolean asc) {
        byte[] result = null;
        ByteArrayOutputStream bos;
        int blockSize = 8192;
        File f = new File(basePath + "/" + dir + "/" + file);
        try {
            FileInputStream is = new FileInputStream(f);
            bos = new ByteArrayOutputStream();
            byte[] buffer = new byte[blockSize];
            int len;
            while ((len = is.read(buffer, 0, blockSize)) != -1) {
                bos.write(buffer, 0, len);
            }
            result = bos.toByteArray();

        } catch (Exception e) {
            e.printStackTrace(System.out);
        }
        return result;
    }

    private void loadHashTable(String base) {
        String path = smartDelim(base, Arachnophilia.ftpLogFileName);
        File f = new File(path);
        try {
            BufferedReader br = new BufferedReader(new FileReader(f));
            String line;
            while ((line = br.readLine()) != null) {
                int p;
                p = line.indexOf(";");
                if (p == -1) {
                    p = line.indexOf("\t");
                    if (p != -1) {
                        String key = line.substring(0, p).trim();
                        String svalue = line.substring(p + 1).trim();
                        long value = Long.parseLong(svalue, 16);
                        fileTable.put(key, new Long(value));
                    }
                }
            }
            br.close();
        } catch (Exception e) {
            // no log file
        }
    }

    private void saveHashTable(String base) {
        String path = smartDelim(base, Arachnophilia.ftpLogFileName);
        File f = new File(path);
        try {
            BufferedWriter bw = new BufferedWriter(new FileWriter(f));
            String explanation =
                    ";\n"
                    + "; Arachnophilia FTP Service Log File -- " + new Date() + "\n"
                    + ";\n"
                    + "; This file contains the most recent modification times\n"
                    + "; for the files in this direcory and its subdirectories.\n"
                    + ";\n"
                    + "; This information is used by the Arachnophilia FTP Service\n"
                    + "; to synchronize these file with your Web site without uploading\n"
                    + "; files unnecessarily.\n"
                    + ";\n"
                    + "; If you delete this file, the next FTP update will upload\n"
                    + "; all files and recreate this file.\n"
                    + ";\n";
            explanation = new SearchReplace().srchRplc(explanation, "\n", ArachConstants.SYSTEM_EOL);
            bw.write(explanation);
            String line;
            Iterator<String> keys = fileTable.keySet().iterator();
            while (keys.hasNext()) {
                String key = keys.next();
                line = key + "\t" + Long.toString(((Long) fileTable.get(key)).longValue(), 16);
                bw.write(line + ArachConstants.SYSTEM_EOL);
            }
            bw.close();
        } catch (Exception e) {
            e.printStackTrace(System.out);
        }
    }

    private void fileTablePut(File f) {
        fileTable.put(f.getPath(), new Long(f.lastModified()));
    }

    private boolean isFileNewer(File f) {
        if (uploadAll) {
            return true;
        }
        Long modTime = (Long) fileTable.get(f.getPath());
        if (modTime != null) {
            return modTime.longValue() < f.lastModified();
        } else {
            return true;
        }
    }

    public void scanDirEntry(String localBase, String remoteBase, String dir, boolean execute, boolean scanSubs, boolean uploadAll)
            throws IOException {
        totalDirs = 0;
        totalFilesScanned = 0;
        totalFilesTransferred = 0;
        totalBytesScanned = 0;
        totalBytesTransferred = 0;
        errorFileList = new ArrayList<String>();
        this.uploadAll = uploadAll;
        fileTable = new TreeMap<String, Long>();
        if (!uploadAll) {
            loadHashTable(localBase);
        }
        scanDir(localBase, remoteBase, dir, execute, scanSubs, syncLog);
        if (panel.cancelOp) {
            panel.processTrace("*** Operation cancelled.", 0);
        } else if (execute || syncLog) {
            saveHashTable(localBase);
        }
    }

    public void scanDir(String localBase, String remoteBase, String dir, boolean execute, boolean scanSubs, boolean syncLog)
            throws IOException {
        String path = smartDelim(localBase, dir);
        panel.processTrace("Scanning directory: " + path, 1);
        File a = new File(path);
        File[] list = a.listFiles();
        if (list != null) {
            for (int i = 0; i < list.length && !panel.cancelOp; i++) {
                File f = list[i];
                String fileName = f.getName();
                if (!fileName.matches(".*~") && !fileName.equals(Arachnophilia.ftpLogFileName) && fileName.indexOf(Arachnophilia.tempFilePrefix) == -1) {
                    if (f.isDirectory()) {
                        totalDirs++;
                        if (scanSubs) {
                            scanDir(localBase, remoteBase, smartDelim(dir, fileName), execute, scanSubs, syncLog);
                        }
                    } else {
                        totalFilesScanned++;
                        totalBytesScanned += f.length();
                        //String remPath = smartDelim(remoteBase, dir);
                        //remPath = smartDelim(remPath,fileName);
                        String localPath = f.getPath();

                        if (isFileNewer(f)) {
                            if (execute) {

                                panel.processTrace("* Uploading " + localPath, 1);
                                boolean asc = main.fileTypes.isTextFile(fileName);
                                currentFile = localPath;
                                if (createDirUploadFile(localBase, remoteBase, dir, fileName, asc)) {
                                    totalFilesTransferred++;
                                    totalBytesTransferred += f.length();
                                    // update entry for this file
                                    fileTablePut(f);
                                }
                            } else { // not execute
                                if (syncLog) {
                                    fileTablePut(f);
                                }
                                totalFilesTransferred++;
                                totalBytesTransferred += f.length();
                                panel.processTrace("Would upload " + localPath, 1);
                            }
                        } else { // not newer
                            panel.processTrace("Scanning " + localPath, 1);
                        }
                    }
                }
            }
        } else { // null
            panel.processTrace("Error: null directory list for " + a.getPath(), 3);
        }
    }

    private String smartDelim(String a, String b) {
        int delim = 0;
        if (a.length() > 0 && a.charAt(a.length() - 1) == '/') {
            delim++;
        }
        if (b.length() > 0 && b.charAt(0) == '/') {
            delim++;
        }
        if (delim == 1) {
            return a + b;
        } else if (delim == 0) {
            return a + "/" + b;
        } else {
            return a + b.substring(1);
        }
    }

    // if the com ends with a space
    // we expect data on the data connection
    /* public String sendCommandArray(String[] coms)
     throws IOException {
     dataSocket = getDataSocket();
     InputStream is = dataSocket.getInputStream();
     StringBuilder sb = new StringBuilder();
     for(int i = 0;i < coms.length;i++) {
     sb.append("reply c: " + sendCommand(coms[i],""));
            
     if(coms[i].charAt(coms[i].length() -1) == ' ') {
     sb.append("reply d: " + getAsString(is));
     }
     }
     logOut(true);
     return sb.toString();
     }*/
    public void logOut(boolean execute) {
        if (execute && !syncLog) {
            try {
                panel.processTrace("* Closing connection with " + server, 1);
                pWriter.print("BYE" + "\r\n");
                pWriter.flush();
                pWriter.close();
                bReader.close();
                comSocket.close();
                if (dataSocket != null) {
                    dataSocket.close();
                }
            } catch (IOException ioe) {
                ioe.printStackTrace(System.out);
            }
        }
    }

    private String sendCommand(String cmd, String expect,boolean checkError)
            throws IOException {
        //boolean outcome = true;
        if (cmd.length() > 0) {
            String copy = cmd;
            if (copy.indexOf("PASS ") != -1) {
                copy = "PASS *******************";
            }
            panel.processTrace("> Local command: " + copy, 3);
        }
        if (cmd.length() > 0) {
            pWriter.print(cmd.trim() + "\r\n");
            pWriter.flush();
        }
        String returnCode = "";
        String reply = "";
        boolean hasHyphen = false;
        boolean fileError = false;
        do {
            reply = bReader.readLine().trim();
            panel.processTrace("< Server reply: " + reply, 3);
            if (reply != null && reply.length() > 4) {
                returnCode = reply.substring(0, 3);
                hasHyphen = reply.charAt(3) == '-';
                if (checkError && hasHyphen && !returnCode.equals("226")) {
                    fileError = true;
                }
            }
        } while (expect.length() > 0 && (!returnCode.equals(expect) || hasHyphen));
        if (reply != null) {
            if (reply.length() > 0) {
                char c = reply.charAt(0);
                if (c == '4' || c == '5') {
                    reply = "";
                }
            }
        }
        if (fileError) {
            errorFileList.add(currentFile);
            panel.processTrace("*** Probable format error in file " + currentFile, 2);
        }
        return reply;
    }

    private String getAsString(InputStream is) {
        int c = 0;
        StringBuilder sb = new StringBuilder();
        int bufSize = 8192;
        byte[] buffer = new byte[bufSize];
        int len;
        try {
            while ((len = is.read(buffer, 0, bufSize)) != -1) {
                sb.append(new String(buffer, 0, len));
            }
        } catch (IOException ioe) {
            ioe.printStackTrace(System.out);
        }

        return sb.toString();
    }

    private byte[] getAsByteArray(InputStream is) {
        ByteArrayOutputStream bao = new ByteArrayOutputStream();
        int bufSize = 8192;
        byte[] buffer = new byte[bufSize];
        int len;
        try {
            while ((len = is.read(buffer, 0, bufSize)) != -1) {
                bao.write(buffer, 0, len);
            }
        } catch (IOException ioe) {
            ioe.printStackTrace(System.out);
        }
        return bao.toByteArray();
    }

    public Socket getDataSocket() throws IOException {
        // Yo server! Give me a data connection!
        String reply = sendCommand("PASV", "227",false);

        // assume very little about the address format
        // it consists of six numbers separated by commas
        // but that is all one can count on.
        // make p point at the comma separating numbers 1 and 2
        int p = reply.indexOf(",");
        if (p != -1) {
            // back up to first valid digit
            while (isValidAddressChar(reply, p)) {
                p--;
            }
            // now scan the entire address
            Integer[] vals = scanAddressPort(reply, p + 1);
            // our array must be six digits long
            if (vals.length != 6) {
                panel.processTrace("Malformed server reply to PASV: \"" + reply + "\"", 3);
            }
            // assemble IP address
            String ip = "";
            for (int i = 0; i < 4; i++) {
                ip += ((i > 0) ? "." : "") + vals[i];
            }
            // create port number
            int port = (vals[4].intValue() << 8) + vals[5].intValue();
            dataSocket = new Socket(ip, port);
        }
        panel.processTrace("socket params: " + dataSocket, 3);
        return dataSocket;
    }

    private boolean isValidAddressChar(String s, int p) {
        if (p >= 0 && p < s.length()) {
            char c = s.charAt(p);
            return Character.isDigit(c) || c == ',';
        }
        return false;
    }

    private Integer[] scanAddressPort(String s, int p) {
        ArrayList<Integer> v = new ArrayList<Integer>();
        boolean inNumber = false;
        int n = 0;
        // a little state machine :)
        while (p < s.length()) {
            char c = s.charAt(p);
            if (Character.isDigit(c)) {
                if (!inNumber) {
                    n = 0;
                    inNumber = true;
                }
                n = (n * 10) + (c - '0');
            } // or not a digit, any other character
            else {
                if (inNumber) {
                    v.add(new Integer(n));
                }
                inNumber = false;
            }
            // are we beyond the address area?
            if (!(Character.isDigit(c) || c == ',')) {
                break;
            }
            p++;
        }
        // did the scan terminate on a valid digit?
        if (inNumber) {
            v.add(new Integer(n));
        }
        return v.toArray(new Integer[]{});
    }
}
