// ***************************************************************************
// *   Copyright (C) 2018 by Paul Lutus                                      *
// *   lutusp@arachnoid.com                                                  *
// *                                                                         *
// *   This program is free software; you can redistribute it and/or modify  *
// *   it under the terms of the GNU General Public License as published by  *
// *   the Free Software Foundation; either version 2 of the License, or     *
// *   (at your option) any later version.                                   *
// *                                                                         *
// *   This program is distributed in the hope that it will be useful,       *
// *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
// *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
// *   GNU General Public License for more details.                          *
// *                                                                         *
// *   You should have received a copy of the GNU General Public License     *
// *   along with this program; if not, write to the                         *
// *   Free Software Foundation, Inc.,                                       *
// *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
// ***************************************************************************
package Arachnophilia;

/*
 * DocumentProcessingFunctions.java
 *
 * Created on February 8, 2002, 8:35 PM
 */
import CustomClassFunctions.*;
import FileTypes.*;
import GotoFunctions.*;
import SourceBeautifiers.MultiCodeBeautifier;
import java.awt.*;
import java.io.*;
import java.nio.charset.Charset;
import java.util.*;
import javax.swing.*;

/**
 *
 * @author Administrator
 * @version
 */
final public class DocumentProcessingFunctions {

    Arachnophilia main;
    int selectionStart;
    int selectionEnd;
    int firstLine;
    int cp;
    int origLen;
    int finalLen;
    ArachDocument doc;
    // TextInterface textComp;
    String content;
    DocContentHandler docHandler;
    SearchReplace searchReplace;
    boolean selectionValid = false;

    DocumentProcessingFunctions(Arachnophilia m) {
        main = m;
        docHandler = new DocContentHandler(m);
        searchReplace = new SearchReplace();
        doc = main.currentSelectedDocument;
    }

    public int checkRange(int v, TextInterface tc) {
        int top = tc.getLength();
        v = (v < 0) ? 0 : v;
        v = (v > top) ? top : v;
        return v;
    }

    // base routine for block indent/outdent
    private String blockInOutDent(String content, int which) {
        String tab = ArachComp.getTabString();
        StringBuilder sb = new StringBuilder();
        // char lastChar = '\n';
        ArrayList<String> v = ArachComp.parseDelimLine(content, "\n");
        int len = v.size();
        int tabLen = tab.length();
        int vsize = v.size();
        for (int i = 0; i < vsize; i++) {
            String term = (i < vsize - 1) ? "\n" : "";
            String line = v.get(i);
            int lineLen = line.length();
            if (which > 0) {
                sb.append(tab).append(line).append(term);
            } else { // which < 0
                if (lineLen >= tabLen && line.substring(0, tabLen).equals(tab)) {
                    sb.append(line.substring(tabLen)).append(term);
                } else {
                    sb.append(line).append(term);
                }
            }
        }
        return sb.toString();
    }

    public void scanToWordEnd(ArachDocument doc, int direction) {

        int offset = doc.textComp.getCaretPosition();
        int length = doc.textComp.getLength();
        boolean inWord = false;
        while (offset >= 0 && offset < length) {
            String s = doc.textComp.getText(offset, 1);
            if (s != null && s.length() > 0) {
                char c = s.charAt(0);
                if (Character.isWhitespace(c)) {
                    if (inWord) {
                        doc.textComp.setCaretPosition(offset);
                        break;
                    } else {
                        inWord = false;
                    }
                } else { // not white space
                    if (!inWord) {
                        inWord = true;
                    }
                }

            }
            offset += direction;
        }
    }

    public void jumpToLineEnd(ArachDocument doc, int direction) {
        try {
            int pos = doc.textComp.getCaretPosition();
            int line = doc.textComp.getCaretLine();
            int totalLines = doc.textComp.getLineCount();
            int offset;
            if (direction < 0) {
                offset = doc.textComp.getLineStartOffset(line);
                if (offset == pos && line > 0) {
                    offset = doc.textComp.getLineStartOffset(line - 1);
                }
            } else { // direction > 0
                offset = doc.textComp.getLineEndOffset(line) - 1;
                if (offset == pos && line < totalLines) {
                    offset = doc.textComp.getLineEndOffset(line + 1) - 1;
                }
            }
            if (offset >= 0 && offset < doc.textComp.getLength() - 1) {
                doc.textComp.setCaretPosition(offset);
            }
        } catch (Exception e) {
            e.printStackTrace(System.out);
        }
    }

    public void jumpToDocEnd(ArachDocument doc, int direction) {
        int offset = (direction < 0) ? 0 : doc.textComp.getLength() - 1;
        doc.textComp.setCaretPosition(offset);
    }

    // BEGIN processing functions
    public void setFileType(String arg) {

        if (main.currentSelectedDocument != null) {
            if (arg.length() > 0) {
                int type = main.fileTypes.getFileTypeForName(arg);
                main.currentSelectedDocument.changeFileType(type);
            } else {
                SetFileTypeDialog setFileTypeDialog = new SetFileTypeDialog(main, main.currentSelectedDocument);
            }
        }
    }

    public String makeOrderedList(String arg) {
        return makeList("ol", arg);
    }

    public String makeUnorderedList(String arg) {
        return makeList("ul", arg);
    }

    private String makeList(String tag, String modifier) {
        if ((content = docHandler.getContent()) != null) {
            ArrayList<String> v = ArachComp.parseDelimLine(content, "\n");
            int len = v.size();
            StringBuilder sb = new StringBuilder();
            String tab = ArachComp.getTabString();
            String firstLineWS = "";
            String lastLineWS = "";
            for (int i = 0; i < len; i++) {
                String leading = "";
                String line = v.get(i);
                int t = 0;
                while (t < line.length() && Character.isWhitespace(line.charAt(t))) {
                    t++;
                }
                if (t > 0 && t < line.length()) {
                    leading = line.substring(0, t);
                    line = line.substring(t);
                    if (i == 0) {
                        firstLineWS = leading;
                    } else if (i == len - 1) {
                        lastLineWS = leading;
                    }
                }
                sb.append(leading).append(tab);
                String lcLine = line.toLowerCase().trim();
                // prepend <li> only if
                // this is not the start
                // of a nested list or
                // is not already tagged
                if (line.length() > 0) {
                    if ((lcLine.length() > 0) && (lcLine.indexOf("<li") == -1) && (lcLine.indexOf("<ul") == -1)
                            && (lcLine.indexOf("<ol") == -1) && (lcLine.indexOf("</ul") == -1)
                            && (lcLine.indexOf("</ol") == -1)) {

                        sb.append("<li>").append(line).append("</li>");
                    } else {
                        sb.append(line);
                    }
                }
                if (i < len - 1) {
                    sb.append("\n");
                }
            }
            if (modifier.length() > 0) {
                content = firstLineWS + "<" + tag + " type=\"" + modifier + "\">\n" + sb.toString() + "\n" + lastLineWS
                        + "</" + tag + ">";
            } else {
                content = firstLineWS + "<" + tag + ">\n" + sb.toString() + "\n" + lastLineWS + "</" + tag + ">";
            }
            docHandler.setContent(content);
        }
        return "";
    }

    // this is a bit complex :)
    // this function uses CustomClassHandler
    // to call a class located at
    // (userHome)/path/className
    public String RunCustomClassArg(String arg) {
        String className = arg;
        String argument = "";
        int p = arg.indexOf(",");
        if (p != -1) {
            className = arg.substring(0, p);
            argument = arg.substring(p + 1);
        }
        return new CustomClassHandler(main).processRequest(className, argument);
    }

    public String RunCustomClassDoc(String className) {
        if ((content = docHandler.getContent()) != null) {
            content = new CustomClassHandler(main).processRequest(className, content);
            docHandler.setContent(content);
        }
        return "";
    }

    // 0 = word, 1 = line 2 = document
    public void jumpTo(int direction, int type) {
        final int JUMP_TO_WORD = 0;
        final int JUMP_TO_LINE = 1;
        final int JUMP_TO_DOC = 2;
        if (main.currentSelectedDocument != null) {
            ArachDocument localDoc = main.currentSelectedDocument;
            switch (type) {
                case JUMP_TO_WORD: {
                    scanToWordEnd(localDoc, direction);
                    break;
                }
                case JUMP_TO_LINE: {
                    jumpToLineEnd(localDoc, direction);
                    break;
                }
                case JUMP_TO_DOC: {
                    jumpToDocEnd(localDoc, direction);
                    break;
                }
            }
            localDoc.requestFocusInWindow();
        }
    }

    public void gotoLine(String arg) {

        if (main.currentSelectedDocument != null) {
            ArachDocument localDoc = main.currentSelectedDocument;
            if (arg.length() > 0) {
                try {
                    int go = Integer.parseInt(arg);
                    localDoc.gotoLine(go - 1);
                } catch (Exception e) {
                }
            } else {
                Goto goDialog = new Goto(main);
                int go = goDialog.gotoLine;
                if (go != -1) {
                    localDoc.gotoLine(go - 1);
                }
            }
            localDoc.requestFocusInWindow();
        }
    }

    public void blockInOutDent(int which) {
        String localContent;
        if ((localContent = docHandler.getContent()) != null) {
            localContent = blockInOutDent(localContent, which);
            docHandler.setContent(localContent);
        }
    }

    public void blockTab() {
        if ((content = docHandler.getContent()) != null) {
            if (docHandler.textSelected()) {
                // System.out.println("selected");
                content = blockInOutDent(content, 11);
                docHandler.setContent(content);
            } else {
                // System.out.println("not selected");
                doc.textComp.replaceSelection(ArachComp.getTabString());
            }
        }
    }

    public void blockReverseTab() {
        if ((content = docHandler.getContent()) != null) {
            if (docHandler.textSelected()) {
                content = blockInOutDent(content, 0);
                docHandler.setContent(content);
            }
        }
    }

    public String createFileLink(String arg) {
        String result = "";
        if (main.currentSelectedDocument != null) {
            ArachDocument localDoc = main.currentSelectedDocument;
            result = main.rightClickProcessor.createFileLink(localDoc, arg);
        }
        return result;
    }

    public void beautifyHTML() {
        String localContent;
        if ((localContent = docHandler.getContent()) != null) {
            String newContent = main.beautifyUtils.beautifyHTML(localContent);
            if (!newContent.equals(localContent)) {
                docHandler.setContent(newContent);
            }
        }
    }

    public void beautifyCode() {
        String localContent;
        if ((localContent = docHandler.getContent()) != null) {
            String newContent = new MultiCodeBeautifier(main).beautifyCodeString(main.currentSelectedDocument,
                    localContent, docHandler.getDoc().name);
            if (!newContent.equals(localContent)) {
                docHandler.setContent(newContent);
            }
        }
    }

    public void validateHTML() {
        String localContent;
        if ((localContent = docHandler.getContent()) != null) {
            main.htmlValidator.validateHTML(localContent);
        }
    }

    public void replaceLiTags() {
        String localContent;
        if ((localContent = docHandler.getContent()) != null) {
            localContent = LiTagResolver.resolve(localContent);
            docHandler.setContent(localContent);
        }
    }

    public void jumpToOppositeTag() {
        String localContent;
        if ((localContent = docHandler.getContent()) != null) {
            JumpToOppositeTag.jump(docHandler.getDoc());
        }
    }

    public String insertFullGraphicTag() {
        String localContent = "";
        if (main.currentSelectedDocument != null) {
            ArachDocument localDoc = main.currentSelectedDocument;
            String tag = main.rightClickProcessor.createGraphicLink(localDoc);
            if (tag.length() > 0) {
                localContent = "<img src=\"" + tag + "\" width=\"\" height=\"\" title=\"\" alt=\"|\"/>";
                localContent = main.rightClickProcessor.sizeAllGraphicTagsInString(localDoc, localContent);
            }
        }
        return localContent;
    }

    public String insertGraphicLink() {
        String localContent = "";
        if (main.currentSelectedDocument != null) {
            ArachDocument localDoc = main.currentSelectedDocument;
            localContent = main.rightClickProcessor.createGraphicLink(localDoc);
        }
        return localContent;
    }

    public void sizeGraphics() {
        String localContent;
        if ((localContent = docHandler.getContent()) != null) {
            localContent = main.rightClickProcessor.sizeAllGraphicTagsInString(doc, localContent);
            docHandler.setContent(localContent);
        }
    }

    public void tabsToSpaces() {
        String localContent;
        if ((localContent = docHandler.getContent()) != null) {
            String tab = ArachComp.makeSpaceTab();
            localContent = searchReplace.srchRplc(localContent, "\t", tab);
            docHandler.setContent(localContent);
        }
    }

    public void spacesToTabs() {
        String localContent;
        if ((localContent = docHandler.getContent()) != null) {
            String tab = ArachComp.makeSpaceTab();
            localContent = searchReplace.srchRplc(localContent, tab, "\t");
            docHandler.setContent(localContent);
        }
    }

    public void convertDocToHTML() {

        if (main.currentSelectedDocument != null) {
            DocToHTMLConverter conv = new DocToHTMLConverter(main);
            String localContent = conv.convertDoc(main.currentSelectedDocument.textComp);
            if (localContent.length() > 0) {
                main.createHTMLDocFromString(localContent);
            } else {
                main.beep();
            }
        }
    }

    public String getSelection() {
        String localContent;
        if ((localContent = docHandler.getContent()) != null) {
            return localContent;
        }
        return "";
    }

    public String launchColorSelectorDialog(String arg) {
        String prompt = "Choose a color";
        int initial = 0;
        if (arg.length() > 0) {
            try {
                initial = Integer.parseInt(arg, 16);
            } catch (NumberFormatException e) {
                prompt = arg;
            }
        }
        Color col = JColorChooser.showDialog(main, prompt, new Color(initial));
        if (col == null) {
            col = new Color(initial);
        }
        return ArachComp.colorIntToString(col.getRGB());
    }

    public void processMacrosInDocument() {
        String localContent;
        if ((localContent = docHandler.getContent()) != null) {
            localContent = main.comSwitchboard.exec(localContent, 0);
            docHandler.setContent(localContent);
        }
    }

    public String processMacrosInString(String content) {
        return main.comSwitchboard.exec(content, 0);
    }

    public void wordCount() {
        String localContent;
        if ((localContent = docHandler.getContent()) != null) {
            int lines = 1, words = 0, characters;
            boolean inWord = false;
            characters = localContent.length();
            for (int i = 0; i < characters; i++) {
                char c = localContent.charAt(i);
                if (c == '\n') {
                    lines++;
                }
                boolean isWordChar = Character.isLetterOrDigit(c) || (c == '\'') || (c == '-');
                if (inWord) {
                    if (!isWordChar) {
                        words++;
                        inWord = false;
                    }
                } else { // between words
                    if (isWordChar) {
                        inWord = true;
                    }
                }
            }
            if (inWord) {
                words++;
            }
            JOptionPane.showMessageDialog(main, words + " words," + lines + " lines," + characters + " characters.",
                    "Word Count", JOptionPane.OK_OPTION);
        }
    }

    public void toLowerCase() {
        String localContent;
        if ((localContent = docHandler.getContent()) != null) {
            localContent = localContent.toLowerCase();
            docHandler.setContent(localContent);
        }
    }

    public void toUpperCase() {
        String localContent;
        if ((localContent = docHandler.getContent()) != null) {
            localContent = localContent.toUpperCase();
            docHandler.setContent(localContent);
        }
    }

    public void convertTextToHTML() {
        String localContent;
        if ((localContent = docHandler.getContent()) != null) {
            localContent = main.entityProcessor.entityToString(localContent, true);
            localContent = searchReplace.srchRplc(localContent, "\n", "<br/>\n");
            localContent = "<html><head><title></title></head><body>\n" + localContent + "</body></html>\n";
            docHandler.setContent(localContent);
        }
    }

    public void convertHTMLToText() {
        String lcoalContent;
        if ((lcoalContent = docHandler.getContent()) != null) {
            // strip all existing linefeeds
            lcoalContent = searchReplace.srchRplc(lcoalContent, "\n", " ");
            // and tabs
            lcoalContent = searchReplace.srchRplc(lcoalContent, "\t", "");
            // make real spaces from symbolic ones
            lcoalContent = searchReplace.srchRplc(lcoalContent, "&nbsp;", " ");
            // now put in some linefeeds
            lcoalContent = searchReplace.srchRplc(lcoalContent, "</p>", "</p>\n\n", false);
            lcoalContent = searchReplace.srchRplc(lcoalContent, "<br/>", "<br/>\n", false);
            lcoalContent = searchReplace.srchRplc(lcoalContent, "</td>", "</td>\n", false);
            lcoalContent = searchReplace.srchRplc(lcoalContent, "</div>", "</div>\n", false);
            // now strip all tags
            StringBuilder sb = new StringBuilder();
            int len = lcoalContent.length();
            boolean inTag = false;
            char c;
            char lastValidChar = 'x';
            for (int i = 0; i < len; i++) {
                c = lcoalContent.charAt(i);
                if (c == '<') {
                    inTag = true;
                }
                if (!inTag) {
                    sb.append(c);
                    lastValidChar = c;
                }
                if (c == '>') {
                    inTag = false;
                    if (i < len - 1 && !Character.isWhitespace(lcoalContent.charAt(i + 1))) {
                        if (!Character.isWhitespace(lastValidChar)) {
                            sb.append(' ');
                        }
                    }
                }
            }
            // unescape HTML entities
            lcoalContent = main.entityProcessor.entityToChar(sb.toString(), true);
            // now trim lines
            ArrayList<String> v = ArachComp.parseDelimLine(lcoalContent, "\n");
            len = v.size();
            sb = new StringBuilder();
            for (int i = 0; i < len; i++) {
                String line = v.get(i);
                sb.append(line.trim()).append(" \n");
            }
            docHandler.setContent(sb.toString());
        }
    }

    public void tagDelimsEscape() {
        String localContent;
        if ((localContent = docHandler.getContent()) != null) {
            localContent = searchReplace.srchRplc(localContent, "<", "&lt;");
            localContent = searchReplace.srchRplc(localContent, ">", "&gt;");
            docHandler.setContent(localContent);
        }
    }

    public void tagDelimsUnescape() {
        String localContent;
        if ((localContent = docHandler.getContent()) != null) {
            localContent = searchReplace.srchRplc(localContent, "&lt;", "<");
            localContent = searchReplace.srchRplc(localContent, "&gt;", ">");
            docHandler.setContent(localContent);
        }
    }

    public void leftFlushLines() {
        String localContent;
        if ((localContent = docHandler.getContent()) != null) {
            ArrayList<String> v = ArachComp.parseDelimLine(localContent, "\n", false);
            StringBuilder result = new StringBuilder();
            int len = v.size();
            for (int i = 0; i < len; i++) {
                String line = v.get(i);
                int j = 0;
                while (j < line.length() && Character.isWhitespace(line.charAt(j))) {
                    j++;
                }
                line = line.substring(j);
                result.append(line);
                if (i < len - 1) {
                    result.append("\n");
                }
            }

            docHandler.setContent(result.toString());
        }
    }

    public void compressText() {
        String localContent;
        if ((localContent = docHandler.getContent()) != null) {
            localContent = main.beautifyUtils.compressText(localContent);
            docHandler.setContent(localContent);
        }
    }

    public void changeTagCase(boolean upperCase) {
        String localContent;
        StringBuilder result = new StringBuilder();
        if ((localContent = docHandler.getContent()) != null) {
            int len = localContent.length();
            int a, b = 0;
            int origin = 0;
            while ((a = localContent.indexOf("<", b)) != -1) {
                b = a + 1;
                char c = localContent.charAt(b);
                if (c != '!') {
                    if (c == '/' || c == '!') {
                        b++;
                    }
                    while (b < len && Character.isLetter(localContent.charAt(b))) {
                        b++;
                    }
                    if (b < len) {
                        result.append(localContent.substring(origin, a));
                        // origin = a;
                        String tag = localContent.substring(a, b);
                        if (upperCase) {
                            result.append(tag.toUpperCase());
                        } else {
                            result.append(tag.toLowerCase());
                        }
                        origin = b;
                    }
                } else {
                    b++;
                }
            }
            result.append(localContent.substring(origin));

            docHandler.setContent(result.toString());
        }
    }

    public void setFindFocus() {
        final String input = docHandler.getContent();
        final int len = ((input != null) ? input.length() : 0);
        SwingUtilities.invokeLater(new Runnable() {
            @Override
            public void run() {

                // is the user trying to mark some text for search?
                if (len > 0 && len <= 64) {
                    main.findReplacePanel.setFindFocus(input);
                } else {
                    main.findReplacePanel.setFindFocus(null);
                }
            }
        });
    }

    public void execSystemCommand(String arg, boolean inputStream) {
        String output = "";
        String errors = "";
        String input = docHandler.getContent();
        try {
            Process p = Runtime.getRuntime().exec(arg);
            InputStream is, es;
            if (inputStream) {
                OutputStream os = p.getOutputStream();
                Writer out = new OutputStreamWriter(os, "UTF8");
                // the "old way", not suitable for UTF-8
                // for (Character c : input.toCharArray()) {
                // os.write(c);
                // }
                // os.close();
                out.write(input);
                out.close();
                os.close();
            }
            is = p.getInputStream();
            output = readData(is, p);
            is.close();
            es = p.getErrorStream();
            errors = readData(es, p);
            es.close();
        } catch (Exception e) {
            e.printStackTrace();
            errors += e.toString();
        }
        if (errors.length() > 0) {
            JOptionPane.showMessageDialog(main, "Problem executing command \"" + arg + "\":\n\n" + errors,
                    "System Command Execution", JOptionPane.OK_OPTION);
        }
        // check for nonzero result and result not equal to input
        if ((output.length() > 0) && (!output.equals(input))) {
            docHandler.setContent(output);
        }
    }

    private String readData(InputStream is, Process p) {
        StringBuilder sa = new StringBuilder();
        String data = "";
        try {
            InputStreamReader isr = new InputStreamReader(is, Charset.forName("UTF-8"));
            int c;
            while ((c = isr.read()) != -1) {
                sa.append((char) c);
            }
        } catch (Exception e) {
            e.printStackTrace(System.out);
        }
        data = sa.toString();
        if (data.length() > 0) {
            data = ArachComp.toJavaLineEndings(data);
        }
        return data;
    }

    public void escapeText() {
        String localContent;
        if ((localContent = docHandler.getContent()) != null) {
            localContent = EscapeUnescapeStringHandler.escapeString(localContent);
            docHandler.setContent(localContent);
        }
    }

    public void unescapeText() {
        String localContent;
        if ((localContent = docHandler.getContent()) != null) {
            localContent = EscapeUnescapeStringHandler.unescapeString(localContent);
            docHandler.setContent(localContent);
        }
    }

    public void entityToChar() {
        String localContent;
        if ((localContent = docHandler.getContent()) != null) {
            localContent = main.entityProcessor.entityToChar(localContent, docHandler.isHTML());
            docHandler.setContent(localContent);
        }
    }

    public void entityToString() {
        if ((content = docHandler.getContent()) != null) {
            content = main.entityProcessor.entityToString(content, docHandler.isHTML());
            docHandler.setContent(content);
        }
    }
}
