/***************************************************************************
 *   Copyright (C) 2012 by Paul Lutus                                      *
 *   lutusp@arachnoid.com                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
package com.arachnoid.tankcalcandroid;

import android.app.AlertDialog;
import android.app.Dialog;
import android.content.DialogInterface;
import android.content.Intent;
import android.content.pm.PackageManager.NameNotFoundException;
import android.net.Uri;
import android.os.Bundle;
import android.os.Environment;
import android.provider.Settings;
import android.support.design.widget.TabLayout;
import android.support.v4.app.Fragment;
import android.support.v4.app.FragmentManager;
import android.support.v4.app.FragmentPagerAdapter;
import android.support.v4.view.ViewPager;
import android.support.v7.app.ActionBar;
import android.support.v7.app.AppCompatActivity;
import android.support.v7.widget.Toolbar;
import android.text.Editable;
import android.text.TextWatcher;
import android.view.Menu;
import android.view.MenuItem;
import android.view.View;
import android.view.ViewTreeObserver;
import android.webkit.WebSettings;
import android.webkit.WebView;
import android.widget.AdapterView;
import android.widget.ArrayAdapter;
import android.widget.CheckBox;
import android.widget.CompoundButton;
import android.widget.CompoundButton.OnCheckedChangeListener;
import android.widget.EditText;
import android.widget.Spinner;
import android.widget.TabWidget;
import android.widget.TextView;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.InputStreamReader;
import java.util.ArrayList;

final public class TankCalcAndroidActivity extends AppCompatActivity {

    /**
     * The {@link android.support.v4.view.PagerAdapter} that will provide
     * fragments for each of the sections. We use a
     * {@link FragmentPagerAdapter} derivative, which will keep every
     * loaded fragment in memory. If this becomes too memory intensive, it
     * may be best to switch to a
     * {@link android.support.v4.app.FragmentStatePagerAdapter}.
     */
    private TankCalcAndroidActivity.SectionsPagerAdapter mSectionsPagerAdapter;

    /**
     * The {@link ViewPager} that will host the section contents.
     */
    private ViewPager mViewPager;

    TankCalcAndroidApplication app;
    //TankCalcAndroidActivity instance;
    //TankProperties sharedTP;
    //TabHost tabHost = null;
    TabWidget tabWidget;
    //TankProcessor tankProcessor;
    boolean taskRunning;
    ArrayList<Double[]> dblArray;
    String[] tableTitle;

    //AndroidConstants constants;
    //Transform3D graphicTransform;
    TankImageView menuView, imageView;

    //ImageGenerator imageGenHi, imageGenLo;

    // MyImageView graphicDisplay, graphicDescription;

    EditText tvTankLValue, tvTankRValue, tvTankLrValue, tvTankRrValue, tvScalingFactor,
            tvTankAngle, tvSensorX, tvSensorY, tvSensorZ, tvResidualVolume,
            tvWallThickness, tvWallDensity, tvContentDensity, tvCylIntSteps,
            tvCapIntSteps, tvTableHeightStepSize, tvTableVolumeStepSize,
            tvRootFinderEpsilon, tvEmailAddress;

    TextView tvTankLValueLbl, tvTankRadiusLbl, tvTankLrValueLbl,
            tvTankRrValueLbl, tvTankAngleLbl, tvSensorXLbl, tvSensorYLbl,
            tvSensorZLbl, tvResidualVolumeLbl, tvWallThicknessLbl,
            tvTableHeightStepSizeLbl, tvTableVolumeStepSizeLbl;

    Spinner lengthUnitSpinner, volumeUnitSpinner, tankOrientationSpinner;
    Spinner tankLrTypeSpinner, tankRrTypeSpinner, tankSensorAxisSpinner,
            tankAngleUnitSpinner, weightUnitSpinner, decimalPlacesSpinner;

    WebView helpWebView, tableWebView;

    CheckBox graphicInverse, graphicAnaglyphic, graphicHints,
            graphicTransparent, graphicDrawLeftCap, graphicDrawCylinder,
            graphicDrawRightCap, graphicDrawSensor;

    boolean initialized = false;
    TabLayout tabLayout;
    View progressBar;

    boolean fullScreen = false;


    /**
     * Called when the activity is first created.
     */
    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);

        //instance = this;

        app = (TankCalcAndroidApplication) getApplication();

        app.currentActivity = this;

        //sharedTP = app.sharedTP;

        setContentView(R.layout.activity_main);

        Toolbar toolbar = (Toolbar) findViewById(R.id.toolbar);
        setSupportActionBar(toolbar);

        ActionBar ab = getSupportActionBar();

        ab.setDisplayShowHomeEnabled(true);
        ab.setLogo(R.mipmap.app_icon);
        ab.setDisplayUseLogoEnabled(true);
        ab.setTitle("  TankCalcAndroid " + app.PROGRAM_VERSION);

        // Create the adapter that will return a fragment for each of the three
        // primary sections of the activity.
        mSectionsPagerAdapter = new TankCalcAndroidActivity.SectionsPagerAdapter(getSupportFragmentManager());

        // Set up the ViewPager with the sections adapter.
        mViewPager = (ViewPager) findViewById(R.id.container);
        mViewPager.setAdapter(mSectionsPagerAdapter);
        mViewPager.setOffscreenPageLimit(4);

        tabLayout = (TabLayout) findViewById(R.id.tabs);
        tabLayout.setupWithViewPager(mViewPager);



    }

    protected void afterResume() {
        // must be sure the layout is complete, everything created
        mViewPager.getViewTreeObserver().addOnGlobalLayoutListener(new ViewTreeObserver.OnGlobalLayoutListener() {
            @Override
            public void onGlobalLayout() {
                mViewPager.getViewTreeObserver().removeOnGlobalLayoutListener(this);
                layoutComplete();
            }
        });
    }


    protected void layoutComplete() {
        app.deSerialize();
        menuView = (TankImageView) findViewById(R.id.desc_graphic_image);
        imageView = (TankImageView) findViewById(R.id.disp_graphic_image);
        helpWebView = (MyWebView) findViewById(R.id.help_view);
        tableWebView = (MyWebView) findViewById(R.id.compute_display);
        progressBar = findViewById(R.id.progress_bar);
        //setupTabs();
        setupLinks();
        setupLists();
        setupHelp();
        readAllControls();
        writeProperties(false);
        initialized = true;
        updateWebView(helpWebView);
        updateWebView(tableWebView);
        int tab = app.sharedTP.currentTab;
        if (tab >= 0) {
            changeTabs(tab);
        }
    }

    private void setupLinks() {

        //tabHost = (TabHost) findViewById(android.R.id.tabhost);
        tabWidget = (TabWidget) findViewById(android.R.id.tabs);
        lengthUnitSpinner = (Spinner) findViewById(R.id.length_unit_spinner);
        volumeUnitSpinner = (Spinner) findViewById(R.id.volume_unit_spinner);
        tankAngleUnitSpinner = (Spinner) findViewById(R.id.tank_angle_units);
        weightUnitSpinner = (Spinner) findViewById(R.id.weight_unit_spinner);
        tankOrientationSpinner = (Spinner) findViewById(R.id.tank_orientation_spinner);
        tankLrTypeSpinner = (Spinner) findViewById(R.id.left_cap_type_spinner);
        tankRrTypeSpinner = (Spinner) findViewById(R.id.right_cap_type_spinner);
        tankSensorAxisSpinner = (Spinner) findViewById(R.id.sensor_axis_spinner);
        decimalPlacesSpinner = (Spinner) findViewById(R.id.decimal_places_spinner);

        tvTankLValue = setupEditText(R.id.cylinder_length);
        tvTankRValue = setupEditText(R.id.cylinder_radius);
        tvTankLrValue = setupEditText(R.id.left_cap_radius);
        tvTankRrValue = setupEditText(R.id.right_cap_radius);
        tvScalingFactor = setupEditText(R.id.scaling_factor);
        tvTankAngle = setupEditText(R.id.tank_angle);
        tvSensorX = setupEditText(R.id.sensor_x_position);
        tvSensorY = setupEditText(R.id.sensor_y_position);
        tvSensorZ = setupEditText(R.id.sensor_z_position);
        tvResidualVolume = setupEditText(R.id.residual_tank_volume);
        tvWallThickness = setupEditText(R.id.wall_thickness);
        tvWallDensity = setupEditText(R.id.wall_density);
        tvContentDensity = setupEditText(R.id.content_density);
        tvCylIntSteps = setupEditText(R.id.cyl_integration_steps);
        tvCapIntSteps = setupEditText(R.id.endcap_integration_steps);
        tvRootFinderEpsilon = setupEditText(R.id.root_finder_epsilon);
        tvTableHeightStepSize = setupEditText(R.id.table_height_step_size);
        tvTableVolumeStepSize = setupEditText(R.id.table_volume_step_size);
        //tvEmailAddress = setupEditText(R.id.email_entry);

        tvTankLValueLbl = (TextView) findViewById(R.id.cylinder_length_lbl);
        tvTankRadiusLbl = (TextView) findViewById(R.id.cylinder_radius_lbl);
        tvTankLrValueLbl = (TextView) findViewById(R.id.left_cap_radius_lbl);
        tvTankRrValueLbl = (TextView) findViewById(R.id.right_cap_radius_lbl);
        tvTankAngleLbl = (TextView) findViewById(R.id.tank_angle_lbl);
        tvSensorXLbl = (TextView) findViewById(R.id.sensor_x_position_lbl);
        tvSensorYLbl = (TextView) findViewById(R.id.sensor_y_position_lbl);
        tvSensorZLbl = (TextView) findViewById(R.id.sensor_z_position_lbl);
        tvResidualVolumeLbl = (TextView) findViewById(R.id.residual_tank_volume_lbl);
        tvWallThicknessLbl = (TextView) findViewById(R.id.wall_thickness_lbl);
        tvTableHeightStepSizeLbl = (TextView) findViewById(R.id.table_height_step_size_lbl);
        tvTableVolumeStepSizeLbl = (TextView) findViewById(R.id.table_volume_step_size_lbl);

        //helpWebView = (WebView) findViewById(R.id.help_view);
        //setupWebView(helpWebView);
        //tableWebView = (WebView) findViewById(R.id.compute_display);
        //setupWebView(tableWebView);
        //summaryWebView = (WebView) findViewById(R.id.summary_display);
        //setupWebView(summaryWebView);

        graphicInverse = setupCheckBox(R.id.check_graphic_inverse);
        graphicAnaglyphic = setupCheckBox(R.id.check_graphic_analgyph);
        graphicHints = setupCheckBox(R.id.check_graphic_hints);
        graphicTransparent = setupCheckBox(R.id.check_transparent);
        graphicDrawLeftCap = setupCheckBox(R.id.check_draw_leftCap);
        graphicDrawCylinder = setupCheckBox(R.id.check_draw_cylinder);
        graphicDrawRightCap = setupCheckBox(R.id.check_draw_rightCap);
        graphicDrawSensor = setupCheckBox(R.id.check_draw_sensor);

    }

    //@SuppressLint("SetJavaScriptEnabled")
    protected void updateWebView(WebView wv) {
        //Log.e("UpdateWebView","" + wv + " <--> " + helpWebView);
        WebSettings set = wv.getSettings();
        set.setBuiltInZoomControls(true);
        set.setJavaScriptEnabled(true);
        set.setDefaultTextEncodingName("utf-8");
        if (wv == helpWebView) {
            //Log.e("UpdateWebView: help","" + wv);
            wv.loadDataWithBaseURL("", app.currentHelpPage, "text/html", "UTF-8", "");
        } else if (wv == tableWebView) {
            //Log.e("UpdateWebView: table","" + wv);
            wv.loadDataWithBaseURL("", app.currentHtmlTable, "text/html", "UTF-8", "");
        }
    }

    private EditText setupEditText(int id) {
        EditText et = (EditText) findViewById(id);
        if (et != null) {
            et.addTextChangedListener(new TextWatcher() {

                public void afterTextChanged(Editable arg0) {

                    writeAllControls(false, false);
                }

                public void beforeTextChanged(CharSequence s, int start,
                                              int count, int after) {


                }

                public void onTextChanged(CharSequence s, int start,
                                          int before, int count) {


                }

            });
        }
        return et;
    }

    private CheckBox setupCheckBox(int resource) {
        CheckBox cb = (CheckBox) findViewById(resource);
        cb.setOnCheckedChangeListener(new OnCheckedChangeListener() {
            public void onCheckedChanged(CompoundButton buttonView,
                                         boolean isChecked) {
                writeAllControls(false, false);
            }
        });
        return cb;
    }

    private void setupLists() {
        /*
         * lengthUnits = new HashMap<String, Double>() { { put("Millieters",
		 * 0.0); put("Centimeters", 0.0); put("Meters", 0.0); put("Inches",
		 * 0.0); put("Feet", 0.0); } };
		 * 
		 * volumeUnits = new HashMap<String, Double>() { { put("Liters", 0.0);
		 * put("Gallons", 0.0); put("Millimeters³", 0.0); put("Centimeters³",
		 * 0.0); put("Meters³", 0.0); put("Inches³", 0.0); put("Feet³", 0.0); }
		 * };
		 * 
		 * weightUnits = new HashMap<String, Double>() { { put("Pounds", 0.0);
		 * put("Kilograms", 0.0); } };
		 */

        setupSpinner(lengthUnitSpinner, app.constants.strLengthUnits);
        setupSpinner(volumeUnitSpinner, app.constants.strVolumeUnits);
        setupSpinner(tankOrientationSpinner, app.constants.strTankOrientation);
        setupSpinner(tankLrTypeSpinner, app.constants.strCapTypes);
        setupSpinner(tankRrTypeSpinner, app.constants.strCapTypes);
        setupSpinner(tankSensorAxisSpinner, app.constants.strSensorAxis);
        setupSpinner(tankAngleUnitSpinner, app.constants.strAngleUnits);
        setupSpinner(weightUnitSpinner, app.constants.strWeightUnits);
        setupSpinner(decimalPlacesSpinner, new String[]{"0", "1", "2", "3",
                "4", "5", "6", "7", "8", "9"});

    }

    private void setupSpinner(final Spinner sp, String[] items) {
        ArrayAdapter<String> aa = new ArrayAdapter<String>(this,
                android.R.layout.simple_spinner_item, items);
        aa.setDropDownViewResource(android.R.layout.simple_spinner_dropdown_item);
        sp.setAdapter(aa);
        sp.setOnItemSelectedListener(new AdapterView.OnItemSelectedListener() {
            public void onItemSelected(AdapterView<?> adapterView, View view,
                                       int i, long l) {
                writeAllControls(true, true);
                if (sp == decimalPlacesSpinner && initialized) {
                    readAllControls();
                }
            }

            public void onNothingSelected(AdapterView<?> adapterView) {
                return;
            }
        });
    }

    private void writeAllControls(boolean isSpinner, boolean updateLabels) {

        if (initialized) {
            //Log.e("writeAllControls", "writing: " + isSpinner + "," +
            // updateLabels);
            app.sharedTP.currentTab = tabLayout.getSelectedTabPosition();
            if (isSpinner) {
                app.sharedTP.sensorOrientation = getSpinnerIndex(tankSensorAxisSpinner);
                app.sharedTP.tankOrientation = getSpinnerIndex(tankOrientationSpinner);
                app.sharedTP.tankAngleUnitIndex = getSpinnerIndex(tankAngleUnitSpinner);
                app.sharedTP.rightEndCapMode = getSpinnerIndex(tankRrTypeSpinner);
                app.sharedTP.leftEndCapMode = getSpinnerIndex(tankLrTypeSpinner);
                app.sharedTP.inputUnitIndex = getSpinnerIndex(lengthUnitSpinner);
                app.sharedTP.outputUnitIndex = getSpinnerIndex(volumeUnitSpinner);
                app.sharedTP.weightUnitIndex = getSpinnerIndex(weightUnitSpinner);
                app.sharedTP.decimalPlaces = getSpinnerIndex(decimalPlacesSpinner);
            } else { // is EditText
                app.sharedTP.g_L = readControlNumber(tvTankLValue);
                app.sharedTP.g_R = readControlNumber(tvTankRValue);
                app.sharedTP.g_rL = readControlNumber(tvTankLrValue);
                app.sharedTP.g_rR = readControlNumber(tvTankRrValue);
                app.sharedTP.enteredAngleDegrees = readControlNumber(tvTankAngle);
                app.sharedTP.scalingFactor = readControlNumber(tvScalingFactor);
                app.sharedTP.sensorPositionX = readControlNumber(tvSensorX);
                app.sharedTP.sensorPositionY = readControlNumber(tvSensorY);
                app.sharedTP.sensorPositionZ = readControlNumber(tvSensorZ);
                app.sharedTP.residualVolume = readControlNumber(tvResidualVolume);
                app.sharedTP.wallThickness = readControlNumber(tvWallThickness);
                app.sharedTP.wallDensity = readControlNumber(tvWallDensity);
                app.sharedTP.contentDensity = readControlNumber(tvContentDensity);
                app.sharedTP.endCapIntegrationSteps = (int) readControlNumber(tvCapIntSteps);
                app.sharedTP.cylinderIntegrationSteps = (int) readControlNumber(tvCylIntSteps);
                app.sharedTP.rootFinderEpsilon = readControlNumber(tvRootFinderEpsilon);
                app.sharedTP.tableHeightStepSize = readControlNumber(tvTableHeightStepSize);
                app.sharedTP.tableVolumeStepSize = readControlNumber(tvTableVolumeStepSize);
                //app.sharedTP.emailAddress = readControlText(tvEmailAddress);
                app.sharedTP.graphicInverseMode = readCheckBox(graphicInverse);
                app.sharedTP.graphicAnaglyphMode = readCheckBox(graphicAnaglyphic);
                app.sharedTP.graphicHints = readCheckBox(graphicHints);
                app.sharedTP.graphicTransparent = readCheckBox(graphicTransparent);
                app.sharedTP.graphicDrawLeftCap = readCheckBox(graphicDrawLeftCap);
                app.sharedTP.graphicDrawCylinder = readCheckBox(graphicDrawCylinder);
                app.sharedTP.graphicDrawRightCap = readCheckBox(graphicDrawRightCap);
                app.sharedTP.graphicDrawSensor = readCheckBox(graphicDrawSensor);
            }
            if (updateLabels) {
                writeAllControlLabels();
            }
            writeProperties(false);
            updateGraphicDisplays();
        }
    }

    double getDouble(String s) {
        return Double.parseDouble(s);
    }

    private double readControlNumber(TextView tv) {
        double v = 0.0;
        try {
            String s = tv.getText().toString();
            v = getDouble(s);
        } catch (Exception e) {
            // app.showStackTrace(e);
        }
        return v;
    }

    private String readControlText(TextView tv) {
        return (String) tv.getText().toString().trim();
    }

    private boolean readCheckBox(CheckBox cb) {
        return cb.isChecked();
    }

    private void readAllControls() {
        //Log.e("readAllControls", "reading...");

        boolean old_init = initialized;
        initialized = false;
        writeControlNumber(tvTankLValue, app.sharedTP.g_L);
        writeControlNumber(tvTankRValue, app.sharedTP.g_R);
        writeControlNumber(tvTankLrValue, app.sharedTP.g_rL);
        setSpinnerIndex(tankLrTypeSpinner, app.sharedTP.leftEndCapMode);
        writeControlNumber(tvTankRrValue, app.sharedTP.g_rR);
        setSpinnerIndex(tankRrTypeSpinner, app.sharedTP.rightEndCapMode);
        setSpinnerIndex(tankOrientationSpinner, app.sharedTP.tankOrientation);
        writeControlNumber(tvTankAngle, app.sharedTP.enteredAngleDegrees);
        writeControlNumber(tvScalingFactor, app.sharedTP.scalingFactor);
        setSpinnerIndex(tankAngleUnitSpinner, app.sharedTP.tankAngleUnitIndex);
        setSpinnerIndex(tankSensorAxisSpinner, app.sharedTP.sensorOrientation);
        writeControlNumber(tvSensorX, app.sharedTP.sensorPositionX);
        writeControlNumber(tvSensorY, app.sharedTP.sensorPositionY);
        writeControlNumber(tvSensorZ, app.sharedTP.sensorPositionZ);
        writeControlNumber(tvResidualVolume, app.sharedTP.residualVolume);

        writeControlNumberInt(tvCapIntSteps, app.sharedTP.endCapIntegrationSteps);

        writeControlNumberInt(tvCylIntSteps, app.sharedTP.cylinderIntegrationSteps);
        writeScientificNumber(tvRootFinderEpsilon, app.sharedTP.rootFinderEpsilon);

        writeControlNumber(tvTableHeightStepSize, app.sharedTP.tableHeightStepSize);
        writeControlNumber(tvTableVolumeStepSize, app.sharedTP.tableVolumeStepSize);
        //writeControlString(tvEmailAddress, app.sharedTP.emailAddress);

        writeCheckBox(graphicAnaglyphic, app.sharedTP.graphicAnaglyphMode);
        writeCheckBox(graphicInverse, app.sharedTP.graphicInverseMode);
        writeCheckBox(graphicHints, app.sharedTP.graphicHints);
        writeCheckBox(graphicTransparent, app.sharedTP.graphicTransparent);
        writeCheckBox(graphicDrawLeftCap, app.sharedTP.graphicDrawLeftCap);
        writeCheckBox(graphicDrawCylinder, app.sharedTP.graphicDrawCylinder);
        writeCheckBox(graphicDrawRightCap, app.sharedTP.graphicDrawRightCap);
        writeCheckBox(graphicDrawSensor, app.sharedTP.graphicDrawSensor);

        writeControlNumber(tvWallThickness, app.sharedTP.wallThickness);
        writeControlNumber(tvWallDensity, app.sharedTP.wallDensity);
        writeControlNumber(tvContentDensity, app.sharedTP.contentDensity);
        setSpinnerIndex(lengthUnitSpinner, app.sharedTP.inputUnitIndex);
        setSpinnerIndex(volumeUnitSpinner, app.sharedTP.outputUnitIndex);
        setSpinnerIndex(weightUnitSpinner, app.sharedTP.weightUnitIndex);
        setSpinnerIndex(decimalPlacesSpinner, app.sharedTP.decimalPlaces);
        writeAllControlLabels();
        initialized = old_init;

    }

    private void writeControlNumber(TextView tv, double value) {
        tv.setText(String.format("%." + app.sharedTP.decimalPlaces + "f", value));
    }

    private void writeScientificNumber(TextView tv, double value) {
        tv.setText(String.format("%." + app.sharedTP.decimalPlaces + "e", value));
    }

    private void writeControlNumberInt(TextView tv, int value) {
        tv.setText(String.format("%d", value));
    }

    private void writeControlString(TextView tv, String value) {
        tv.setText(value.trim());
    }

    private void writeCheckBox(CheckBox cb, boolean state) {
        cb.setChecked(state);
    }

    private void writeAllControlLabels() {
        //Log.e("writeAllControlLabels", "writing...");
        String lu = app.constants.strLengthUnits[app.sharedTP.inputUnitIndex];
        String vu = app.constants.strVolumeUnits[app.sharedTP.outputUnitIndex];
        String au = app.constants.strAngleUnits[app.sharedTP.tankAngleUnitIndex];
        String s = String.format("Cylinder length (L) %s", lu);
        tvTankLValueLbl.setText(s);
        s = String.format("Cylinder radius (R) %s", lu);
        tvTankRadiusLbl.setText(s);
        s = String.format("Left cap radius (r) %s", lu);
        tvTankLrValueLbl.setText(s);
        s = String.format("Right cap radius (r) %s", lu);
        tvTankRrValueLbl.setText(s);
        s = String.format("Tank angle (%s)", au);
        tvTankAngleLbl.setText(s);
        s = String.format("Sensor X position %s", lu);
        tvSensorXLbl.setText(s);
        s = String.format("Sensor Y position %s", lu);
        tvSensorYLbl.setText(s);
        s = String.format("Sensor Z position %s", lu);
        tvSensorZLbl.setText(s);
        s = String.format("Residual volume %s", vu);
        tvResidualVolumeLbl.setText(s);
        s = String.format("Wall thickness %s", lu);
        tvWallThicknessLbl.setText(s);
        // must check for inverse case
        // s = String.format("Table step size %s", lu);
        // tvTableStepSizeLbl.setText(s);

    }

    // private void setSpinnerItem(Spinner sp, String item) {
    // ArrayAdapter<String> aa = (ArrayAdapter<String>) sp.getAdapter();
    // sp.setSelection(aa.getPosition(item));
    // }

    private void setSpinnerIndex(Spinner sp, int index) {
        sp.setSelection(index);
    }

    // private String getSpinnerItem(Spinner sp) {
    // return (String) sp.getSelectedItem();
    // }

    private int getSpinnerIndex(Spinner sp) {
        return sp.getSelectedItemPosition();
    }

    private void setupTabs() {
        int resources[] = {R.id.description_view, R.id.compute_view,
                R.id.display_view, R.id.help_view};
        //if (tabHost != null) {
        //    tabHost.setup();
        //    TabHost.TabSpec spec;
        //    for (int i = 0; i < app.constants.strTabLabels.length; i++) {
        //        spec = tabHost.newTabSpec(app.constants.strTabLabels[i]);
        //        spec.setIndicator(app.constants.strTabLabels[i]);
        //        spec.setContent(resources[i]);
        //        tabHost.addTab(spec);
        //        tabHost.setCurrentTab(app.sharedTP.currentTab);
        //        tabHost.setOnTabChangedListener(new OnTabChangeListener() {

        //            public void onTabChanged(String tabId) {
        //                tabChangeActions();
        //            }
        //        });
        //    }
        //    for (int i = 0; i < tabHost.getTabWidget().getChildCount(); i++) {
        //        TextView tv = (TextView) tabHost.getTabWidget().getChildAt(i)
        //                .findViewById(android.R.id.title); // Unselected Tabs
        //        tv.setTextColor(Color.GREEN);
        //    }
        //}
    }

    private void tabChangeActions() {
        //app.sharedTP.currentTab = tabHost.getCurrentTab();
        if (app.sharedTP.currentTab == AndroidConstants.TABTABLE) {
            buildSummary(null);
        }
        updateGraphicDisplays();

    }

    private void updateGraphicDisplays() {
        app.imageGenHi.rebuildModel(app.sharedTP);
        app.imageGenLo.rebuildModel(app.sharedTP);
        menuView.invalidate();
        imageView.invalidate();
        // tabHost.invalidate();
    }

    private void resetToDefaults() {
        app.sharedTP = new TankProperties();
        //app.sharedTP = app.sharedTP;
        readAllControls();
        writeProperties(false);
        updateGraphicDisplays();
    }

    public void resetToDefaults(View v) {
        String title = "Reset to Defaults";
        String message = "Do you really want to reset all values to their defaults?";
        FunctionInterface f = new FunctionInterface() {
            public void function() {
                resetToDefaults();
            }
        };
        actionDialog(v, title, message, f);
    }

    public void actionDialog(View v, String title, String message, final FunctionInterface f) {
        new AlertDialog.Builder(this)
                .setTitle(title)
                .setMessage(message)
                .setIcon(R.mipmap.app_icon)
                .setPositiveButton(android.R.string.yes,
                        new DialogInterface.OnClickListener() {

                            public void onClick(DialogInterface dialog,
                                                int whichButton) {
                                f.function();
                            }
                        })
                .setNegativeButton(android.R.string.no, null).show();
    }

    public void buildSummary(View v) {
        app.tableGen.buildSummaryTable();
    }

    public void buildVolumeTable(View v) {
        app.tableGen.buildTable(false);
    }

    public void buildHeightTable(View v) {
        app.tableGen.buildTable(true);
    }

    private void shareTable(String title, String data,String type) {

        try {
            File file;
            File base = Environment.getExternalStorageDirectory();
            File dir = new File(base, "Documents/TankCalcAndroid");
            dir.mkdirs();
            file = new File(dir, "TankCalcAndroidData." + type);
            BufferedWriter out = new BufferedWriter(new FileWriter(file));
            out.write(data);
            out.close();
            Uri uri = Uri.fromFile(file);
            Intent sendIntent = new Intent(Intent.ACTION_SEND);
            //Uri uri = Uri.parse("file://" + path);
            sendIntent.putExtra(Intent.EXTRA_STREAM, uri);
            sendIntent.setType("text/" + type);
            //String title = String.format("This is a %s data table from TankCalcAndroid", type.toUpperCase());
            //startActivity(Intent.createChooser(sendIntent, title));
            //sendIntent.putExtra(Intent.EXTRA_SUBJECT, msg);
            //sendIntent.putExtra(android.content.Intent.EXTRA_TEXT, msg);
            sendIntent.setType("text/" + type);
            startActivity(Intent.createChooser(sendIntent, title));
            //startActivity(sendIntent);
        } catch (Exception e) {
            //Log.e("Error:","shareTable: " + e.toString());
            permissionsDialog();
        }
    }

    protected void permissionsDialog() {
        AlertDialog ad = new AlertDialog.Builder(this).create();
        ad.setTitle("Enable Storage for TankCalcAndroid");
        ad.setIcon(R.mipmap.app_icon);
        ad.setMessage("To use the file saving feature, TankCalcAndroid needs permission to save files. In the next screen, choose \"Permissions\" and enable the \"Storage\" option. Press OK to proceed.");
        ad.setButton(Dialog.BUTTON_POSITIVE, "OK", new DialogInterface.OnClickListener() {
            public void onClick(DialogInterface dialog, int which) {
                Intent intent = new Intent(Settings.ACTION_APPLICATION_DETAILS_SETTINGS,
                        Uri.fromParts("package", getPackageName(), null));
                intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
                startActivity(intent);
            }
        });
        ad.setButton(Dialog.BUTTON_NEGATIVE, "Cancel", new DialogInterface.OnClickListener()

        {
            public void onClick(DialogInterface dialog, int which) {
            }
        });
        ad.show();
    }

    private void sendCSVEmail(String msg, String data) {

        try {
            File file;
            File base = Environment.getExternalStorageDirectory();
            File dir = new File(base, "publicCache");
            dir.mkdirs();
            file = new File(dir, "TankCalcAndroidData.csv");
            BufferedWriter out = new BufferedWriter(new FileWriter(file));
            out.write(data);
            out.close();
            Uri emailUri = Uri.fromFile(file);
            Intent sendIntent = new Intent(Intent.ACTION_SEND);
            sendIntent.putExtra(android.content.Intent.EXTRA_EMAIL,
                    new String[]{app.sharedTP.emailAddress});
            sendIntent.putExtra(Intent.EXTRA_SUBJECT, "TankCalcAndroid Data");
            sendIntent.putExtra(android.content.Intent.EXTRA_TEXT, msg);
            sendIntent.putExtra(Intent.EXTRA_STREAM, emailUri);
            sendIntent.setType("text/csv");
            startActivity(sendIntent);
        } catch (Exception e) {
            app.showStackTrace(e);
        }
    }

    public void sendHTMLTable(View v) {
        if (!app.tableGen.validTable) {
            userAlertDialog("No Table Data","Please generate a table first.");
        } else {
            shareTable("This is a TankCalcAndroid HTML data table",
                    app.currentHtmlTable,"html");
        }
    }

    public void sendCSVTable(View v) {
        if (!app.tableGen.validTable) {
            userAlertDialog("No Table Data","Please generate a table first.");
        } else {
            shareTable("This is a TankCalcAndroid CSV data table",
                    app.currentCsvTable,"csv");
        }
    }

    public void userAlertDialog(String title,String message) {
        new AlertDialog.Builder(this)
                .setTitle(title)
                .setMessage(message)
                .setIcon(R.mipmap.app_icon)
                .setPositiveButton(android.R.string.yes, new DialogInterface.OnClickListener() {
                    public void onClick(DialogInterface dialog, int which) {

                    }
                })
                .show();
    }

    private String loadLocalAsset(String name) {
        StringBuilder data = new StringBuilder();
        try {
            BufferedReader isr = new BufferedReader(new InputStreamReader(getAssets().open(
                    name)));
            String line;

            while ((line = isr.readLine()) != null) {
                line = line.trim();
                if (line.length() > 0) {
                    data.append(line);
                    data.append("\n");
                }
            }
            isr.close();
        } catch (Exception e) {
            app.showStackTrace(e);
        }
        //Log.e("loadLocalAsset", data.toString());
        return data.toString();
    }

    private void setupHelp() {
        String helpPage = loadLocalAsset("help.html");
        String version;
        try {
            version = getPackageManager().getPackageInfo(getPackageName(), 0).versionName;
            helpPage = helpPage.replaceAll("#version#", version);
        } catch (NameNotFoundException e) {
            e.printStackTrace();
        }
        app.currentHelpPage = helpPage;
    }

    private void serialize() {
        app.sharedTP.currentTab = mViewPager.getCurrentItem();
        writeAllControls(true, false);
        writeAllControls(false, false);
        app.serialize();
    }

    protected void changeTabs(final int tab) {
        if (tab >= 0) {
            mViewPager.setCurrentItem(tab, true);
        }
    }

    @Override
    public void onStart() {
        super.onStart();
    }

    @Override
    public void onResume() {
        super.onResume();
        afterResume();
    }

    @Override
    public void onStop() {
        serialize();
        super.onStop();
    }

    @Override
    public void onPause() {
        serialize();
        super.onPause();
    }

    @Override
    public void onDestroy() {
        // end of this view
        serialize();
        super.onDestroy();
    }

    public ArrayList<String[]> tankSummary() {
        writeProperties(false);
        ArrayList<String[]> al = new ArrayList<String[]>();
        double iuToLength = app.tankProcessor.convertLengthUnits(1.0, true);
        double iuToArea = iuToLength * iuToLength;
        double iuToVolume = iuToArea * iuToLength;
        String weightUnit = app.constants.strWeightUnits[app.sharedTP.weightUnitIndex];
        String ius = app.constants.strLengthUnits[app.sharedTP.inputUnitIndex];
        String vius = app.constants.strVolumeUnits[app.sharedTP.inputUnitIndex];
        String sius = vius.replaceAll("\\^3", "&sup2;");
        String cius = vius.replaceAll("\\^3", "&sup3;");
        double ftv = app.tankProcessor.tankFullVolume().volume * app.sharedTP.scalingFactor;
        double fcsa = app.tankProcessor.cylinderFullSurfaceArea() * app.sharedTP.scalingFactor; // meters^2
        double lca = app.tankProcessor.endCapFullSurfaceArea(true) * app.sharedTP.scalingFactor; // meters^2
        double rca = app.tankProcessor.endCapFullSurfaceArea(false) * app.sharedTP.scalingFactor; // meters^2
        //double wt = sharedTP.wallThickness;
        //double wt_conv = tankProcessor.convertLengthUnits(wt, false);
        double wd = app.sharedTP.wallDensity;
        double cd = app.sharedTP.contentDensity;
        // total surface area
        double tsa = fcsa + lca + rca; // meters^2
        // this was wrong
        //double twv = wt_conv * tsa; //  total wall volume in meters^3
        // subtract two volumes instead
        double twv = app.tankProcessor.tankFullVolumeWithWallThickness().volume - app.tankProcessor.tankFullVolume().volume;
        twv *= app.sharedTP.scalingFactor;
        double tm = twv * wd * 1e6; // grams
        tm /= 1000.0; // kilograms
        if (app.sharedTP.weightUnitIndex == Constants.WEIGHTPOUNDS) { // if
            // pounds
            tm *= 2.2046244;
        }
        double contentw = ftv * cd * 1e6; // grams
        contentw /= 1000.0; // kilogram
        if (app.sharedTP.weightUnitIndex == Constants.WEIGHTPOUNDS) { // pounds
            contentw *= 2.2046244;
        }
        assembleRecord(al, "Name", "Value", "Unit");
        assembleRecord(al, "Input units",
                app.constants.strLengthUnits[app.sharedTP.inputUnitIndex], "");
        assembleRecord(al, "Output units",
                app.constants.strVolumeUnits[app.sharedTP.outputUnitIndex], "");
        String s = app.constants.strTankOrientation[app.sharedTP.tankOrientation];
        if (app.sharedTP.tankOrientation == Constants.TANKTILTED) {
            s += " (" + formatNum(app.sharedTP.angleDegrees) + "°)";
        }
        assembleRecord(al, "Tank orientation", s, "");

        assembleRecord(al, "Cylinder length (L)", formatNum(app.sharedTP.g_L), ius);
        assembleRecord(al, "Cylinder radius (R)", formatNum(app.sharedTP.g_R), ius);
        assembleRecord(al, "Left/bottom cap radius (r)",
                formatNum(app.sharedTP.g_rL), ius);
        assembleRecord(al, "Right/top cap radius (r)",
                formatNum(app.sharedTP.g_rR), ius);
        assembleRecord(al, "Scaling factor",
                formatNum(app.sharedTP.scalingFactor), "(width/height ratio)");
        assembleRecord(al, "Tank internal height",
                formatNum(app.tankProcessor.convertLengthUnits(
                        app.tankProcessor.maxHeight, true)), ius);
        assembleRecord(al, "Sensor vertical offset",
                formatNum(app.tankProcessor.convertLengthUnits(
                        app.tankProcessor.trueSensorOffset, true)), ius);

        assembleRecord(al, "Cylinder surface area",
                convertDimFormat(fcsa, iuToArea), sius);
        assembleRecord(al, "Left/bottom cap surface area",
                convertDimFormat(lca, iuToArea), sius);
        assembleRecord(al, "Right/top cap surface area",
                convertDimFormat(rca, iuToArea), sius);
        assembleRecord(al, "Tank surface area (cylinder+caps)",
                convertDimFormat(tsa, iuToArea), sius);
        assembleRecord(al, "Tank wall thickness",
                formatNum(app.sharedTP.wallThickness), ius);

        assembleRecord(al, "Tank wall density", formatNum(wd), "(water=1)");
        assembleRecord(al, "Tank content density", formatNum(cd), "(water=1)");
        assembleRecord(al, "Tank wall volume (input units^3)",
                convertDimFormat(twv, iuToVolume), cius);
        assembleRecord(al, "Tank interior volume (input units^3)",
                convertDimFormat(ftv, iuToVolume), cius);
        assembleRecord(al, "Tank interior volume (output units)",
                formatNum(app.tankProcessor.convertVolumeUnits(ftv, true)),
                app.constants.strVolumeUnits[app.sharedTP.outputUnitIndex]);
        assembleRecord(al, "Tank content weight (full tank)",
                formatNum(contentw), weightUnit);
        assembleRecord(al, "Tank empty weight", formatNum(tm), weightUnit);
        assembleRecord(al, "Tank gross weight (tank+contents)",
                formatNum(contentw + tm), weightUnit);
        return al;
    }

    String formatNum(double v) {
        if (Double.isNaN(v)) {
            return " -- ";
        }
        return String.format("%." + app.sharedTP.decimalPlaces + "f", v);
    }

    //String convertLengthFormat(double v) {
    //	return formatNum(tankProcessor.convertLengthUnits(v, true));
    //}

    String convertDimFormat(double v, double scale) {
        return formatNum(v * scale);
    }

    private void assembleRecord(ArrayList<String[]> al, String... s) {
        al.add(s);
    }

    void writeProperties(boolean inverse) {
        app.sharedTP.inverseMode = inverse;
        app.tankProcessor.setValues(this, app.sharedTP);
        app.serialize();
    }

    public ArrayList<String[]> buildDataTable(boolean inverse) {
        writeProperties(inverse);
        // boolean makeTableFlag = true;
        // setModeAndLabels();

        // if (getEnteredValues()) {
        ArrayList<String[]> stringList = new ArrayList<String[]>();
        // tableTextPane.setText("");
        // tableProgressBar.setStringPainted(true);
        double step = (app.sharedTP.inverseMode) ? app.sharedTP.tableVolumeStepSize
                : app.sharedTP.tableHeightStepSize;
        if (!Double.isNaN(step)) {
            double max, y;
            // StringBuilder sb = new StringBuilder();
            // create table title strings
            String u1 = getUnitLabel(false, true, false, true, true);
            String u2 = app.constants.strAreaUnits[app.sharedTP.inputUnitIndex];
            String u3 = getUnitLabel(false, false, false, false, true);
            String a, b;

            if (app.sharedTP.inverseMode) {
                b = "Height " + u1;
                a = "LV " + u3;

            } else {
                a = "Height " + u1;
                b = "LV " + u3;
            }
            String c = "LWA " + u2;
            String d = "LSA " + u2;
            String e = "%";
            // String a = getUnitLabel(true, true, inverseMode, true, true);
            // String b = getUnitLabel(true, false, inverseMode, false, true);
            // String c = "%";
            if (app.sharedTP.inverseMode) {
                b = "Sensor " + b;
            } else {
                a = "Sensor " + a;
            }

            if (app.sharedTP.inverseMode) {
                step = app.tankProcessor.convertVolumeUnits(step, false);
                max = app.tankProcessor.computeMaxVolume().volume * app.sharedTP.scalingFactor;
            } else {
                step = app.tankProcessor.convertLengthUnits(step, false);
                max = app.tankProcessor.computeFullTankHeight();
                max /= app.tankProcessor.sensorScalingFactor();
            }
            if (Double.isNaN(max) || Math.abs(max) > 1e10) {
                // beep();
                // JOptionPane.showMessageDialog(this,
                // "Sensor path anomaly -- cannot create table",
                // "Table Generation Error", JOptionPane.INFORMATION_MESSAGE);
            } else {
                // tableTitle = new String[] { a, b, c };
                tableTitle = new String[]{a, b, c, d, e};
                // generate table data rows
                // int n = 0;
                dblArray = new ArrayList<Double[]>();
                stringList.add(tableTitle);
                double maxVal = 1;
                computeLine(dblArray, max, maxVal);
                maxVal = dblArray.get(0)[1];
                // dblArray.clear();
                dblArray = new ArrayList<Double[]>();
                String[] array = {};
                for (int iy = 0; ((y = iy * step) <= max) && taskRunning; iy++) {
                    array = computeLine(dblArray, y, maxVal);
                    stringList.add(array);
                }
                String[] lastLine = array;
                array = computeLine(dblArray, max, maxVal);
                stringList.add(array);
                if (taskRunning) {
                    // sometimes the "max" value line is identical to the
                    // prior line, don't include if this is so
                    if (stringArraysMatch(lastLine, array)) {
                        stringList.remove(stringList.size() - 1);
                        dblArray.remove(dblArray.size() - 1);
                    }
                } else {
                    stringList = null;
                }
            }
        }
        return stringList;
    }

    String getUnitLabel(boolean hvLabel, boolean height, boolean inverse,
                        boolean input, boolean capitalize) {
        String result = "";
        if (hvLabel) {
            result += ((height ^ inverse) ? "Height " : "Volume ");
        }
        result += ((input ^ inverse) ? app.constants.strLengthUnits[app.sharedTP.inputUnitIndex]
                : app.constants.strVolumeUnits[app.sharedTP.outputUnitIndex]);
        if (!capitalize) {
            result = result.toLowerCase();
        }
        return result;
    }

    boolean stringArraysMatch(String[] a, String[] b) {
        if (a != null && b != null && a.length > 0 && a.length == b.length) {
            int i = 0;
            for (String s : a) {
                if (!s.equals(b[i])) {
                    return false;
                }
                i++;
            }
            return true;
        } else {
            return false;
        }
    }

    String[] computeLine(ArrayList<Double[]> array, double y, double max) {
        double x;
        TankCompResult z = app.tankProcessor.computeHeightOrVolume(y);
        if (Double.isNaN(z.volume)) {
            if (!app.sharedTP.inverseMode) {
                z = app.tankProcessor.tankFullVolume();
                z.volume *= app.sharedTP.scalingFactor;
            } else {
                z.volume = app.tankProcessor.maxHeight;
            }
        }
        if (app.sharedTP.inverseMode) {
            x = app.tankProcessor.convertLengthUnits(z.height, true);
            y = app.tankProcessor.convertVolumeUnits(y, true);
        } else {
            x = app.tankProcessor.convertVolumeUnits(z.volume, true);
            y = app.tankProcessor.convertLengthUnits(y, true);
        }
        z.wettedArea = app.tankProcessor.convertAreaUnits(z.wettedArea, true);
        z.surfaceArea = app.tankProcessor.convertAreaUnits(z.surfaceArea, true);
        if (array != null) {
            array.add(new Double[]{y, z.volume, z.wettedArea, z.surfaceArea});
        }
        String a = formatNum(y);
        String b = formatNum(x);
        String c = formatNum(z.wettedArea);
        String d = formatNum(z.surfaceArea);
        String pct = formatNum(z.volume * 100.0 / max);
        return new String[]{a, b, c, d, pct};
    }

    /**
     * Returns a new instance of fragment for the given section
     * number.
     */
    public PlaceholderFragment newFragment(int sectionNumber) {
        //Log.e("newFragment: ", "" + sectionNumber);
        PlaceholderFragment fragment = new PlaceholderFragment();
        Bundle args = new Bundle();
        args.putInt("index", sectionNumber);
        // always use default setArguments()
        fragment.setArguments(args);
        return fragment;
    }


    /**
     * A {@link FragmentPagerAdapter} that returns a fragment corresponding to
     * one of the sections/tabs/pages.
     */
    public class SectionsPagerAdapter extends FragmentPagerAdapter {

        public SectionsPagerAdapter(FragmentManager fm) {
            super(fm);
        }

        @Override
        public Fragment getItem(int position) {
            // getItem is called to instantiate the fragment for the given page.
            // Return a PlaceholderFragment (defined as a static inner class below).
            //Log.e("getItem","position = " + position);
            return newFragment(position);
        }

        @Override
        public int getCount() {

            return 4;
        }

        @Override
        public CharSequence getPageTitle(int position) {
            switch (position) {
                case 0:
                    return "Setup";
                case 1:
                    return "Table";
                case 2:
                    return "Model";
                case 3:
                    return "Help";
            }
            return null;
        }
    }

    @Override
    public boolean onCreateOptionsMenu(Menu menu) {
        // Inflate the menu; this adds items to the action bar if it is present.
        getMenuInflater().inflate(R.menu.menu_main, menu);
        return true;
    }

    @Override
    public boolean onOptionsItemSelected(MenuItem item) {
        // Handle action bar item clicks here. The action bar will
        // automatically handle clicks on the Home/Up button, so long
        // as you specify a parent activity in AndroidManifest.xml.
        int id = item.getItemId();

        //noinspection SimplifiableIfStatement
        if (id == R.id.action_settings) {
            return true;
        }

        return super.onOptionsItemSelected(item);
    }

}