/***************************************************************************
 *   Copyright (C) 2011 by Paul Lutus                                      *
 *   http://arachnoid.com/administration                                   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

package com.arachnoid.carpetempus;

import java.io.Serializable;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.SimpleTimeZone;
import java.util.TimeZone;

import android.text.format.DateFormat;

public abstract class GenericEvent implements Serializable {
	private static final long serialVersionUID = -4395820892173840876L;
	protected long timeToGo = 0;
	protected long oldTime = -1;
	final protected long snoozeTimeValueSec = 300; // five minutes
	private long snoozeTimeSec = 0;
	protected long eventTimeUTCSec = 0;
	protected long lastAlarmTime = -1;
	protected boolean active = false;
	private boolean voice = true;
	private boolean tone = false;
	private boolean specDate = false;
	private boolean specWeekDays = false;
	private boolean specMonthDays = false;
	protected boolean recurring = false;
	protected int eventIndex = -1;
	// weekdayBits has bits for days of the week
	// Sunday = 1, Monday = 2 ... Saturday = 64
	private int weekDayBits = 0;
	// monthDayBits follows the same pattern as
	// weekDayBits, but 31 bits long
	private int monthDayBits = 0;
	// default alarm repetitions
	private int alarmRepetitions = 5;
	protected int alarmDM;
	protected int alarmDDM;
	protected int alarmDDW;
	protected int alarmDY;
	protected int alarmTH;
	protected int alarmTM;
	protected int alarmTS;
	String[] ampmStrings = new String[] { "am", "pm" };

	private String message = "";

	public GenericEvent(long wSec, String m) {
		setEventTimeUTCSec(wSec);
		message = m;
	}

	public GenericEvent(GenericEvent ge) {
		copyValuesFrom(ge);
	}
	
	protected long getTimeZoneOffsetSec() {
		return CarpeTempusApplication
		.getLocalTimeZoneOffsetSeconds();
	}

	public void copyValuesFrom(GenericEvent ge) {
		setEventTimeUTCSec(ge.eventTimeUTCSec);
		message = ge.message;
		active = ge.active;
		voice = ge.voice;
		tone = ge.tone;
		specDate = ge.specDate;
		specWeekDays = ge.specWeekDays;
		weekDayBits = ge.weekDayBits;
		specMonthDays = ge.specMonthDays;
		monthDayBits = ge.monthDayBits;
		alarmRepetitions = ge.alarmRepetitions;
		recurring = ge.recurring;
		eventIndex = ge.eventIndex;
		resetNow();
	}

	public long getEventTimeUTCSec() {
		return eventTimeUTCSec;
	}

	public abstract long getNextEventTimeUTCSec();

	// must sort events using local time, not UTC

	public long getSortEventTimeLocalSec() {
		long t = eventTimeUTCSec + getTimeZoneOffsetSec();
		return (specDate) ? t : t % 86400L;
	}

	public void setEventTimeUTCSec(long ts) {
		eventTimeUTCSec = ts;
		makeLocalDateTimeFieldsfromEventTime();
		lastAlarmTime = -1;
	}

	public abstract boolean canRecur();

	public long getSnoozeTimeSec() {
		return snoozeTimeSec;
	}

	public void setSnoozeTimeSec(long t) {
		snoozeTimeSec = t;
	}

	public void resetNow() {
		lastAlarmTime = -1;
	}

	public String getNameTag() {
		return "Generic";
	}

	public int getRepetitions() {
		return alarmRepetitions;
	}

	public void setRepetitions(int r) {
		alarmRepetitions = r;
	}

	public int getYear() {
		return alarmDY;
	}

	public void setYear(int y) {
		alarmDY = y;
	}

	public int getMonth() {
		return alarmDM;
	}

	public void setMonth(int m) {
		alarmDM = m;
	}

	public int getDay() {
		return alarmDDM;
	}

	public void setDay(int d) {
		alarmDDM = d;
	}

	public int getHour() {
		return alarmTH;
	}

	public void setHour(int h) {
		alarmTH = h;
	}

	public int getMinute() {
		return alarmTM;
	}

	public void setMinute(int m) {
		alarmTM = m;
	}

	public int getSecond() {
		return alarmTS;
	}

	public void setSecond(int s) {
		alarmTS = s;
	}

	public int getWeekDayBits() {
		return weekDayBits;
	}

	public void setWeekDayBits(int b) {
		weekDayBits = b;
	}

	public int getMonthDayBits() {
		return monthDayBits;
	}

	public void setMonthDayBits(int b) {
		monthDayBits = b;
	}

	public boolean getRecurring() {
		return recurring;
	}

	public void setRecurring(boolean s) {
		recurring = s;
	}

	public boolean getWeekDays() {
		return specWeekDays;
	}

	public void setWeekDays(boolean s) {
		specWeekDays = s;
	}

	public boolean getMonthDays() {
		return specMonthDays;
	}

	public void setMonthDays(boolean s) {
		specMonthDays = s;
	}

	public boolean getSpecDate() {
		return specDate;
	}

	public void setSpecDate(boolean s) {
		specDate = s;
	}

	public void setSnooze(boolean s) {
	}

	public boolean getSnooze() {
		return false;
	}

	public boolean getVoice() {
		return voice;
	}

	public void setVoice(boolean v) {
		voice = v;
	}

	public boolean getTone() {
		return tone;
	}

	public void setTone(boolean v) {
		tone = v;
	}

	public boolean getActive() {
		return active;
	}

	public String getMessage() {
		return message;
	}

	public void setMessage(String m) {
		message = m;
	}

	protected long currentTimeUTCSec() {
		return System.currentTimeMillis() / 1000;
	}
	
	// use today's date if no date tests are enabled
	// this is a hack to try to avoid DST problems
	protected void adjustDaylight(GregorianCalendar gc) {
		//if (!getSpecDate() && !getWeekDays()
		//		&& !getMonthDays()) {
			long gt = gc.getTimeInMillis();
			long nt = System.currentTimeMillis();
			gt = (gt % 86400000L) + (nt - nt % 86400000L);
			gc.setTimeInMillis(gt);
		//}
	}

	protected String secToDateString(CarpeTempusApplication app) {
		GregorianCalendar gc = new GregorianCalendar();
		gc.setTimeInMillis(getEventTimeUTCSec() * 1000);
		java.text.DateFormat df = DateFormat.getDateFormat(app);
		return df.format(gc.getTime());
	}

	public String dtStringFromTimeMS(long timeMS) {
		GregorianCalendar gc = new GregorianCalendar();
		gc.setTimeInMillis(timeMS);
		SimpleDateFormat sdf = new SimpleDateFormat(
				(timeMS < 86400000L) ? "HH:mm:ss.SSS"
						: "yyyy-MM-dd HH:mm:ss.SSS");
		return sdf.format(gc.getTime());
	}

	public String dtFromNow() {
		return dtStringFromTimeMS(System.currentTimeMillis());
	}

	public abstract GenericEvent copy();

	public abstract void setActive(boolean state);

	public abstract boolean alarmTime();

	protected abstract String secToTimeString(CarpeTempusApplication app);

	protected abstract void makeLocalDateTimeFieldsfromEventTime();

	public abstract void readEventDateTime();

}
