/***************************************************************************
 *   Copyright (C) 2011 by Paul Lutus                                      *
 *   http://arachnoid.com/administration                                   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

package com.arachnoid.carpetempus;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.Iterator;
import java.util.List;
import java.util.Timer;
import android.annotation.SuppressLint;
import android.app.Activity;
import android.app.AlarmManager;
import android.app.AlertDialog;
import android.app.AlertDialog.Builder;
import android.app.Application;
import android.app.PendingIntent;
import android.content.Context;
import android.content.Intent;
import android.graphics.Color;
import android.media.AudioManager;
import android.media.ToneGenerator;
import android.os.Build;
import android.os.Handler;
import android.util.Log;
import android.view.View;
import android.view.inputmethod.InputMethodManager;
import android.widget.TextView;
import android.widget.Toast;

import androidx.core.content.ContextCompat;

final public class CarpeTempusApplication extends Application {
    // 2 == show all , 1 = show only alarms, 0 = off
    int DEBUG = 0;
    private static CarpeTempusApplication singleton = null;
    protected SetupBase activity = null;
    SerializedData serialData;
    ArrayList<GenericEvent> syncEvents;
    List<GenericEvent> alarms;
    //GenericEvent pendingAlarm = null;
    int currentIndex = -1;
    Intent alarmIntent = null;
    String PROGRAM_VERSION = "";
    String programTitle = "";
    String serialObjectPath = "CarpeTempus.obj";
    StringBuffer logBuffer = null;
    String debugLogPath = null;
    File appBase;
    Context context = null;
    Intent alarmSchedulerService = null;
    boolean installed = false;
    boolean is24HourFormat = false;
    boolean timerIsActive = false;
    MyRingtone ringtone = null;
    Intent dialogIntent = null;
    Handler handler;
    long oldTimeSec = 0;
    Timer alarmScheduleTimer = null;
    PendingIntent alarmTimeoutIntent = null;
    Runnable displayUpdate = null;
    Runnable timeZoneMonitor = null;
    // check for time zone change at one-hour intervals
    long timeZoneMonitorDelayMillis = 3600 * 1000;

    public CarpeTempusApplication getInstance() {
        return singleton;
    }

    @Override
    public void onCreate() {
        super.onCreate();
        singleton = this;
        context = getApplicationContext();
        set24HourFormat();
        logBuffer = new StringBuffer();
        appBase = getFilesDir().getParentFile();
        debugLogPath = appBase + "/debugLog.txt";
        ringtone = new MyRingtone(this);
        if (DEBUG > 0) {
            // make base directory readable by anyone
            createDir(appBase, 15);
        } else {
            createDir(appBase, 7);
        }
        try {
            PROGRAM_VERSION = getPackageManager().getPackageInfo(
                    getPackageName(), 0).versionName;
            programTitle = "CarpeTempus " + PROGRAM_VERSION;
        } catch (Exception e) {
            logError(e);
        }
        serialData = new SerializedData();
        syncEvents = new ArrayList<GenericEvent>();
        // this populates syncEvents from the serial class
        deSerialize();
        alarms = Collections.synchronizedList(syncEvents);
        if (alarms.size() == 0) {
            int count = 1;
            for (int n = 0; n < count; n++) {
                newTimerButton(null);
            }

            for (int n = 0; n < count; n++) {
                newAlarmButton(null);
            }
        }
        initializeAlarms();
        startAlarmSchedulerIfNeeded();
        setAlarmTimeout();
        setTimeZoneMonitor();
        installed = true;
    }

    protected void startAlarmSchedulerIfNeeded() {
        if (alarmSchedulerService == null) {
            alarmSchedulerService = new Intent(this,
                    AlarmSchedulerService.class);
            alarmSchedulerService
                    .setAction("com.arachnoid.carpetempus.UpdaterServiceManager");

            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
                startForegroundService(alarmSchedulerService);

            } else {
                startService(alarmSchedulerService);
            }
        }
    }

    // check for time zone change at one-hour intervals
    protected void setTimeZoneMonitor() {
        final Handler handler = new Handler();
        timeZoneMonitor = new Runnable() {
            public void run() {
                testForTimeZoneChange();
                handler.postDelayed(this, timeZoneMonitorDelayMillis);
            }
        };
        handler.postDelayed(timeZoneMonitor, 1000);
    }

    // test time zone and update alarm timeout
    // if TZ or DST status has changed
    protected boolean testForTimeZoneChange() {
        long offset = getLocalTimeZoneOffsetSeconds();
        // long lh = offset / 3600;
        boolean changed = (offset != serialData.timeZoneOffsetSeconds);
        //Log.e("testForTimeZoneChange","Offset hours: " + lh + ", changed " +
        // changed);
        // if time zone offset has changed, reset alarms
        if (changed) {
            serialData.timeZoneOffsetSeconds = offset;
            setAlarmTimeout();
        }
        return changed;
    }

    private void setDisplayRefresh(boolean activeTimer) {
        // only update display for an active timer
        if (activeTimer && activity != null) {
            handler = new Handler();
            displayUpdate = new Runnable() {
                public void run() {
                    if (activity != null) {
                        activity.updateDisplay();
                        handler.postDelayed(this, 500);
                    }
                }
            };
            handler.postDelayed(displayUpdate, 0);
        } else {
            if (handler != null && displayUpdate != null) {
                handler.removeCallbacks(displayUpdate);
                displayUpdate = null;
            }
        }
        if (activity != null) {
            activity.updateDisplay();
        }
    }

    protected void cancelAlarmTimeout() {
        // stop any existing timer
        if (alarmTimeoutIntent != null) {
            AlarmManager alarmMgr = (AlarmManager) this
                    .getSystemService(Context.ALARM_SERVICE);
            alarmMgr.cancel(alarmTimeoutIntent);
            alarmTimeoutIntent = null;
            logToFile("Timer cancelled", false);
        } else {
            logToFile("No timer running", false);
        }
        setDisplayRefresh(false);
    }

    protected void setAlarmTimeout() {
        int index = getNextAlarmTime();
        if (index != -1) {
            GenericEvent target = alarms.get(index);
            //Log.e("ZZZ: replacing","was:" + target.getMessage() + " with " + target.getMessage());
            AlarmManager alarmMgr = (AlarmManager) this
                    .getSystemService(Context.ALARM_SERVICE);
            Intent intent = new Intent(this, AlarmHandler.class);
            intent.putExtra("eventIndex",index);
            alarmTimeoutIntent = PendingIntent.getBroadcast(this, 0, intent,
                    PendingIntent.FLAG_CANCEL_CURRENT);
            long timeMsec = target.getNextEventTimeUTCSec() * 1000;

            // Choose best alarm method based on client Android version

            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
                // Wakes up the device in Doze Mode
                alarmMgr.setExactAndAllowWhileIdle(AlarmManager.RTC_WAKEUP, timeMsec, alarmTimeoutIntent);
            } else if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
                // Wakes up the device in Idle Mode
                alarmMgr.setExact(AlarmManager.RTC_WAKEUP, timeMsec, alarmTimeoutIntent);
            } else {
                // Old APIs
                alarmMgr.set(AlarmManager.RTC_WAKEUP, timeMsec, alarmTimeoutIntent);
            }
            // original code:
            //alarmMgr.set(AlarmManager.RTC_WAKEUP, timeMsec, alarmTimeoutIntent);
            setDisplayRefresh(timerIsActive);
            logToFile("Scheduled timer at " + target, false);
        } else {
            logToFile("No alarms scheduled", false);
        }
    }

    // this is called periodically to recover from unanticipated
    // factors that might kill off the alarm timers
    // and to deal with time zone/DST changes

    protected void restoreTimerIfNeeded() {
        boolean changed = testForTimeZoneChange();
        // only run alarm timer if needed
        int index = getNextAlarmTime();
        if (index != -1) {
            // logToFile("compare timers:" + target + " = " +
            // pendingAlarm,false);
            if (!changed && index == currentIndex) {
                // test if this timer is already running
                if (PendingIntent.getBroadcast(context, 0, new Intent(this,
                        AlarmHandler.class), PendingIntent.FLAG_NO_CREATE) == null) {
                    setAlarmTimeout();
                    logToFile("Restored alarm timer", false);
                } else {
                    logToFile(
                            "AlArm timer already running for " + alarms.get(index),
                            false);
                }
            } else {
                setAlarmTimeout();
            }
        } else {
            // no alarm needed, cancel any existing one
            cancelAlarmTimeout();
        }
        currentIndex = index;
    }

    private int getNextAlarmTime() {
        int result = -1;
        long nextTime = -1;
        timerIsActive = false;
        synchronized (alarms) {
            int n = 0;
            Iterator<GenericEvent> i = alarms.iterator();
            while (i.hasNext()) {
                GenericEvent ae = i.next();
                if (ae.getActive()) {
                    if (ae instanceof TimerEvent) {
                        timerIsActive = true;
                    }
                    long eventMillis = ae.getNextEventTimeUTCSec();
                    if (nextTime == -1 || eventMillis < nextTime) {
                        nextTime = eventMillis;
                        result = n;
                        //Log.e("ZZZ: next message",alarms.get(n).getMessage());
                    }
                }
                n += 1;
            }
        }
        return result;
    }

    protected void deSerialize() {
        FileInputStream fis = null;
        ObjectInputStream in = null;
        try {
            fis = openFileInput(serialObjectPath);
            if (fis != null && fis.available() > 0) {
                in = new ObjectInputStream(fis);
                serialData = (SerializedData) in.readObject();
                in.close();
                transferAllEvents(syncEvents, serialData.alarmEvents, false);
            }
        } catch (Exception e) {
            logError(e);
        }
    }

    protected void serialize() {
        FileOutputStream fos = null;
        ObjectOutputStream out = null;
        try {
            transferAllEvents(syncEvents, serialData.alarmEvents, true);
            fos = openFileOutput(serialObjectPath, Context.MODE_PRIVATE);
            out = new ObjectOutputStream(fos);
            out.writeObject(serialData);
            out.close();
        } catch (Exception e) {
            logError(e);
        }
    }

    protected void transferAllEvents(List<GenericEvent> alarms,
                                     List<GenericEvent> serial, boolean write) {
        synchronized (alarms) {
            if (write) {
                serial.clear();
                serial.addAll(alarms);
            } else {
                alarms.clear();
                alarms.addAll(serial);
            }
        }
    }

    // this tests whether the generated time zone value is sane
    protected void tzTest() {
        for (int m = 1; m < 13; m++) {
            GregorianCalendar cal = new GregorianCalendar(2013, m, 0, 0, 0, 0);
            long lt = cal.getTimeZone().getOffset(cal.getTimeInMillis());
            int off = (int) (lt / (3600000.0));
            String s = String.format("month: %d, offset hours: %d", m, off);
            Log.i("tzTest", s);
        }
    }

    static public int getLocalTimeZoneOffsetSeconds() {
        GregorianCalendar local = new GregorianCalendar();
        long lt = local.getTimeInMillis();
        int offset = local.getTimeZone().getOffset(lt);
        return (offset / 1000);
    }

    public boolean get24HourFormat() {
        return is24HourFormat;
    }

    public void set24HourFormat() {
        is24HourFormat = android.text.format.DateFormat.is24HourFormat(this);
    }

    private class AlarmComp implements Comparator<GenericEvent> {
        boolean complete = false;

        public AlarmComp(boolean comp) {
            complete = comp;
        }

        public int compare(GenericEvent a, GenericEvent b) {

            if ((a instanceof TimerEvent) && !(b instanceof TimerEvent))
                return -1;
            if (!(a instanceof TimerEvent) && (b instanceof TimerEvent))
                return 1;
            if (!complete)
                return 0;
            long dt = b.getSortEventTimeLocalSec()
                    - a.getSortEventTimeLocalSec();
            int q = (dt > 0) ? -1 : (dt < 0) ? 1 : 0;
            if (q == 0) {
                q = a.getMessage().compareToIgnoreCase(b.getMessage());
            }
            return q;
        }
    };

    protected void sortAlarms(boolean complete) {
        synchronized (alarms) {
            Collections.sort(alarms, new AlarmComp(complete));
        }
        int n = 0;
        for (GenericEvent ge : alarms) {
            ge.eventIndex = n;
            n += 1;
        }
    }

    private void initializeAlarms() {
        synchronized (alarms) {
            Iterator<GenericEvent> i = alarms.iterator();
            int n = 0;
            while (i.hasNext()) {
                GenericEvent ae = i.next();
                ae.setSnooze(false);
                ae.resetNow();
                ae. eventIndex = n;
                n += 1;
            }
        }
    }

    protected void updateDisplay() {
        if (activity != null) {
            activity.updateDisplay();
        }
    }

    protected void addEvent(GenericEvent ae) {
        synchronized (alarms) {
            alarms.add(ae);
            sortAlarms(false);
        }
    }

    protected void deleteEvent(GenericEvent ae) {
        synchronized (alarms) {
            alarms.remove(ae);
            sortAlarms(false);
        }
    }

    protected int getIndexForEvent(GenericEvent ae) {

        return ae.eventIndex;
        //int index = -1;
        //synchronized (alarms) {
        //	index = alarms.indexOf(ae);
        //}
        //return index;
    }

    protected GenericEvent getEventForIndex(int i) {
        GenericEvent ae = null;
        try {
            synchronized (alarms) {
                ae = alarms.get(i);
            }
        } catch (Exception e) {
        }
        return ae;
    }

    protected int getAlarmCount() {
        int n = 0;
        synchronized (alarms) {
            n = alarms.size();
        }
        return n;
    }

    private void createDir(File path, int perms) {
        if (!path.exists()) {
            path.mkdir();
        }
        boolean ownerOnly = (perms & 8) == 0;
        path.setExecutable((perms & 1) != 0, ownerOnly);
        path.setWritable((perms & 2) != 0, ownerOnly);
        path.setReadable((perms & 4) != 0, ownerOnly);
    }

    public void beep() {
        android.media.ToneGenerator tg = new ToneGenerator(
                AudioManager.STREAM_ALARM, 50);
        tg.startTone(ToneGenerator.TONE_CDMA_SOFT_ERROR_LITE);
    }

    public void showDebugMessage(Activity activity, String msg) {
        new Builder(activity).setTitle("Debug Message").setMessage(msg)
                .setPositiveButton("OK", null).show();
    }

    public void showMessage(Context context, String title, String msg) {
        AlertDialog.Builder b = new AlertDialog.Builder(context);
        b.setTitle(title);
        TextView tv = new TextView(context);
        tv.setText(msg);
        tv.setBackgroundColor(Color.TRANSPARENT);
        b.setView(tv);
        b.setNegativeButton("OK", null);
        b.show();
    }

    protected void makeToast(String msg) {
        Toast.makeText(context, msg, Toast.LENGTH_SHORT).show();
    }

    public String stringError(Exception e) {
        return "Error:" + e.getStackTrace()[1] + ":" + e.toString();
    }

    public void logString(String s, boolean date) {
        s = strip(s);
        if (s.length() > 0) {
            if (logBuffer != null) {
                if (date) {
                    s = dateTag(new Date()) + " " + s;
                }
                logBuffer.append("\n" + s);
            }
            if (DEBUG > 0) {
                appendTextFile(debugLogPath, s + "\n");
            }
        }
    }

    public String strip(String s) {
        if (s != null) {
            s = s.replaceFirst("^\\s*(.*?)\\s*$", "$1");
        }
        return s;
    }

    // print name of calling function and error
    protected void logError(Exception e) {
        String s = stringError(e);
        logString(s, true);
    }

    @SuppressLint("SimpleDateFormat")
    public String gcFromTimeMS(long timeMS) {
        GregorianCalendar gc = new GregorianCalendar();
        gc.setTimeInMillis(timeMS);
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss.SSS");
        return sdf.format(gc.getTime());
    }

    @SuppressLint("SimpleDateFormat")
    public String gcFromNow() {
        return gcFromTimeMS(System.currentTimeMillis());
    }

    @SuppressLint("SimpleDateFormat")
    public String dateTag(Date d) {
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        java.util.Calendar cal = Calendar.getInstance();
        sdf.setCalendar(cal);
        return sdf.format(d);
    }

    public void logToFile(String s, boolean alarm) {
        if (DEBUG > 1 || (alarm && DEBUG > 0)) {
            String msg = gcFromNow() + " " + s;
            String path = getFilesDir().getPath();
            String logFile = path + "/CarpeTempusLog.txt";
            appendTextFile(logFile, msg + "\n");
            Log.w("*** Debug:", msg);
        }
    }

    public void appendTextFile(String path, String data) {
        try {
            File f = new File(path);
            if (f.exists()) {
                f.setReadable(true, false);
            }
            FileWriter fw = new FileWriter(f, true);
            fw.write(data);
            fw.close();
        } catch (Exception e) {
            Log.w("appendTextFile", e.toString());
        }
    }

    public String readTextFile(String path) {
        try {
            File f = new File(path);
            if (!f.exists())
                return null;
            byte[] buffer = new byte[(int) f.length()];
            BufferedInputStream bis = new BufferedInputStream(
                    new FileInputStream(path));
            bis.read(buffer);
            bis.close();
            return new String(buffer);
        } catch (Exception e) {
            logError(e);
        }
        return null;
    }

    public void writeTextFile(String path, String data) {
        try {
            FileWriter fw = new FileWriter(path);
            fw.write(data);
            fw.close();
        } catch (Exception e) {
            logError(e);
        }
    }

    public void onInit(int status) {

    }

    protected long getNowUTCSecs() {
        long nowUTCSecs = System.currentTimeMillis() / 1000L;
        logToFile("get now UTC secs " + nowUTCSecs, false);
        return nowUTCSecs;
    }

    public void newAlarmButton(View v) {
        AlarmEvent ae = new AlarmEvent(getNowUTCSecs(),
                "Enter an alarm message");
        addEvent(ae);
    }

    public void newTimerButton(View v) {
        TimerEvent ae = new TimerEvent(60, "Enter a timer message");
        addEvent(ae);
    }

    protected void showKeyboard(Activity a, boolean show) {
        View v = (View) a.findViewById(R.id.main_layout);
        InputMethodManager imm = (InputMethodManager) getSystemService(Context.INPUT_METHOD_SERVICE);
        if (show) {
            imm.showSoftInput((View) v.getWindowToken(),
                    InputMethodManager.SHOW_IMPLICIT);
        } else {
            imm.hideSoftInputFromWindow(v.getWindowToken(),
                    InputMethodManager.HIDE_NOT_ALWAYS);
        }
    }

}
