/***************************************************************************
 *   Copyright (C) 2019 by Paul Lutus                                      *
 *   lutusp@arachnoid.com                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

package tankflow;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.EventQueue;
import java.awt.Font;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.GridLayout;
import java.awt.Insets;
import java.awt.Toolkit;
import java.awt.datatransfer.Clipboard;
import java.awt.datatransfer.ClipboardOwner;
import java.awt.datatransfer.StringSelection;
import java.awt.datatransfer.Transferable;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.awt.image.BufferedImage;
import java.io.BufferedReader;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.ArrayList;

import javax.swing.DefaultComboBoxModel;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTabbedPane;
import javax.swing.JTextField;
import javax.swing.JTextPane;
import javax.swing.ScrollPaneConstants;
import javax.swing.SwingConstants;
import javax.swing.SwingWorker;
import javax.swing.UIManager;
import javax.swing.UIManager.LookAndFeelInfo;
import javax.swing.text.SimpleAttributeSet;
import javax.swing.text.StyleConstants;
import java.awt.FlowLayout;

public class TankFlow implements ClipboardOwner {

	public String VERSION = "1.5";
	public String programName = getClass().getSimpleName() + " " + VERSION;
	protected JFrame frame;
	HelpPane helpPanel;
	private JTextPane textpane_field_data;
	private JTextPane textpane_sensor_volume;

	InitManager initManager = null;
	protected StateProperties sharedSP;
	protected ArrayList<Pair> results;
	ArrayList<Pair> reverse_results;
	private JLabel lbl_conversion_results;
	private JTextPane results_textpane;
	private String[] output_forms;
	private String[] funct_array, label_array;
	protected ArrayList<Double> terms = null;
	protected ArrayList<Pair> arg = null;
	private boolean reverse = false;
	private JTextPane table_textpane;
	private JTextField table_start_textfield;
	private JTextField table_end_textfield;
	private JTextField table_step_textfield;
	private boolean table_running = false;
	private JButton table_generate_button;
	private JTabbedPane main_tabbed_pane;
	private JPanel panel_drawing;
	private JPanel panel_help;
	protected double xlow, xhigh;
	private JPanel panel_graphic_control;
	private JButton btnCopyGraphic;
	private JButton increase_button;
	private JButton decrease_button;
	private JTextField graphic_degree_textfield;
	private JLabel poly_results_label;
	private JPanel panel;
	private JButton btnReadSampleTable;
	private JButton btnClearData;
	private JPanel panel_2;
	private JPanel panel_3;
	private JLabel lblDegree;
	private JTextField polynomial_degree_textfield;
	private JLabel lblCorrLow;
	private JTextField correct_low_textfield;
	private JCheckBox correct_low_checkbox;
	private JLabel lblCorrHigh;
	private JCheckBox correct_high_checkbox;
	private JTextField correct_high_textfield;
	private JPanel panel_4;
	private JComboBox<String> output_form_combobox;
	private JCheckBox reverse_checkbox;
	private JButton button;
	private JButton button_1;
	private JButton quit_button;
	private JComboBox<Integer> decimal_places_combobox;
	private String conversion_string = "";
	// protected BufferedImage image = null;

	/**
	 * Create the application.
	 */
	public TankFlow() {
		initialize();
		resetConfiguration();
		setup_function_strings();
		output_form_combobox.setModel(new DefaultComboBoxModel<String>(label_array));
		
		ArrayList<Integer> places = new ArrayList<Integer>();
		for(int n = 0;n <= 16; n++) {
			places.add(n);
		}
		Integer[] temp = places.toArray(new Integer[places.size()]);
		decimal_places_combobox.setModel(new DefaultComboBoxModel<Integer>(temp));
		set_conversion_string();
		
		quit_button = new JButton("Quit");
		quit_button.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				fullClose();
			}
		});
		quit_button.setToolTipText("Exit this program, all data saved");
		panel_4.add(quit_button);
		initManager = new InitManager(this);
		ImageIcon programIcon = new ImageIcon(TankFlow.class.getResource("/tankflow/icons/TankFlow.png"));
		frame.setIconImage(programIcon.getImage());
		frame.setTitle(programName);
		helpPanel = new HelpPane(this, 0);
		SimpleAttributeSet attribs = new SimpleAttributeSet();
		StyleConstants.setAlignment(attribs, StyleConstants.ALIGN_RIGHT);
		output_forms = new String[] { "simple list (x^0 to x^n)", "mathematical function", "C function", "C++ function",
				"Java function", "JavaScript function", "Python function" };

		panel_help = new JPanel();
		main_tabbed_pane.addTab("Help", null, panel_help, null);
		panel_help.setLayout(new BorderLayout(0, 0));
		panel_help.add(helpPanel);
		initManager.getDefaultValues();
		initManager.readConfig();
		writeValues();
		perform_regression();
	}

	protected void resetConfiguration() {
		sharedSP = new StateProperties();
		sharedSP.programFrame = frame;
		frame.setBounds(100, 100, 640, 480);
		results = new ArrayList<Pair>();
	}
	
	protected void fullClose() {
			saveBeforeClose();
			System.exit(0);
	}

	protected void saveBeforeClose() {
		readValues();
		initManager.writeConfig();
	}

	protected void set_degree_limit() {
		sharedSP.polynomialDegree = Math.max(0, sharedSP.polynomialDegree);
	}
	
	// allow the user to set decimal places
	
	protected void set_conversion_string() {
		int n = decimal_places_combobox.getSelectedIndex();
		conversion_string = String.format("%%.%dg",n);
	}
	
	protected String format_number(double v) {
		return String.format(conversion_string,v);
	}

	protected void writeValues() {
		set_degree_limit();
		polynomial_degree_textfield.setText("" + sharedSP.polynomialDegree);
		graphic_degree_textfield.setText("" + sharedSP.polynomialDegree);
		correct_low_textfield.setText("" + sharedSP.correctionLow);
		correct_high_textfield.setText("" + sharedSP.correctionHigh);
		correct_low_checkbox.setSelected(sharedSP.correctionLowApply);
		correct_high_checkbox.setSelected(sharedSP.correctionHighApply);
		reverse_checkbox.setSelected(sharedSP.reverseXYData);
		output_form_combobox.setSelectedIndex(sharedSP.output_form);
		String data = sharedSP.timeFlowData.replaceFirst("(?s).*<<<(.*)>>>.*", "$1");
		textpane_field_data.setText(data);
		data = sharedSP.sensorVolumeData.replaceFirst("(?s).*<<<(.*)>>>.*", "$1");
		textpane_sensor_volume.setText(data);
		//data = sharedSP.polyResultData.replaceFirst("(?s).*<<<(.*)>>>.*", "$1");
		//results_textpane.setText(data);
		data = sharedSP.tableData.replaceFirst("(?s).*<<<(.*)>>>.*", "$1");
		table_textpane.setText(data);
		
		main_tabbed_pane.setSelectedIndex(sharedSP.tabbedPane);
		graphic_degree_textfield.setText("" + sharedSP.polynomialDegree);
		table_start_textfield.setText("" + sharedSP.tableStart);
		table_end_textfield.setText("" + sharedSP.tableEnd);
		table_step_textfield.setText("" + sharedSP.tableStep);
	    decimal_places_combobox.setSelectedIndex(sharedSP.decimal_places);
	    set_conversion_string();
	}

	protected void readValues() {
		String data = textpane_field_data.getText();
		sharedSP.timeFlowData = "<<<" + data + ">>>";
		data = textpane_sensor_volume.getText();
		sharedSP.sensorVolumeData = "<<<" + data + ">>>";
		sharedSP.tabbedPane = main_tabbed_pane.getSelectedIndex();
		sharedSP.polynomialDegree = (int) Double.parseDouble(polynomial_degree_textfield.getText());
		sharedSP.correctionLow = Double.parseDouble(correct_low_textfield.getText());
		sharedSP.correctionLowApply = correct_low_checkbox.isSelected();
		sharedSP.correctionHigh = Double.parseDouble(correct_high_textfield.getText());
		sharedSP.correctionHighApply = correct_high_checkbox.isSelected();
		sharedSP.reverseXYData = reverse_checkbox.isSelected();
		sharedSP.output_form = output_form_combobox.getSelectedIndex();
		sharedSP.tableStart = Double.parseDouble(table_start_textfield.getText());
		sharedSP.tableEnd = Double.parseDouble(table_end_textfield.getText());
		sharedSP.tableStep = Double.parseDouble(table_step_textfield.getText());
		//data = results_textpane.getText();
		//sharedSP.polyResultData = "<<<" + data + ">>>";
		data = table_textpane.getText();
		sharedSP.tableData = "<<<" + data + ">>>";
		sharedSP.decimal_places = decimal_places_combobox.getSelectedIndex();
		set_degree_limit();
	}

	void p(String s) {
		System.out.println(s);
	}
	
	protected void update_low_correction() {
		try {
		sharedSP.correctionLow = Double.parseDouble(correct_low_textfield.getText());
		}
		catch(Exception e) {
			
		}
	}
	
	protected void update_high_correction() {
		try {
		sharedSP.correctionHigh = Double.parseDouble(correct_high_textfield.getText());
		}
		catch(Exception e) {
			
		}
	}

	private String rjexponent(double v, String suffix) {
		return String.format(format_number(v) + suffix);
	}

	protected void read_sample_data() {
		int response = JOptionPane.showConfirmDialog(frame, "Okay to replace data entries with sample data?",
				"Read Sample Data", JOptionPane.OK_CANCEL_OPTION);
		if (response == JOptionPane.OK_OPTION) {
			String s = readFile("/tankflow/sample_data_table.csv");
			textpane_field_data.setText(s);
		}
	}

	protected void clear_data_tables() {
		int response = JOptionPane.showConfirmDialog(frame, "Okay to clear all entered data?", "Clear Data Tables",
				JOptionPane.OK_CANCEL_OPTION);
		if (response == JOptionPane.OK_OPTION) {
			textpane_field_data.setText("");
			textpane_sensor_volume.setText("");
		}
	}
	
	protected void reset_defaults() {
		int response = JOptionPane.showConfirmDialog(frame, "Okay to reset all program values to defaults?",
				"Reset Program Defaults",JOptionPane.OK_CANCEL_OPTION);
		if (response == JOptionPane.OK_OPTION) {
			resetConfiguration();
			results_textpane.setText("");
			writeValues();
		}
	}

	void convert_field_data() {
		int n = 0;
		StringBuilder sb = new StringBuilder();
		String data = textpane_field_data.getText();
		// if(data.length() > 0) {
		String[] array = data.split("\n");
		if (array != null && array.length > 1) {
			results = new ArrayList<Pair>();
			String[] fields = new String[3];
			FieldDatum olddatum = null;
			double flowrate, sensor, hh, mm, ss;
			String[] hms;
			double volume = 0;
			for (String line : array) {
				// p(line);
				try {
					fields = line.split("\\s+|,|\t", 0);
					flowrate = Double.parseDouble(fields[1]);
					sensor = Double.parseDouble(fields[2]);
					hms = fields[0].split(":", 0);
					hh = Double.parseDouble(hms[0]);
					mm = Double.parseDouble(hms[1]);
					if (hms.length > 2) {
						ss = Double.parseDouble(hms[2]);
					} else {
						ss = 0;
					}
					ss = ss + mm * 60 + hh * 3600;
					FieldDatum newdatum = new FieldDatum(ss, sensor, flowrate);
					if (olddatum != null) {
						double dt = newdatum.time - olddatum.time;
						// this deals with both 12-hour and 24-hour rollovers
						while (dt < 0) {
							dt += 43200;
						}
						double averageflow = (newdatum.flow + olddatum.flow) / 2.0;
						// we're assuming flow per minute,
						// this may need to be changed
						volume += averageflow * dt / 60;
					}
					sb.append(String.format(format_number(newdatum.sensor) + "," + format_number(volume) + "\n"));
					n += 1;
					olddatum = newdatum;
				} catch (Exception e) {
					// p("error on " + line + "" + e);
				}
			}
		} else {
			// don't erase existing data
			if (textpane_sensor_volume.getText().length() < 2) {
				textpane_sensor_volume.setText("No field records available.");
			}
		}
		if (n > 0) {
			// p("converted: " + n);
			for (Pair p : results) {
				sb.append(p.toString());
				sb.append("\n");
			}
			textpane_sensor_volume.setText(sb.toString());
			textpane_sensor_volume.setCaretPosition(0);
		}
		lbl_conversion_results.setText(String.format("Converted %d field records", n));
		results_textpane.setText(String.format("%d field records available.", n));
	}

	protected void set_poly_degree(JTextField field) {
		try {
			String s = field.getText();
			int d = Integer.parseInt(s);
			update_poly_degree(d);
		} catch (Exception e) {
			change_poly_degree(0);

		}
	}

	protected void change_poly_degree(int n) {
		int d = sharedSP.polynomialDegree + n;
		update_poly_degree(d);
	}

	protected void update_poly_degree(int d) {
		d = Math.max(0, d);
		sharedSP.polynomialDegree = d;
		this.polynomial_degree_textfield.setText("" + d);
		this.graphic_degree_textfield.setText("" + d);
		perform_regression();
	}

	private void perform_regression() {
		results = new ArrayList<Pair>();
		xlow = 1e9;
		xhigh = -1e9;
		terms = null;
		String[] fields = new String[2];
		String data = textpane_sensor_volume.getText();
		String[] array = data.split("\n");
		if (array.length > 1) {
			double volume, sensor;
			for (String line : array) {
				try {
					// p("Line: " + line);
					fields = line.split("\\s+|,|\t", 0);
					sensor = Double.parseDouble(fields[0]);
					volume = Double.parseDouble(fields[1]);
					results.add(new Pair(sensor, volume));
				} catch (Exception e) {
					p("" + e);
				}
			}
		}
		if (results.size() > 1) {
			interpolate_data();
			arg = results;
			reverse = reverse_checkbox.isSelected();
			if (reverse) {
				reverse_results = new ArrayList<>();
				for (Pair p : results) {
					reverse_results.add(new Pair(p.y, p.x));
				}
				arg = reverse_results;
			}
			terms = MatrixFunctions.compute_coefficients(arg, sharedSP.polynomialDegree);
		} else {
			results_textpane.setText("No converted sensor/volume data.");
		}
		if (terms != null) {
			for (Pair p : arg) {
				xlow = Math.min(xlow, p.x);
				xhigh = Math.max(xhigh, p.x);
			}
			int points = arg.size();
			table_start_textfield.setText("" + xlow);
			table_end_textfield.setText("" + xhigh);
			double result_cc = MatrixFunctions.corr_coeff(arg, terms);
			double result_se = MatrixFunctions.std_error(arg, terms);
			StringBuilder sb = new StringBuilder();
			sb.append(String.format("Mode: %s analysis.\n", (reverse) ? "reverse (y,x)" : "normal x,y"));
			String short_label = String.format("Polynomial degree %d with %d data points\n", sharedSP.polynomialDegree,
					points);
			sb.append(short_label);
			String medium_label = String.format("Pts: %d, CC: " + format_number(result_cc) + ", SE: " + format_number(result_se), points);
			poly_results_label.setText(medium_label);
			String long_tip = String.format(
					"<html>Data points: %d<br/>Correlation Coefficient: " + format_number(result_cc) + "<br/>Standard Error: " + format_number(result_se), points);
			poly_results_label.setToolTipText(long_tip);
			sb.append(String.format("Correlation coefficient: " + format_number(result_cc) + ".\n"));
			sb.append(String.format("Standard Error: " + format_number(result_se) + "\n\n"));
			int index = output_form_combobox.getSelectedIndex();
			String type = output_forms[index];
			sb.append(String.format("Output form: %s\n", type));
			String line = "--------------------------------------";
			StringBuffer cb = new StringBuffer();
			switch (index) {
			case 0:
				cb.append("\n");
				for (double term : terms) {
					cb.append(rjexponent(term, "\n"));
				}
				break;
			case 1:
				int p = 0;
				for (double term : terms) {
					if (p == 0) {
						cb.append("\nf(x) = ");
					} else {
						cb.append("     + ");
					}
					cb.append(String.format("%s * x^%d\n", rjexponent(term, ""), p));
					p += 1;
				}
				break;
			default:
				cb.append("\n");
				StringBuffer sf = new StringBuffer();
				int i = 0;
				for (double term : terms) {
					String comma = (i < terms.size() - 1) ? "," : "";
					sf.append(String.format("    %s%s\n", rjexponent(term, ""), comma));
					i += 1;
				}
				String content = funct_array[index - 2];
				content = content.replace("TERMS", sf.toString());
				cb.append(content);
				cb.append("\n");

				break;
			}
			sb.append(line);
			sb.append(cb);
			sb.append(line);
			sb.append("\nCopyright (c) 2019, P. Lutus -- http://arachnoid.com.");
			results_textpane.setText(sb.toString());
			results_textpane.setCaretPosition(0);
		}
		if (panel_drawing != null) {
			panel_drawing.repaint();
		}
	}

	private double ntrp(double x, double xl, double xh, double yl, double yh) {
		return (x - xl) * (yh - yl) / (xh - xl) + yl;
	}

	private void interpolate_data() {
		boolean perform = false;
		if (results != null) {
			double y_low = results.get(0).y;
			double y_high = results.get(results.size() - 1).y;
			double y_dl = y_low;
			double y_dh = y_high;
			if (correct_low_checkbox.isSelected()) {
				y_dl = Double.parseDouble(correct_low_textfield.getText());
				perform = true;
			}
			if (correct_high_checkbox.isSelected()) {
				y_dh = Double.parseDouble(correct_high_textfield.getText());
				perform = true;
			}
			if (perform) {
				ArrayList<Pair> conv = new ArrayList<>();
				for (Pair p : results) {
					double dy = ntrp(p.y, y_low, y_high, y_dl, y_dh);
					conv.add(new Pair(p.x, dy));
				}
				results = conv;
			}
		}
	}

	private String format_table_value(double x) {
		double y = MatrixFunctions.regress(x, terms);
		return String.format(format_number(x) + "," + format_number(y) + "\n");
	}

	private void create_table() {
		if (table_running) {
			table_running = false;
			table_textpane.setText("Interrupted.");
			table_generate_button.setText("Create");

		} else {
			table_running = true;
			table_generate_button.setText("Stop");
			table_worker();
		}
	}

	private void table_worker() {
		SwingWorker<?, ?> worker = new SwingWorker<Object, Object>() {
			@Override
			public String doInBackground() throws Exception {
				if (terms != null) {
					table_textpane.setText("Generating table ...");
					double x = Double.parseDouble(table_start_textfield.getText());
					double t_end = Double.parseDouble(table_end_textfield.getText());
					double t_step = Double.parseDouble(table_step_textfield.getText());
					StringBuilder sb = new StringBuilder();
					String s = "";
					while (x < t_end && table_running) {
						s = format_table_value(x);
						sb.append(s);
						x += t_step;
					}
					if (table_running) {
						String se = format_table_value(t_end);
						if (!se.equals(s)) {
							sb.append(se);
						}
						table_textpane.setText(sb.toString());
					}
				} else {
					table_textpane.setText("No regression data avalable.");
				}
				done();
				return "";
			}

			@Override
			protected void done() {
				table_running = false;
				table_generate_button.setText("Create");
			}
		};
		worker.execute();
	}

	/**
	 * Launch the application.
	 */
	
	
	public static void main(String[] args) {
		try {
			for (LookAndFeelInfo info : UIManager.getInstalledLookAndFeels()) {
				if ("Nimbus".equals(info.getName())) {
					UIManager.setLookAndFeel(info.getClassName());
					break;
				}
			}
		} catch (Exception e) {
			// If Nimbus is not available, you can set the GUI to another look and feel.
		}
		
		EventQueue.invokeLater(new Runnable() {
			public void run() {
				try {
					TankFlow app_window = new TankFlow();
					app_window.frame.setVisible(true);
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		});
	}

	private void setup_function_strings() {

		String cppFunct = "double regress(double x) {\n  double terms[] = {\nTERMS};\n  \n  double t = 1;\n  double r = 0;\n  for (double c : terms) {\n    r += c * t;\n    t *= x;\n  }\n  return r;\n}";
		String cFunct = "double regress(double x) {\n  double terms[] = {\nTERMS};\n  \n  size_t csz = sizeof terms / sizeof *terms;\n  \n  double t = 1;\n  double r = 0;\n  for (int i = 0; i < csz;i++) {\n    r += terms[i] * t;\n    t *= x;\n  }\n  return r;\n}";
		String pyFunct = "terms = [\nTERMS]\n\ndef regress(x):\n  t = 1\n  r = 0\n  for c in terms:\n    r += c * t\n    t *= x\n  return r";
		String javaFunct = "double regress(double x) {\n    double terms[] = {\nTERMS};\n    \n    double t = 1;\n    double r = 0;\n    for (double c : terms) {\n      r += c * t;\n      t *= x;\n    }\n    return r;\n}";
		String javascriptFunct = "terms = [\nTERMS];\n\nregress(x, terms) {\n    var r = 0;\n    var t = 1;\n    for (var i in terms) {\n      r += terms[i] * t;\n      t *= x;\n    }\n    return r;\n}";

		funct_array = new String[] { cFunct, cppFunct, javaFunct, javascriptFunct, pyFunct };
		label_array = new String[] { "Simple list","Mathematical function","C function", "C++ function", "Java function", "JavaScript function",
				"Python function" };
	}

	void showNotifyMessage(String message, String title) {
		JOptionPane.showMessageDialog(frame, message, programName + ": " + title, JOptionPane.INFORMATION_MESSAGE);
	}

	void setClipboardContents(String s) {
		StringSelection stringSelection = new StringSelection(s);
		Clipboard clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();
		clipboard.setContents(stringSelection, this);
	}

	protected void copy_graphic_to_clipboard() {
		BufferedImage image = ((GraphicPane) panel_drawing).grabScreen(1280, 800);
		ImageTransferable imt = new ImageTransferable(image);
		Clipboard clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();
		clipboard.setContents(imt, null);
	}

	protected String readFile(String path) {
		StringBuilder sb = new StringBuilder();
		try {
			InputStream is = TankFlow.class.getResourceAsStream(path);
			InputStreamReader isr = new InputStreamReader(is);
			BufferedReader br = new BufferedReader(isr);
			String line;
			while ((line = br.readLine()) != null) {
				sb.append(line + "\n");
			}
			is.close();
		} catch (Exception e) {
			e.printStackTrace();
		}
		return sb.toString();
	}

	/**
	 * Initialize the contents of the frame.
	 */
	private void initialize() {
		frame = new JFrame();
		frame.addWindowListener(new WindowAdapter() {
			@Override
			public void windowClosing(WindowEvent e) {
				fullClose();
			}
		});
		// frame.setBounds(100, 100, 640, 480);
		frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);

		main_tabbed_pane = new JTabbedPane(JTabbedPane.BOTTOM);
		frame.getContentPane().add(main_tabbed_pane, BorderLayout.CENTER);

		JPanel panel_flow = new JPanel();
		panel_flow.setToolTipText("");
		main_tabbed_pane.addTab("Flow Data", null, panel_flow, null);
		panel_flow.setLayout(new BorderLayout(0, 0));

		JScrollPane scrollPane = new JScrollPane();
		scrollPane.setHorizontalScrollBarPolicy(ScrollPaneConstants.HORIZONTAL_SCROLLBAR_NEVER);
		panel_flow.add(scrollPane, BorderLayout.CENTER);

		textpane_field_data = new JTextPane();
		textpane_field_data.setToolTipText("Enter time/flow rate/sensor data here");
		textpane_field_data.setFont(new Font("Monospaced", Font.PLAIN, 12));
		scrollPane.setViewportView(textpane_field_data);

		panel = new JPanel();
		panel_flow.add(panel, BorderLayout.NORTH);

		btnReadSampleTable = new JButton("Read Sample Table ...");
		btnReadSampleTable.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				read_sample_data();
			}
		});
		btnReadSampleTable.setToolTipText("Read a sample data table for tutorial purposes");
		panel.add(btnReadSampleTable);

		btnClearData = new JButton("Clear Data ...");
		btnClearData.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				clear_data_tables();
			}
		});
		btnClearData.setToolTipText("Clear all entered data");
		panel.add(btnClearData);

		JPanel panel_volume = new JPanel();
		main_tabbed_pane.addTab("Sensor/Volume", null, panel_volume, null);
		panel_volume.setLayout(new BorderLayout(0, 0));

		JScrollPane scrollPane_1 = new JScrollPane();
		panel_volume.add(scrollPane_1, BorderLayout.CENTER);

		textpane_sensor_volume = new JTextPane();
		textpane_sensor_volume.setToolTipText("Convert flow data or enter sensor/volume data here");
		textpane_sensor_volume.setFont(new Font("Monospaced", Font.PLAIN, 12));
		scrollPane_1.setViewportView(textpane_sensor_volume);

		JPanel control_panel_flow = new JPanel();
		panel_volume.add(control_panel_flow, BorderLayout.NORTH);
		control_panel_flow.setLayout(new BorderLayout(0, 0));

		JButton btn_convert = new JButton("Convert");
		btn_convert
				.setToolTipText("Convert field time/flow/sensor data from Flow Data tab into sensor/volume table here");
		btn_convert.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				convert_field_data();
			}
		});
		btn_convert.setHorizontalAlignment(SwingConstants.LEFT);
		control_panel_flow.add(btn_convert, BorderLayout.EAST);

		lbl_conversion_results = new JLabel("Results ...");
		lbl_conversion_results.setHorizontalAlignment(SwingConstants.CENTER);
		control_panel_flow.add(lbl_conversion_results, BorderLayout.CENTER);

		JPanel panel_regression = new JPanel();
		panel_regression.setToolTipText("");
		main_tabbed_pane.addTab("Regression", null, panel_regression, null);
		panel_regression.setLayout(new BorderLayout(0, 0));

		JScrollPane scrollPane_2 = new JScrollPane();
		panel_regression.add(scrollPane_2, BorderLayout.CENTER);

		results_textpane = new JTextPane();
		results_textpane.setToolTipText("This panel shows the most recent regression analysis result");
		results_textpane.setFont(new Font("Monospaced", Font.PLAIN, 12));
		scrollPane_2.setViewportView(results_textpane);

		panel_2 = new JPanel();
		panel_regression.add(panel_2, BorderLayout.NORTH);
		panel_2.setLayout(new GridLayout(2, 0, 0, 0));

		panel_3 = new JPanel();
		panel_2.add(panel_3);
		panel_3.setLayout(new FlowLayout(FlowLayout.CENTER, 5, 5));

		lblDegree = new JLabel("Degree");
		lblDegree.setToolTipText("Enter a polynomial degree here");
		lblDegree.setHorizontalAlignment(SwingConstants.CENTER);
		panel_3.add(lblDegree);

		polynomial_degree_textfield = new JTextField();
		polynomial_degree_textfield.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				set_poly_degree(polynomial_degree_textfield);
			}
		});
		polynomial_degree_textfield.setToolTipText("Enter polynomial degree here");
		polynomial_degree_textfield.setText("12");
		polynomial_degree_textfield.setMinimumSize(new Dimension(26, 19));
		polynomial_degree_textfield.setHorizontalAlignment(SwingConstants.RIGHT);
		polynomial_degree_textfield.setColumns(2);
		panel_3.add(polynomial_degree_textfield);

		lblCorrLow = new JLabel("Corr. Low");
		lblCorrLow.setToolTipText("");
		lblCorrLow.setHorizontalAlignment(SwingConstants.CENTER);
		panel_3.add(lblCorrLow);

		correct_low_textfield = new JTextField();
		correct_low_textfield.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				update_low_correction();
			}
		});
		correct_low_textfield.setToolTipText("Enter an optional low correction value here");
		correct_low_textfield.setText("0");
		correct_low_textfield.setHorizontalAlignment(SwingConstants.RIGHT);
		correct_low_textfield.setColumns(8);
		panel_3.add(correct_low_textfield);

		correct_low_checkbox = new JCheckBox("Apply");
		correct_low_checkbox.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				perform_regression();
			}
		});
		correct_low_checkbox.setToolTipText("Enable to apply the low correction");
		correct_low_checkbox.setSelected(false);
		panel_3.add(correct_low_checkbox);

		lblCorrHigh = new JLabel(" | Corr. High");
		lblCorrHigh.setToolTipText("");
		lblCorrHigh.setHorizontalAlignment(SwingConstants.CENTER);
		panel_3.add(lblCorrHigh);

		correct_high_textfield = new JTextField();
		correct_high_textfield.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				update_high_correction();
			}
		});
		correct_high_textfield.setToolTipText("Enter an optional high correction value here");
		correct_high_textfield.setText("100");
		correct_high_textfield.setHorizontalAlignment(SwingConstants.RIGHT);
		correct_high_textfield.setColumns(8);
		panel_3.add(correct_high_textfield);

		correct_high_checkbox = new JCheckBox("Apply");
		correct_high_checkbox.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				perform_regression();
			}
		});
		correct_high_checkbox.setToolTipText("Enable to apply the high correction");
		correct_high_checkbox.setSelected(false);
		correct_high_checkbox.setHorizontalAlignment(SwingConstants.LEFT);
		panel_3.add(correct_high_checkbox);

		panel_4 = new JPanel();
		panel_2.add(panel_4);

		output_form_combobox = new JComboBox<String>();
		output_form_combobox.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				perform_regression();
			}
		});
		
		decimal_places_combobox = new JComboBox<Integer>();
		decimal_places_combobox.setToolTipText("Choose the number of decimal places in tables and results");
		decimal_places_combobox.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				set_conversion_string();
			}
		});
		panel_4.add(decimal_places_combobox);
		output_form_combobox.setModel(new DefaultComboBoxModel<String>(new String[] { "\"a\",\"b\",\"c\"" }));
		output_form_combobox.setSelectedIndex(0);
		output_form_combobox.setToolTipText("Select the desired output format");
		panel_4.add(output_form_combobox);

		reverse_checkbox = new JCheckBox("Reverse X-Y");
		reverse_checkbox.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				perform_regression();
			}
		});
		reverse_checkbox.setToolTipText("Enable to reverse the relationship between X and Y");
		reverse_checkbox.setSelected(false);
		panel_4.add(reverse_checkbox);

		button = new JButton("Perform");
		button.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				perform_regression();
			}
		});
		button.setToolTipText("Carry out this regression");
		panel_4.add(button);

		button_1 = new JButton("Reset");
		button_1.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				reset_defaults();
			}
		});
		button_1.setToolTipText("Reset all program values to defaults");
		panel_4.add(button_1);

		JPanel panel_graphic = new JPanel();
		main_tabbed_pane.addTab("Graphic", null, panel_graphic, null);
		panel_graphic.setLayout(new BorderLayout(0, 0));

		panel_drawing = new GraphicPane(this);
		panel_drawing.setForeground(Color.WHITE);
		panel_drawing.setBackground(Color.WHITE);
		panel_graphic.add(panel_drawing, BorderLayout.CENTER);
		panel_drawing.setLayout(new BorderLayout(0, 0));

		panel_graphic_control = new JPanel();
		panel_graphic_control.setBackground(Color.WHITE);
		panel_graphic.add(panel_graphic_control, BorderLayout.NORTH);
		GridBagLayout gbl_panel_graphic_control = new GridBagLayout();
		gbl_panel_graphic_control.columnWidths = new int[] { 32, 32, 32, 32, 32 };
		gbl_panel_graphic_control.rowHeights = new int[] { 25, 0 };
		gbl_panel_graphic_control.columnWeights = new double[] { 0.0, 0.0, 0.0, 1.0, 0.0 };
		gbl_panel_graphic_control.rowWeights = new double[] { 1.0, Double.MIN_VALUE };
		panel_graphic_control.setLayout(gbl_panel_graphic_control);

		decrease_button = new JButton("");
		decrease_button.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				change_poly_degree(-1);
			}
		});
		decrease_button.setToolTipText("Decrease polynomial degree");
		decrease_button.setIcon(new ImageIcon(TankFlow.class.getResource("/tankflow/icons/go-previous.png")));
		decrease_button.setVerticalAlignment(SwingConstants.TOP);
		decrease_button.setHorizontalAlignment(SwingConstants.LEFT);
		GridBagConstraints gbc_decrease_button = new GridBagConstraints();
		gbc_decrease_button.fill = GridBagConstraints.HORIZONTAL;
		gbc_decrease_button.insets = new Insets(0, 0, 0, 5);
		gbc_decrease_button.anchor = GridBagConstraints.WEST;
		gbc_decrease_button.gridx = 0;
		gbc_decrease_button.gridy = 0;
		panel_graphic_control.add(decrease_button, gbc_decrease_button);

		graphic_degree_textfield = new JTextField();
		graphic_degree_textfield.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				set_poly_degree(graphic_degree_textfield);
			}
		});
		graphic_degree_textfield.setToolTipText("Enter polynomial degree");
		graphic_degree_textfield.setHorizontalAlignment(SwingConstants.CENTER);
		GridBagConstraints gbc_graphic_degree_textfield = new GridBagConstraints();
		gbc_graphic_degree_textfield.anchor = GridBagConstraints.WEST;
		gbc_graphic_degree_textfield.insets = new Insets(0, 0, 0, 5);
		gbc_graphic_degree_textfield.gridx = 1;
		gbc_graphic_degree_textfield.gridy = 0;
		panel_graphic_control.add(graphic_degree_textfield, gbc_graphic_degree_textfield);
		graphic_degree_textfield.setColumns(2);

		increase_button = new JButton("");
		increase_button.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				change_poly_degree(1);
			}
		});
		increase_button.setToolTipText("Increase polynomial degree");
		increase_button.setIcon(new ImageIcon(TankFlow.class.getResource("/tankflow/icons/go-next.png")));
		GridBagConstraints gbc_increase_button = new GridBagConstraints();
		gbc_increase_button.fill = GridBagConstraints.HORIZONTAL;
		gbc_increase_button.anchor = GridBagConstraints.NORTHWEST;
		gbc_increase_button.insets = new Insets(0, 0, 0, 5);
		gbc_increase_button.gridx = 2;
		gbc_increase_button.gridy = 0;
		panel_graphic_control.add(increase_button, gbc_increase_button);

		btnCopyGraphic = new JButton("Copy Chart");
		btnCopyGraphic.setHorizontalAlignment(SwingConstants.RIGHT);
		btnCopyGraphic.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				copy_graphic_to_clipboard();
			}

		});

		poly_results_label = new JLabel();
		poly_results_label.setBackground(Color.WHITE);
		poly_results_label.setText("Polynomial results");
		GridBagConstraints gbc_poly_results_label = new GridBagConstraints();
		gbc_poly_results_label.insets = new Insets(0, 0, 0, 5);
		gbc_poly_results_label.fill = GridBagConstraints.HORIZONTAL;
		gbc_poly_results_label.gridx = 3;
		gbc_poly_results_label.gridy = 0;
		panel_graphic_control.add(poly_results_label, gbc_poly_results_label);
		btnCopyGraphic.setToolTipText("Copy this chart as an image to the system clipboard");
		GridBagConstraints gbc_btnCopyGraphic = new GridBagConstraints();
		gbc_btnCopyGraphic.fill = GridBagConstraints.HORIZONTAL;
		gbc_btnCopyGraphic.insets = new Insets(0, 0, 0, 5);
		gbc_btnCopyGraphic.anchor = GridBagConstraints.EAST;
		gbc_btnCopyGraphic.gridx = 4;
		gbc_btnCopyGraphic.gridy = 0;
		panel_graphic_control.add(btnCopyGraphic, gbc_btnCopyGraphic);

		JPanel panel_table = new JPanel();
		main_tabbed_pane.addTab("Table", null, panel_table, null);
		panel_table.setLayout(new BorderLayout(0, 0));

		JScrollPane scrollPane_3 = new JScrollPane();
		panel_table.add(scrollPane_3, BorderLayout.CENTER);

		table_textpane = new JTextPane();
		table_textpane.setToolTipText("Create data table here");
		scrollPane_3.setViewportView(table_textpane);

		JPanel panel_1 = new JPanel();
		panel_table.add(panel_1, BorderLayout.NORTH);
		panel_1.setLayout(new GridLayout(0, 7, 0, 0));

		JLabel label = new JLabel("Start");
		label.setHorizontalAlignment(SwingConstants.CENTER);
		panel_1.add(label);

		table_start_textfield = new JTextField();
		table_start_textfield.setToolTipText("Table start X value");
		table_start_textfield.setText("0");
		table_start_textfield.setMaximumSize(new Dimension(100, 2147483647));
		table_start_textfield.setHorizontalAlignment(SwingConstants.RIGHT);
		table_start_textfield.setColumns(10);
		panel_1.add(table_start_textfield);

		JLabel label_1 = new JLabel("End");
		label_1.setHorizontalAlignment(SwingConstants.CENTER);
		panel_1.add(label_1);

		table_end_textfield = new JTextField();
		table_end_textfield.setToolTipText("Table end X value");
		table_end_textfield.setText("100");
		table_end_textfield.setMaximumSize(new Dimension(100, 2147483647));
		table_end_textfield.setHorizontalAlignment(SwingConstants.RIGHT);
		table_end_textfield.setColumns(10);
		panel_1.add(table_end_textfield);

		JLabel label_2 = new JLabel("Step");
		label_2.setHorizontalAlignment(SwingConstants.CENTER);
		panel_1.add(label_2);

		table_step_textfield = new JTextField();
		table_step_textfield.setToolTipText("Table step size");
		table_step_textfield.setText("1");
		table_step_textfield.setMaximumSize(new Dimension(100, 2147483647));
		table_step_textfield.setHorizontalAlignment(SwingConstants.RIGHT);
		table_step_textfield.setColumns(10);
		panel_1.add(table_step_textfield);

		table_generate_button = new JButton("Create");
		table_generate_button.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				create_table();
			}
		});
		table_generate_button.setToolTipText("Create the table");
		panel_1.add(table_generate_button);

	}

	@Override
	public void lostOwnership(Clipboard clipboard, Transferable contents) {
		// TODO Auto-generated method stub

	}
}
