/***************************************************************************
 *   Copyright (C) 2004 by Paul Lutus                                      *
 *   lutusp@arachnoid.com                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <qtooltip.h>
#include "networkcontrol.h"
#include "comthread.h"
#include <qmessagebox.h>
#include "bellthread.h"
#include "red_led.xpm"
#include "grn_led.xpm"
#include "yel_led.xpm"
#include "networkcontrol.xpm"

const string NetworkControl::programName = "NetworkControl";

const string NetworkControl::programVersion = "3.4";

const char *NetworkControl::interfacePath = "/etc/sysconfig/network-scripts";

const char *NetworkControl::interfacePrefix = "ifcfg-";

const char *NetworkControl::wirelessDataLocation = "/proc/net/wireless";

const char *NetworkControl::interfaceDataSource = "/sbin/ifconfig";

const char *NetworkControl::upDownCommands[] =
    {"/sbin/ifdown","/sbin/ifup"
    };

QColor NetworkControl::textColors[] = { QColor ( 255,220,220 ), QColor ( 255,255,220 ), QColor ( 220,255,220 ) };

NetworkControl::NetworkControl ( QWidget* parent, const char* name, WFlags fl )
		: MainFrame ( parent,name,fl )
{
	usrPath = ( string ) getenv ( "HOME" );
	progPath = usrPath + "/." + programName;
	initHandler = new InitFileHandler ( usrPath,programName );
	stateIcons[0] = QPixmap ( red_led );
	stateIcons[1] = QPixmap ( yel_led );
	stateIcons[2] = QPixmap ( grn_led );
	comthread = new ComThread();
	bellthread = new BellThread();
	timer = new QTimer ( this );
	idm = new InterfaceDataManager();
	interfaceCount = idm->readDirectory ( string ( interfacePath ),string ( interfacePrefix ) );
	comStates = new int[interfaceCount];
	oldStates = new int[interfaceCount];
	timerIntervalMs = 1000;
	setup();

	setCaption ( programName + " " + programVersion );
	// set program icon using XPM file
	setIcon ( QPixmap ( networkcontrol ) );
	IconQLabel->setScaledContents ( false );
	readConfigFile();
	setWirelessState ( false );
	startTimer();
	showInterfaceData();
}

NetworkControl::~NetworkControl()
{
	delete ( idm );
	timer->stop();
	delete ( timer );
	delete ( comStates );
	delete ( oldStates );
	delete ( comthread );
	delete ( bellthread );
	delete ( initHandler );
}

void NetworkControl::setup()
{
	for ( int i = 0;i < interfaceCount;i++ )
	{
		comStates[i] = getInterfaceState ( i );
		oldStates[i] = comStates[i];
	}
	vector <string> v = idm->getInterfaceList();
	for ( unsigned int i = 0;i < v.size();i++ )
	{
		InterfaceComboBox->insertItem ( v[i] );
	}
	selectionCounter = 0;
}


void NetworkControl::startTimer()
{
	connect ( timer, SIGNAL ( timeout() ),this, SLOT ( periodic() ) );
	timer->start ( timerIntervalMs, false );
}

void NetworkControl::periodic()
{
	showInterfaceData();
	string s = comthread->getErrorMessage();
	if ( s.length() > 0 )
	{
		QMessageBox::warning ( this, ( string ) programName + ": Error",s.c_str() );
	}
}

string NetworkControl::strip_alias ( string s )
{
	unsigned int p = s.find ( ":" );
	if ( p != string::npos )
	{
		s = s.substr ( 0,p );
	}
	return s;
}

const string NetworkControl::stateNames[] =
    {"not connected","connected"
    };
const string NetworkControl::deltaStates[] =
    {"disconnecting","connecting"
    };

void NetworkControl::showInterfaceData()
{
	if ( !textIsSelected() || selectionCounter-- <= 0 )
	{
		selectionCounter = 8;
		DataDisplay->setSelection ( -1,-1,-1,-1 );
		int i = InterfaceComboBox->currentItem();
		string name = idm->getInterfaceName ( i );
		string data = getInterfaceData ( i );
		int ifstate = getInterfaceState ( data );
		int delta = comStates[i]-ifstate;
		string label = stateNames[ifstate];
		int colorIndex = ifstate * 2;
		if ( delta != 0 )
		{
			colorIndex = 1;
			label = ( delta < 0 )
			        ?deltaStates[0]:deltaStates[1];
		}
		setState ( colorIndex,ifstate,i ); // 0,1,2
		// refetch the full data
		data = getInterfaceData ( i, true );
		string r;
		r.append ( name );
		r.append ( " " );
		r.append ( label );
		if ( ifstate )
		{
			r.append ( "\n\n" );
			r.append ( data );
		}
		DataDisplay->setText ( r );
		processWirelessData ( idm->getDeviceName ( i ),ifstate );
	}
}

string NetworkControl::getInterfaceData ( int i,bool stripalias )
{
	string interface = idm->getDeviceName ( i );
	if ( stripalias )
	{
		interface = strip_alias ( interface );
	}
	string com = interfaceDataSource;
	com.append ( " " );
	com.append ( interface );
	com.append ( " 2> /dev/null" );
	FILE *fp = popen ( com.c_str(),"r" );
	char buf[256];
	string data;
	while ( fgets ( buf,sizeof ( buf ),fp ) )
	{
		string s = buf;
		s = idm->trimWhitespace ( s );
		if ( s.length() > 0 )
		{
			data.append ( s + "\n" );
		}
	}
	pclose ( fp );
	return data;
}

int NetworkControl::getInterfaceState ( int i )
{
	string id = getInterfaceData ( i );
	return getInterfaceState ( id );
}

int NetworkControl::getInterfaceState ( string id )
{
	bool result = ( id.find ( "UP " ) != string::npos );
	if ((id.size() < 3) || (id.substr ( 0,3 ) != "ppp" ))
	{
		result = result && ( id.find ( "inet addr" ) != string::npos );
	}
	return ( result ) ?1:0;
}

/*$SPECIALIZATION$*/
void NetworkControl::interfaceChange()
{
	showInterfaceData();
}

void NetworkControl::disconnectButton()
{
	issueCommand ( 0 );
}

void NetworkControl::connectButton()
{
	issueCommand ( 1 );
}

void NetworkControl::aboutButtonPressed()
{
	QMessageBox::information ( this,"About " + programName + " " + programVersion,
	                           programName + " " + programVersion + " Copyright 2004, P. Lutus\n"
	                           + programName + " is released under the General Public License (GPL).\n\n"
	                           + "For more information about " + programName + ", please visit\n"
	                           + "http://www.arachnoid.com/" + programName + "."
	                         );
}

void NetworkControl::setState ( int colorState,int state,int i )
{
	IconQLabel->setPixmap ( stateIcons[colorState] );
	//IconLED->setColor(iconColors[colorState]);
	DataDisplay->setPaletteBackgroundColor ( textColors[colorState] );
	if ( state == 1 && oldStates[i] == 0 )
	{
		ringBell();
	}
	oldStates[i] = state;

}

bool NetworkControl::textIsSelected()
{
	int a,b,c,d;
	DataDisplay->getSelection ( &a,&b,&c,&d );
	return ! ( a == -1 || b == -1 || c == -1 || d == -1 );
}

void NetworkControl::issueCommand ( int com )
{
	int i = InterfaceComboBox->currentItem();
	comStates[i] = com;
	string s = upDownCommands[com];
	s.append ( " " );
	s.append ( idm->getInterfaceName ( i ) );
	comthread->execute ( s );
	showInterfaceData();
}

int convertSSVal ( double val )
{
	int v = ( int ) ( 100 * val / 256.0 );
	if ( v < 0 )
	{
		v = 100 + v;
	}
	return v;
}

void NetworkControl::processWirelessData ( string ifn, bool state )
{
	ifn = strip_alias ( ifn );
	ifstream ifs ( wirelessDataLocation );
	string line;
	bool found = false;
	while ( !false && getline ( ifs,line ) )
	{
		line = idm->trimWhitespace ( line );
		vector <string> v = idm->splitOnWhitespace ( line );
		//printf("size: %d, string= [%s]\n",v.size(), v[0].c_str());
		fflush ( stdout );
		if ( ( v.size() == 11 ) && ( v[0] == ( ifn+":" ) ) )
		{
			found = true;
			WirelessLabel->setText ( ifn );
			double dlq,dsignal,dnoise;
			sscanf ( v[2].c_str(),"%lf",&dlq );
			sscanf ( v[3].c_str(),"%lf",&dsignal );
			sscanf ( v[4].c_str(),"%lf",&dnoise );
			int linkQ = ( int ) dlq;
			int signal = convertSSVal ( dsignal );
			int noise = convertSSVal ( dnoise );
			paintBarValue ( WirelessQuality,linkQ,"Quality: ",linkQ,"%" );
			paintBarValue ( WirelessSignal,signal,"Signal: -",dsignal,"dbm" );
			paintBarValue ( WirelessNoise,noise,"Noise: -",dnoise ,"dbm" );
		}
	}
	ifs.close();
	setWirelessState ( found && state );
}

void NetworkControl::setWirelessState ( bool state )
{
	WirelessQuality->setShown ( state );
	WirelessSignal->setShown ( state );
	WirelessNoise->setShown ( state );
	WirelessLabel->setShown ( state );
}

void NetworkControl::paintBarValue ( QProgressBar *bar,int value,string pref,double v,string suff )
{
	//printf("value: %d\n",value);
	value = ( value < 0 ) ?0:value;
	value = ( value > 100 ) ?100:value;
	bar->setProgress ( value );
	char tip[256];
	if ( suff == "dbm" )
	{
		if ( v > 0 ) v = 256 - v;
		v = fabs ( v );
	}
	sprintf ( tip,"%s%.0lf%s",pref.c_str(),v,suff.c_str() );
	QToolTip::add ( bar,tip );
}

void NetworkControl::closeEvent ( QCloseEvent * e )
{
	writeConfigFile();
	e->accept();
}

void NetworkControl::ringBell()
{
	if ( !bellthread->running() )
	{
		bellthread->start();
	}
}

void NetworkControl::readConfigFile()
{
	initHandler->readConfigFile();
	// set some defaults
	int xv = x();
	int yv = y();
	initHandler->readValue ( "PosX",xv );
	initHandler->readValue ( "PosY",yv );
	move ( xv,yv );

	xv = width();
	yv = height();
	initHandler->readValue ( "SizeX",xv );
	initHandler->readValue ( "SizeY",yv );
	resize ( xv,yv );

	int v;
	initHandler->readValue ( "LastInterface",v );
	v = ( v >= interfaceCount ) ?interfaceCount-1:v;
	InterfaceComboBox->setCurrentItem ( v );
}

void NetworkControl::writeConfigFile()
{
	initHandler->writeValue ( "PosX",x() );
	initHandler->writeValue ( "PosY",y() );
	initHandler->writeValue ( "SizeX",width() );
	initHandler->writeValue ( "SizeY",height() );
	initHandler->writeValue ( "LastInterface",InterfaceComboBox->currentItem() );
	initHandler->writeConfigFile();
}

#include "networkcontrol.moc"

