/***************************************************************************
 *   Copyright (C) 2009 by Paul Lutus                                      *
 *   lutusp@arachnoid.com                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
package magiclantern;

import java.awt.*;
import javax.swing.*;
import java.io.*;
import java.util.Scanner.*;
import java.text.*;

/**
 *
 * @author lutusp
 */
final public class CommonCode {

    // a convenience for debugging
    static public void p(String s) {
        System.out.println(s);
    }

    static public void populateComboBox(JComboBox box, String[] data, int initialIndex) {
        try {
            box.removeAllItems();
            if (data != null) {
                for (int i = 0; i < data.length; i++) {
                    String name = CommonCode.strip(data[i]);
                    if (name.length() > 0) {
                        box.addItem(name);
                    }
                }
            }
            if (initialIndex < box.getItemCount()) {
                box.setSelectedIndex(initialIndex);
            }
        } catch (Exception e) {
            System.out.println("CommonCode.populateComboBox: " + e);
        }
    }

    static String readTextFile(String path, String lineSep) {
        String s = null;
        File f = new File(path);
        if (f.exists()) {
            StringBuffer sb = new StringBuffer();
            try {
                BufferedReader br = new BufferedReader(new FileReader(f));
                String line;
                while ((line = br.readLine()) != null) {
                    sb.append(line + lineSep);
                }
                br.close();
                s = sb.toString();
            } catch (Exception e) {
                //System.out.println(e);
            }
        }
        return s;
    }

    static boolean saveTextFile(JFrame parent, String data, String path, boolean confirm) {
        boolean proceed = true;
        boolean saved = false;
        File f = new File(path);
        if (confirm) {
            if (f.exists()) {
                String name = f.getName();
                int reply = JOptionPane.showConfirmDialog(parent, "File \"" + name + "\" exists. Okay to overwrite?", "Overwrite Existing File", JOptionPane.YES_NO_OPTION);
                proceed = (reply == JOptionPane.YES_OPTION);
            }
        }
        if (proceed) {
            try {
                FileWriter fw = new FileWriter(f);
                fw.write(data);
                fw.close();
                saved = true;
            } catch (Exception e) {
                System.out.println(e);
            }
        }
        return saved;
    }

    static boolean testMakeDirs(String path) {
        File fpath = new File(path);
        if (fpath.exists()) {
            return false;
        } else {
            fpath.mkdirs();
            return true;
        }
    }

    static public String join(String[] array, String delim) {
        StringBuffer sb = new StringBuffer();
        for (int i = 0; i < array.length; i++) {
            if (i > 0) {
                sb.append(delim);
            }
            sb.append(array[i]);
        }
        return sb.toString();
    }

    static GridBagConstraints addComponent2(Component source, JComponent target, GridBagConstraints gridBagConstraints) {
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        target.add(source, gridBagConstraints);
        return gridBagConstraints;
    }

    static public GridBagConstraints addComponent(Component source, JComponent target) {
        GridBagConstraints gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.weighty = 1.0;
        return addComponent2(source, target, gridBagConstraints);
    }

    static public GridBagConstraints addComponent(Component source, JComponent target, int x, int y) {
        GridBagConstraints gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = x;
        gridBagConstraints.gridy = y;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.weighty = 1.0;
        return addComponent2(source, target, gridBagConstraints);
    }

    static public GridBagConstraints addComponent(Component source, JComponent target, int x, int y, int inset) {
        GridBagConstraints gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = x;
        gridBagConstraints.gridy = y;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.weighty = 1.0;
        gridBagConstraints.insets = new Insets(inset, inset, inset, inset);
        return addComponent2(source, target, gridBagConstraints);
    }

    static public GridBagConstraints addComponent(Component source, JComponent target, int x, int y, int inset, double wx, double wy) {
        GridBagConstraints gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = x;
        gridBagConstraints.gridy = y;
        gridBagConstraints.insets = new Insets(inset, inset, inset, inset);
        gridBagConstraints.weightx = wx;
        gridBagConstraints.weighty = wy;
        return addComponent2(source, target, gridBagConstraints);
    }

    static void beep() {
        // a way to trace random program beeps
        //System.out.println(getStackTrace());
        Toolkit.getDefaultToolkit().beep();
    }

    static String getStackTrace() {
        final Throwable throwable = new IllegalArgumentException("CommonCode.getStackTrace");
        final Writer result = new StringWriter();
        final PrintWriter printWriter = new PrintWriter(result);
        throwable.printStackTrace(printWriter);
        return result.toString();
    }

    static public int max(int a, int b) {
        return ((a > b) ? a : b);
    }

    static public int min(int a, int b) {
        return ((a < b) ? a : b);
    }

    static public String strip(String s) {
        return s.replaceFirst("^\\s*(.*)\\s*$", "$1");
    }

    // this is meant to avoid a really horrible filechooser dialog
    // design that only exists in the GTK L&F
    static public boolean changeLookAndFeel(Component comp) {
        boolean changed = false;
        String laf = UIManager.getLookAndFeel().getDescription();
        if (laf != null && laf.matches("GTK.*")) {
            String lf = UIManager.getCrossPlatformLookAndFeelClassName();
            try {
                UIManager.setLookAndFeel(lf);
                // just this one component
                SwingUtilities.updateComponentTreeUI(comp);
                changed = true;
            } catch (Exception e) {
                System.out.println(e);
            }
        }
        return changed;
    }

    // recover from the above indulgence
    static public void restoreSystemLookAndFeel() {
        try {
            // Default to system-specific L&F
            UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
        } catch (Exception e) {
            // handle exception
        }
    }

    static public void setTextScrollToTop(JTextArea area, String text) {
        area.setText(text);
        area.setSelectionStart(0);
        area.setSelectionEnd(0);
    }

    static public String createDecimalFormat(long n) {
        DecimalFormat df = new DecimalFormat("###,###");
        return df.format(n);
    }

    static public String createDecimalFormat(double n) {
        DecimalFormat df = new DecimalFormat("###,###.00");
        return df.format(n);
    }
}
