// ***************************************************************************
// *   Copyright (C) 2017 by Paul Lutus                                      *
// *   lutusp@arachnoid.com                                                  *
// *                                                                         *
// *   This program is free software; you can redistribute it and/or modify  *
// *   it under the terms of the GNU General Public License as published by  *
// *   the Free Software Foundation; either version 2 of the License, or     *
// *   (at your option) any later version.                                   *
// *                                                                         *
// *   This program is distributed in the hope that it will be useful,       *
// *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
// *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
// *   GNU General Public License for more details.                          *
// *                                                                         *
// *   You should have received a copy of the GNU General Public License     *
// *   along with this program; if not, write to the                         *
// *   Free Software Foundation, Inc.,                                       *
// *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
// ***************************************************************************
package jsqliteclient;

import java.io.File;
import java.net.URL;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Scanner;

/**
 *
 * @author lutusp
 */
final public class TerminalFunctions implements Executable {

    JSQLiteClient parent;
    Connection conn = null;
    ArrayList<String> sqlColNames;
    int[] columnWidths;
    ArrayList<ArrayList<Object>> data;
    String prompt = "JDBClient> ";
    String currentCom;

    public TerminalFunctions(JSQLiteClient p) {
        parent = p;
        setState();
        parent.terminalTextDisplay.setEditable(false);
        parent.sv_termEngine = new HistoryEngine(this, parent.terminalCommandTextField, true);
        connect(false);
    }

    protected void connect(boolean connect) {
        long startTime = System.currentTimeMillis();
        String error = null;
        try {
            if (connect) {
                String dbUrl = parent.makeDBUrl();
                QueryProcessor qp = new QueryProcessor(parent,null,true);
                conn = qp.configureConection(dbUrl);
                assert(conn != null);
                //new SQLRegex(parent).defineRegex(conn);
                clear();
                addText(prompt);
            } else {
                if (conn != null) {
                    conn.close();
                }
                conn = null;
                clear();
                addText("(Not connected)\n");
            }
        } catch (Exception e) {
            if (parent.debug) {
                e.printStackTrace(System.out);
            }
            logErrors(e);
            error = e.getMessage();
        }
        setState();
        long endTime = System.currentTimeMillis();
        parent.logEventAction("T: connect", startTime, endTime, error);
    }

    protected void wrapControl() {
        parent.terminalTextDisplay.setLineWrap(parent.sv_termWrapCheckBox.isSelected());
    }

    protected void logErrors(Exception err) {
        addText(String.format("Error: %s\n", err.getMessage()));
    }

    protected void addText(String s) {
        parent.terminalTextDisplay.append(s);
        parent.terminalTextDisplay.setCaretPosition(parent.terminalTextDisplay.getDocument().getLength());
    }

    protected void setFont() {
        parent.terminalTextDisplay.setFont(parent.baseFont);
        parent.terminalCommandTextField.setFont(parent.baseFont);
    }

    protected void getResults(ResultSet rs) {

        data = new ArrayList<>();
        int internalPad = 2;
        sqlColNames = new ArrayList<>();
        try {
            ResultSetMetaData md = rs.getMetaData();
            int cols = md.getColumnCount();
            columnWidths = new int[cols];
            // get column names
            for (int i = 0; i < cols; i++) {
                String cl = md.getColumnLabel(i + 1);
                sqlColNames.add(cl);
                columnWidths[i] = cl.length() + internalPad;
            }
            ArrayList<Object> row;
            while (rs.next()) {
                row = new ArrayList<>();
                int i = 0;
                for (Object obj : sqlColNames) {
                    String key = obj.toString().trim();
                    Object out = null;
                    try {
                        out = rs.getObject(key);
                    } catch (Exception e) {
                        if (parent.debug) {
                            e.printStackTrace(System.out);
                        }
                        logErrors(e);
                    }
                    out = (out == null) ? "NULL" : out;
                    row.add(out);
                    columnWidths[i] = Math.max(columnWidths[i], out.toString().length() + internalPad);
                    i += 1;
                }
                data.add(row);
            }
        } catch (Exception e) {
            if (parent.debug) {
                e.printStackTrace(System.out);
            }
            logErrors(e);
        }

    }

    // t: 0 = left, 1 = center, 2 = right
    protected String padString(String s, String pad, int len, int t) {
        s = String.format("%s%s%s", pad, s, pad);
        StringBuilder sb = new StringBuilder();
        if (t == 0) {
            sb.append(s);
        }
        int alen = len - s.length();
        int blen = 0;
        if (t == 1) {
            alen = (len - s.length()) / 2;
            blen = len - (alen + s.length());
        }
        for (int i = 0; i < alen; i++) {
            sb.append(pad);
        }
        if (t == 1) {
            sb.append(s);
            while (blen > 0) {
                sb.append(pad);
                blen -= 1;
            }
        }
        if (t == 2) {
            sb.append(s);
        }
        return sb.toString();
    }

    protected String tableLine(int n) {
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < n; i++) {
            sb.append("-");
        }
        sb.append("+");
        return sb.toString();
    }

    protected void showResultSet(ResultSet rs, long st, long et, boolean display) {
        if (rs != null) {
            getResults(rs);
            StringBuilder sb = new StringBuilder();
            int rows = data.size();
            if (rows >= 0) {
                String pad = " ";
                StringBuilder sh = new StringBuilder("|");
                StringBuilder sl = new StringBuilder("+");
                int i = 0;
                for (String s : sqlColNames) {
                    sl.append(tableLine(columnWidths[i]));
                    sh.append(padString(s, pad, columnWidths[i], 1));
                    sh.append("|");
                    i += 1;
                }
                sl.append("\n");
                sb = new StringBuilder(sl);
                sb.append(sh);
                sb.append("\n");
                sb.append(sl);
                for (ArrayList<Object> arl : data) {
                    sb.append("|");
                    i = 0;
                    for (Object obj : arl) {

                        int mode = (obj instanceof java.lang.String) ? 0 : 2;
                        sb.append(padString(obj.toString(), pad, columnWidths[i], mode));
                        sb.append("|");
                        i += 1;
                    }
                    sb.append("\n");
                }
                sb.append(sl);
            }
            long jt = System.currentTimeMillis();
            long md = et - st;
            long jd = jt - et;
            long tt = jt - st;
            sb.append(String.format("%s rows in set (%.2f total, SQLite %.2f, Java %.2f sec)\n", rows,
                    getDoubleTime(tt), getDoubleTime(md), getDoubleTime(jd)));
            addText(sb.toString());
        }
    }

    protected double getDoubleTime(long v) {
        return (double) v / 1000.0;
    }

    protected void clear() {
        parent.terminalTextDisplay.setText("");
        setState();
    }

    protected void copyToClipboard() {
        parent.writeToClipboard(parent.terminalTextDisplay.getText());
    }

    public String readSQLDumpFile(String path) {
        long startTime = System.currentTimeMillis();
        String error = null;
        String result = null;
        try {
            if (path.matches(".*(http|ftp):.*")) {
                URL url = new URL(path);
                result = new Scanner(url.openStream()).useDelimiter("\\Z").next();
            } else {
                result = new Scanner(new File(path)).useDelimiter("\\Z").next();
            }
        } catch (Exception e) {
            if (parent.debug) {
                e.printStackTrace(System.out);
            }
            error = e.getMessage();
            logErrors(e);
        }
        long endTime = System.currentTimeMillis();
        parent.logEventAction(String.format("T: read file/URL %s", path), startTime, endTime, error);
        return result;
    }

    protected void processFile(String path) {
        String fdata = readSQLDumpFile(path);
        if (fdata != null) {
            String[] array = fdata.split("\n");
            StringBuilder com = new StringBuilder();
            for (String item : array) {
                addText(".");
                item = item.trim();
                if (item.length() > 0) {
                    if (!item.matches("--.*")) {
                        com.append(item).append(" ");
                        if (item.matches(".*;$")) {
                            executeOneCom(com.toString(), false);
                            com = new StringBuilder();
                        }
                    }
                }
            }
            addText("\n");
        }
    }

    @Override
    public void execute(String text) {
        try {
            String[] array = text.split(";");
            for (String com : array) {
                com = com.trim();
                if (com.matches("(?i).*source\\s.*")) {
                    addText(com + "\n");
                    String path = com.replaceFirst("(?i).*source\\s+(.*)$", "$1");
                    processFile(path);
                    addText(prompt);
                } else {
                    executeOneCom(com, true);
                }
            }
        } catch (Exception e) {
            if (parent.debug) {
                e.printStackTrace(System.out);
            }
            logErrors(e);
        }
    }

    protected void executeOneCom(String com, boolean display) {
        long startTime = System.currentTimeMillis();
        if (com != null && com.length() > 0) {
            String error = null;

            try {
                currentCom = com;
                if (display) {
                    addText(" " + com + "\n");
                }
                long st = System.currentTimeMillis();
                Statement stmt = conn.createStatement();
                String escCom = parent.escapeQueryArg(com);
                stmt.execute(escCom);
                long et = System.currentTimeMillis();
                ResultSet rs = stmt.getResultSet();
                showResultSet(rs, st, et, display);
                if (display) {
                    addText(prompt);
                }
                stmt.close();
            } catch (Exception e) {
                if (parent.debug) {
                    e.printStackTrace(System.out);
                }
                error = e.getMessage();
                logErrors(e);
                if (display) {
                    addText(prompt);
                }
            }
            long endTime = System.currentTimeMillis();
            parent.logEventAction(String.format("T: %s", currentCom), startTime, endTime, error);
        }
    }

    protected void connectDisconnect() {
        connect(conn == null);
        setFont();
    }

    protected void setState() {
        parent.connectButton.setText((conn == null) ? "Connect" : "Disconnect");
        parent.terminalCommandTextField.setEnabled(conn != null);
        parent.terminalCommandTextField.grabFocus();
        wrapControl();
    }

    public void p(String s) {
        System.out.println(s);
    }

}
