// ***************************************************************************
// *   Copyright (C) 2012 by Paul Lutus                                      *
// *   lutusp@arachnoid.com                                                  *
// *                                                                         *
// *   This program is free software; you can redistribute it and/or modify  *
// *   it under the terms of the GNU General Public License as published by  *
// *   the Free Software Foundation; either version 2 of the License, or     *
// *   (at your option) any later version.                                   *
// *                                                                         *
// *   This program is distributed in the hope that it will be useful,       *
// *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
// *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
// *   GNU General Public License for more details.                          *
// *                                                                         *
// *   You should have received a copy of the GNU General Public License     *
// *   along with this program; if not, write to the                         *
// *   Free Software Foundation, Inc.,                                       *
// *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
// ***************************************************************************

package jdbclient;

import java.awt.event.KeyEvent;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import javax.swing.JTextField;

/**
 *
 * @author lutusp
 */
final public class HistoryEngine implements Configurable {

    Executable exec;
    JTextField entryField;
    ArrayList<String> history;
    int historyIndex = 0;
    boolean interactive;
    int maxHistorySize = 32;

    public HistoryEngine(Executable e, JTextField ef, boolean interact) {
        exec = e;
        entryField = ef;
        interactive = interact;
        history = new ArrayList<>();
        ef.addKeyListener(new java.awt.event.KeyAdapter() {
            @Override
            public void keyReleased(java.awt.event.KeyEvent evt) {
                testKey(evt);
            }
        });

    }

    private void testKey(java.awt.event.KeyEvent evt) {
        switch (evt.getKeyCode()) {
            case KeyEvent.VK_ENTER:
                processEntry();
                break;
            case KeyEvent.VK_DOWN:
                fetchHistory(1);
                break;
            case KeyEvent.VK_UP:
                fetchHistory(-1);
                break;
        }
    }

    private void processEntry() {
        String text = entryField.getText().trim();
        insertIntoHistory(text);
        if (interactive) {
            insertIntoHistory("");
            entryField.setText("");
        }
        exec.execute(text);
    }

    protected JTextField getControl() {
        return entryField;
    }

    protected void fetchHistory(int n) {
        int top = history.size();
        if (top == 0) {
            Beep.beep();
            return;
        }
        if (historyIndex == top - 1) {
            String pending = entryField.getText().trim();
            if (pending.length() > 0) {
                insertIntoHistory(pending);
            }
        }
        top = history.size();
        historyIndex += n;
        int oth = historyIndex;
        historyIndex = Math.max(0, historyIndex);
        historyIndex = Math.min(top - 1, historyIndex);
        if (oth != historyIndex) {
            Beep.beep();
        }
        String s = history.get(historyIndex);
        entryField.setText(s);
    }

    protected void insertIntoHistory(String com) {
        if (com != null) {
            while (history.contains(com)) {
                history.remove(com);
            }
            history.add(com);
            int sz = history.size();
            if(sz >= maxHistorySize) {
                history = new ArrayList<>(history.subList(sz-maxHistorySize,sz));
            }
            historyIndex = history.size() - 1;
        }
    }

    protected void clearHistory() {
        history = new ArrayList<>();
    }

    // override of Configurable class interface
    @Override
    public void fromString(String shist) {
        String[] array = shist.split("\t");
        history = new ArrayList<>(Arrays.asList(array));
        history.add("");
        historyIndex = history.size() - 1;
    }

    @Override
    public String toString() {
        String result = "";
        Iterator<String> iter = history.iterator();
        if (iter.hasNext()) {
            StringBuilder sb = new StringBuilder(iter.next());
            while (iter.hasNext()) {
                sb.append("\t");
                sb.append(iter.next());
            }
            result = sb.toString();
        }
        return result;
    }
}
