#!/usr/bin/env python

import cadquery as cq
import os

# all dimensions in mm

pitch = 1.5 # interval between thread peaks

# gap to allow threading to bolt
# this value is shared between bolt and nut
clearance = 0.25

thread_a = 3.845 - clearance # "valley" of thread
thread_b = 5.160 - clearance # "peak" of thread

bolt_head_dia = 18 # mm
bolt_head_height = 6.5
# subtract a tiny value
# to avoid a collision between parts
shaft_radius = thread_a-0.001
shaft_height = 31

bolt_shaft = (
  cq.Workplane("XY")
  .circle(shaft_radius)
  .extrude(shaft_height)
)

bolt_head = (
  cq.Workplane("XY")
  .polygon(6,bolt_head_dia, circumscribed=False)
  .extrude(bolt_head_height)  
)


thread_vals = [
  (thread_a,0),
  (thread_b,pitch/2),
  (thread_a,pitch),
  (thread_a,0)
]

r = thread_a+1 # helix radius (base dimension)
h = shaft_height-bolt_head_height+pitch 

# create helix
wire = cq.Wire.makeHelix(pitch=pitch, height=h, radius=r)
helix = cq.Workplane(obj=wire)

thread = (
    cq.Workplane('XZ')
    .polyline(thread_vals)
    .close()
    .sweep(helix, isFrenet=True)
    .translate((0,0,bolt_head_height-pitch))
)

#show(thread + bolt_shaft + bolt_head)

bolt = bolt_head + bolt_shaft + thread

v1 = bolt_head_dia/2

r = bolt_head_dia / 8 # radius of bolt head curve
top_slope = 3

enclosure = ( cq.Workplane("XZ")
          .lineTo(-v1+r,0)
          .radiusArc((-v1,r),r)
          .lineTo(-v1,shaft_height-top_slope)
          .lineTo(0,shaft_height)
          .close()
          .revolve(360,(0,0),(0,1))
        )

assembled = enclosure.intersect(bolt, clean=False)

if 'JPY_PARENT_PID' in os.environ:
    assembled
else:
    cq.exporters.export(assembled,'cadquery_bolt.stl')
    ibh = enclosure.intersect(bolt_head, clean=False)
    ibs = enclosure.intersect(bolt_shaft, clean=False)
    ibt = enclosure.intersect(thread, clean=False)
    assy = cq.Assembly()
    assy.add(ibh, color=cq.Color(1,0,0), name = "Hex head")
    assy.add(ibs, color=cq.Color(0,1,0), name = "Shaft")
    assy.add(ibt, color=cq.Color(0,.5,.5), name = "Thread")
    assy.save("cadquery_bolt.gltf")
